/*
    read.d -- Read.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/

object standard_readtable;

object Vreadtable;
object Vread_default_float_format;
object Vread_base;
object Vread_suppress;

object Kradix;
object Kjunk_allowed;

object siSsharp_comma;

#ifndef MTCL
object READtable;
int READdefault_float_format;
int READbase;
bool READsuppress;
bool preserving_whitespace_flag;
bool escape_flag;
object delimiting_char;
bool detect_eos_flag;
bool in_list_flag;
bool dot_flag;
object default_dispatch_macro;
object big_register_0;
int sharp_eq_context_max;
#endif MTCL

#ifdef CLOS
object Sstream_read_line,
  Sstream_read_char,
  Sstream_unread_char,
  Sstream_peek_char,
  Sstream_listen,
  Sstream_clear_input;
#endif

#ifndef MTCL
extern object readc();
object (*read_ch_fun)() = readc;
#endif

/******************************* ------- ******************************/

static spice_count = 0;

#define	token_buffer	token->st.st_self

object dispatch_reader;


#define	cat(c)	(READtable->rt.rt_self[char_code((c))].rte_chattrib)

#define	SHARP_EQ_CONTEXT_SIZE	64

void setup_READtable()
{
	READtable = current_readtable();
}

struct sharp_eq_context_struct {
	object	sharp_index;
	object	sharp_eq;
	object	sharp_sharp;
} sharp_eq_context[SHARP_EQ_CONTEXT_SIZE];


void setup_READ()
{
	object x;

	READtable = current_readtable();
	x = symbol_value(Vread_default_float_format);
	if (x == Ssingle_float || x == Sshort_float)
		READdefault_float_format = 'S';
	else if (x == Sdouble_float || x == Slong_float)
		READdefault_float_format = 'F';
	else {
		Vread_default_float_format->s.s_dbind = Ssingle_float;
	FEerror("The value of *READ-DEFAULT-FLOAT-FORMAT*, ~S, was illegal.",
			1, x);
	}
	x = symbol_value(Vread_base);
	if (!FIXNUMP(x) || fix(x) < 2 || fix(x) > 36) {
		Vread_base->s.s_dbind = MAKE_FIXNUM(10);
		FEerror("The value of *READ-BASE*, ~S, was illegal.", 1, x);
	}
	READbase = fix(x);
	READsuppress = symbol_value(Vread_suppress) != Cnil;
}

void setup_standard_READ()
{
	READtable = standard_readtable;
	READdefault_float_format = 'S';
	READbase = 10;
	READsuppress = FALSE;
	sharp_eq_context_max = 0;
	backq_level = 0;
}

#ifdef CLOS
object
interactive_readc(object stream)
{
	funcall(2, Sstream_read_char, stream);
	return(VALUES(0));
}
#endif CLOS

object
readc(object in)
{
	return(code_char(readc_stream(in)));
}

#define	read_char(in)	(*read_ch_fun)(in)

void unread_char(object c, object in)
{
	if (type_of(c) != t_character)
		FEwrong_type_argument(Scharacter, c);
#ifdef CLOS
	if (type_of(in) == t_instance)
	  funcall(3, Sstream_unread_char, in, c);
	else
#endif
	  unreadc_stream(char_code(c), in);
}

/*
	Peek_char corresponds to COMMON Lisp function PEEK-CHAR.
	When pt is TRUE, preceeding whitespaces are ignored.
*/
object
peek_char(bool pt, object in)
{
	object c;

	c = read_char(in);
	if (pt)
	  while (cat(c) == cat_whitespace)
	    c = read_char(in);
	unread_char(c, in);
	return(c);
}
		

object
read_object_recursive(object in)
{
	volatile object x;
	bool e;

	object old_READtable = READtable;
	int old_READdefault_float_format = READdefault_float_format;
	int old_READbase = READbase;
	bool old_READsuppress = READsuppress;
	int nr;

	if ((nr = frs_push(FRS_PROTECT, Cnil)) != 0)
		e = TRUE;
	else {
	  setup_READ();
	  x = read_object(in);
	  e = FALSE;
	}

	frs_pop();

	READtable = old_READtable;
	READdefault_float_format = old_READdefault_float_format;
	READbase = old_READbase;
	READsuppress = old_READsuppress;

	if (e) unwind(nlj_fr, nlj_tag, nr);
	return(x);
}


object
read_object_non_recursive(object in)
{
	volatile object x;
	int i, nr;
	bool e;
	object old_READtable;
	int old_READdefault_float_format;
	int old_READbase;
	int old_READsuppress;
	int old_sharp_eq_context_max;
	struct sharp_eq_context_struct
		old_sharp_eq_context[SHARP_EQ_CONTEXT_SIZE];
	int old_backq_level;

	old_READtable = READtable;
	old_READdefault_float_format = READdefault_float_format;
	old_READbase = READbase;
	old_READsuppress = READsuppress;
	old_sharp_eq_context_max = sharp_eq_context_max;
	for (i = 0;  i < sharp_eq_context_max;  i++)
		old_sharp_eq_context[i] = sharp_eq_context[i];
	old_backq_level = backq_level;
	setup_READ();
	sharp_eq_context_max = 0;
	backq_level = 0;

	if ((nr = frs_push(FRS_PROTECT, Cnil)) != 0)
		e = TRUE;
	else {
		e = FALSE;
		x = read_object(in);
	
		if (sharp_eq_context_max > 0)
			x = patch_sharp(x);
	}

	frs_pop();

	READtable = old_READtable;
	READdefault_float_format = old_READdefault_float_format;
	READbase = old_READbase;
	READsuppress = old_READsuppress;
	sharp_eq_context_max = old_sharp_eq_context_max;
	for (i = 0;  i < sharp_eq_context_max;  i++)
		sharp_eq_context[i] = old_sharp_eq_context[i];
	backq_level = old_backq_level;
	if (e) unwind(nlj_fr, nlj_tag, nr);
	return(x);
}

/* Beppe: faster code for inner loop from file stream */
#if defined(unix) && !defined(CLOS)
#define READ_CHAR_TO(res, in, eof_code) \
  {FILE *fp; \
     if (fp=in->sm.sm_fp) \
	{int ch = getc(fp); \
	   if (ch == EOF && feof(fp)) \
	      {eof_code;} \
           else res = code_char(ch);} \
      else \
	if (stream_at_end(in)) \
	   {eof_code;} \
	else res = read_char(in);}
#else
#define READ_CHAR_TO(res, in, eof_code) \
  {if (stream_at_end(in)) {eof_code;} else res = read_char(in);}
#endif unix

/*
	Read_object(in) reads an object from stream in.
	This routine corresponds to COMMON Lisp function READ.
*/
object
read_object(object in)
{
	object x;
	object c;
	enum chattrib a;
	object old_delimiter;
	object p;
	int length, colon, colon_type;
	int i;
	bool df, ilf;

	cs_check(in);

	old_delimiter = delimiting_char;
	delimiting_char = OBJNULL;
	df = detect_eos_flag;
	detect_eos_flag = FALSE;
	ilf = in_list_flag;
	in_list_flag = FALSE;
	dot_flag = FALSE;

BEGIN:
	/* Beppe: */
	do {
		READ_CHAR_TO(c, in, {if (df) return(OBJNULL);
					else
					    end_of_stream(in);});
		a = cat(c);
	} while (a == cat_whitespace);
	delimiting_char = OBJNULL;
	if (old_delimiter != OBJNULL && old_delimiter == c)
		return(OBJNULL);
	if (a == cat_terminating || a == cat_non_terminating) {
		x = READtable->rt.rt_self[char_code(c)].rte_macro;
		i = funcall(3, x, in, c);

		if (i == 0) goto BEGIN;
		if (i > 1) FEerror("The readmacro ~S returned ~D values.",
				 2, x, MAKE_FIXNUM(i));
		return(VALUES(0));
	}
	escape_flag = FALSE;
	length = 0;
	colon_type = 0;
	for (;;) {
		if (a == cat_single_escape) {
			c = read_char(in);
			a = cat_constituent;
			escape_flag = TRUE;
		} else if (a == cat_multiple_escape) {
			escape_flag = TRUE;
			for (;;) {
				if (stream_at_end(in))
					end_of_stream(in);
				c = read_char(in);
				a = cat(c);
				if (a == cat_single_escape) {
					c = read_char(in);
					a = cat_constituent;
				} else if (a == cat_multiple_escape)
					break;
				if (length >= token->st.st_dim)
					too_long_token();
				token_buffer[length++] = char_code(c);
			}
			goto NEXT;
		} else if ('a' <= char_code(c) && char_code(c) <= 'z')
			c = code_char(char_code(c) - ('a' - 'A'));
		else if (char_code(c) == ':') {
			if (colon_type == 0) {
				colon_type = 1;
				colon = length;
			} else if (colon_type == 1 && colon == length-1)
				colon_type = 2;
			else
				colon_type = -1;
				/*  Colon has appeared twice.  */
		}
		if (a == cat_whitespace || a == cat_terminating) {
			if (preserving_whitespace_flag ||
			    cat(c) != cat_whitespace)
				unread_char(c, in);
			break;
		}
		if (length >= token->st.st_dim)
			too_long_token();
		token_buffer[length++] = char_code(c);
	NEXT:
		READ_CHAR_TO(c, in, break);
		a = cat(c);
	}

	if (READsuppress) {
		token->st.st_fillp = length;
		return(Cnil);
	}
	if (ilf && !escape_flag &&
	    length == 1 && token->st.st_self[0] == '.') {
		dot_flag = TRUE;
		return(Cnil);
	} else if (!escape_flag && length > 0) {
		for (i = 0;  i < length;  i++)
			if (token->st.st_self[i] != '.')
				goto N;
		FEerror("Dots appeared illegally.", 0);
	}

N:
	token->st.st_fillp = length;
	if (escape_flag || (READbase <= 10 && token_buffer[0] > '9'))
		goto SYMBOL;
	x = parse_number(token_buffer, length, &i, READbase);
	if (x != OBJNULL && length == i)
		return(x);

SYMBOL:
	if (colon_type == 1 /* && length > colon + 1 */) {
		if (colon == 0)
			p = keyword_package;
		else {
			token->st.st_fillp = colon;
			p = find_package(token);
			if (Null(p))
			    FEerror("There is no package with the name ~A.",
				    1, copy_simple_string(token));
		}
		for (i = colon + 1;  i < length;  i++)
			token_buffer[i - (colon + 1)]
			= token_buffer[i];
		token->st.st_fillp = length - (colon + 1);
		if (colon > 0) {
			token->st.st_self[token->st.st_fillp] = '\0';
			x = find_symbol(token->st.st_self,
					token->st.st_fillp, p);
			if (intern_flag != EXTERNAL)
			FEerror("Cannot find the external symbol ~A in ~S.",
						2, copy_simple_string(token), p);
			return(x);
		}
	} else if (colon_type == 2 /* && colon > 0 && length > colon + 2 */) {
		token->st.st_fillp = colon;
		p = find_package(token);
		if (Null(p))
			FEerror("There is no package with the name ~A.",
				1, copy_simple_string(token));
		for (i = colon + 2;  i < length;  i++)
			token_buffer[i - (colon + 2)] = token_buffer[i];
		token->st.st_fillp = length - (colon + 2);
	} else
		p = current_package();
	token_buffer[token->st.st_fillp] = '\0';
	x = intern(token_buffer, p);
	if (x->s.s_self == token_buffer) {
		x->s.s_self = alloc_contblock(token->st.st_fillp+1);
		memcpy(x->s.s_self, token_buffer, token->st.st_fillp+1);
	}
	return(x);
}

Lleft_parenthesis_reader(int narg, object in, object c)
{
	object x, y;
	object *p;

	check_arg(2);
	y = Cnil;
	for (p = &y ; ; p = &(CDR(*p))) {
		delimiting_char = code_char(')');
		in_list_flag = TRUE;
		x = read_object(in);
		if (x == OBJNULL)
			break;
		if (dot_flag) {
			if (p == &y)
		FEerror("A dot appeared after a left parenthesis.", 0);
			in_list_flag = TRUE;
			*p = read_object(in);
			if (dot_flag)
		FEerror("Two dots appeared consecutively.", 0);
			c = read_char(in);
			while (cat(c) == cat_whitespace)
				c = read_char(in);
			if (char_code(c) != ')')
		FEerror("A dot appeared before a right parenthesis.", 0);
			break;
		}
		*p = CONS(x, Cnil);
	}

	VALUES(0) = y;
	RETURN(1);
}

#define	is_exponent_marker(i)	\
	((i) == 'e' || (i) == 'E' ||	\
	 (i) == 's' || (i) == 'S' || (i) == 'f' || (i) == 'F' || \
	 (i) == 'd' || (i) == 'D' || (i) == 'l' || (i) == 'L' || \
	 (i) == 'b' || (i) == 'B')

/*
	Parse_number(s, end, ep, radix) parses C string s
	up to (but not including) s[end]
	using radix as the radix for the rational number.
	(For floating numbers, radix should be 10.)
	When parsing has been succeeded,
	the index of the next character is assigned to *ep,
	and the number is returned as a lisp data object.
	If not, OBJNULL is returned.
*/
object
parse_number(char *s, int end, int *ep, int radix)
{
	object x;
	fixnum sign;
	struct bignum *integer_part;
	double fraction, fraction_unit, f;
	char exponent_marker;
	int exponent;
	int i, j, k;
	int d;

	if (s[end-1] == '.')
		radix = 10;
		/*
			DIRTY CODE!!
		*/
BEGIN:
	exponent_marker = 'E';
	i = 0;
	sign = 1;
	if (s[i] == '+')
		i++;
	else if (s[i] == '-') {
		sign = -1;
		i++;
	}
	integer_part = (struct bignum *)big_register_0;
	integer_part->big_car = 0;
	integer_part->big_cdr = NULL;
	if (i >= end)
		goto NO_NUMBER;
	if (s[i] == '.') {
		if (radix != 10) {
			radix = 10;
			goto BEGIN;
		}
		i++;
		goto FRACTION;
	}
	if ((d = digitp(s[i], radix)) < 0)
		goto NO_NUMBER;
	do {
		mul_int_big(radix, integer_part);
		add_int_big(d, integer_part);
		i++;
	} while (i < end && (d = digitp(s[i], radix)) >= 0);
	if (i >= end)
		goto MAKE_INTEGER;
	if (s[i] == '.') {
		if (radix != 10) {
			radix = 10;
			goto BEGIN;
		}
		if (++i >= end)
			goto MAKE_INTEGER;
		else if (digitp(s[i], radix) >= 0)
			goto FRACTION;
		else if (is_exponent_marker(s[i])) {
			fraction
			= (double)sign * big_to_double(integer_part);
			goto EXPONENT;
		} else
			goto MAKE_INTEGER;
	}
	if (s[i] == '/') {
		i++;
		goto DENOMINATOR;
	}
	if (is_exponent_marker(s[i])) {
		fraction = (double)sign * big_to_double(integer_part);
		goto EXPONENT;
	}
/*
	goto NO_NUMBER;
*/

MAKE_INTEGER:
	if (sign < 0)
		complement_big(integer_part);
	x = normalize_big_to_object(integer_part);
/**/
	if (x == big_register_0)
		big_register_0 = alloc_object(t_bignum);
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
/**/
	goto END;

FRACTION:
/*
	if (radix != 10)
		goto NO_NUMBER;
*/
	radix = 10;
	if ((d = digitp(s[i], radix)) < 0)
		goto NO_NUMBER;
	fraction = 0.0;
	fraction_unit = 0.000000001;
	for (;;) {
		k = j = 0;
		do {
			j = 10*j + d;
			i++;
			k++;
			if (i < end)
				d = digitp(s[i], radix);
			else
				break;
		} while (k < 9 && d >= 0);
		while (k++ < 9)
			j *= 10;
		fraction += fraction_unit * (double)j;
		if (i >= end || d < 0)
			break;
		fraction_unit *= 0.000000001;
	}
	fraction += big_to_double(integer_part);
	fraction *= (double)sign;
	if (i >= end)
		goto MAKE_FLOAT;
	if (is_exponent_marker(s[i]))
		goto EXPONENT;
	goto MAKE_FLOAT;

EXPONENT:
/*
	if (radix != 10)
		goto NO_NUMBER;
*/
	radix = 10;
	exponent_marker = s[i];
	i++;
	if (i >= end)
		goto NO_NUMBER;
	sign = 1;
	if (s[i] == '+')
		i++;
	else if (s[i] == '-') {
		sign = -1;
		i++;
	}
	if (i >= end)
		goto NO_NUMBER;
	if ((d = digitp(s[i], radix)) < 0)
		goto NO_NUMBER;
	exponent = 0;
	do {
		exponent = 10 * exponent + d;
		i++;
	} while (i < end && (d = digitp(s[i], radix)) >= 0);
	d = exponent;
	f = 10.0;
	fraction_unit = 1.0;
	while (d > 0)
		if (d%2 == 0) {
			d /= 2;
			f *= f;
		} else {
			--d;
			fraction_unit *= f;
		}
	if (sign > 0)
		fraction *= fraction_unit;
	else
		fraction /= fraction_unit;

MAKE_FLOAT:
#ifdef IEEEFLOAT
	if ((*((int *)&fraction + HIND) & 0x7ff00000) == 0x7ff00000)
		FEerror("Floating-point overflow.", 0);
#endif
	switch (exponent_marker) {

	case 'e':  case 'E':
		exponent_marker = READdefault_float_format;
		goto MAKE_FLOAT;

	case 's':  case 'S':
		x = make_shortfloat((shortfloat)fraction);
		break;

	case 'f':  case 'F':  case 'd':  case 'D':  case 'l':  case 'L':
		x = make_longfloat((longfloat)fraction);
		break;

	case 'b':  case 'B':
		goto NO_NUMBER;
	}
/**/
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
/**/
	goto END;

DENOMINATOR:
	if (sign < 0)
		complement_big(integer_part);
	x = normalize_big_to_object(integer_part);
/**/
	if (x == big_register_0)
		big_register_0 = alloc_object(t_bignum);
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
/**/
	if ((d = digitp(s[i], radix)) < 0)
		goto NO_NUMBER;
	integer_part = (struct bignum *)alloc_object(t_bignum);
	integer_part->big_car = 0;
	integer_part->big_cdr = NULL;
	do {
		mul_int_big(radix, integer_part);
		add_int_big(d, integer_part);
		i++;
	} while (i < end && (d = digitp(s[i], radix)) >= 0);
	x = make_ratio(x, normalize_big_to_object(integer_part));
	goto END;

END:
	*ep = i;
	return(x);

NO_NUMBER:
	*ep = i;
/**/
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
/**/
	return(OBJNULL);
}

object
parse_integer(char *s, int end, int *ep, int radix)
{
	object x;
	fixnum sign;
	struct bignum *integer_part;
	int i, d;

	i = 0;
	sign = 1;
	if (s[i] == '+')
		i++;
	else if (s[i] == '-') {
		sign = -1;
		i++;
	}
	integer_part = (struct bignum *)big_register_0;
	if (i >= end)
		goto NO_NUMBER;
	if ((d = digitp(s[i], radix)) < 0)
		goto NO_NUMBER;
	do {
		mul_int_big(radix, integer_part);
		add_int_big(d, integer_part);
		i++;
	} while (i < end && (d = digitp(s[i], radix)) >= 0);
	if (sign < 0)
		complement_big(integer_part);
	x = normalize_big_to_object(integer_part);
/**/
	if (x == big_register_0)
		big_register_0 = alloc_object(t_bignum);
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
/**/
	*ep = i;
	return(x);

NO_NUMBER:
	*ep = i;
/**/
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
/**/
	return(OBJNULL);
}
/*
	Read_string(delim, in) reads
	a simple string	terminated by character code delim
	and places it in token.
	Delim is not included in the string but discarded.
*/
read_string(int delim, object in)
{
	int i;
	object c;

	i = 0;
	for (;;) {
		c = read_char(in);
		if (char_code(c) == delim)
			break;
		else if (cat(c) == cat_single_escape)
			c = read_char(in);
		if (i >= token->st.st_dim)
			too_long_string();
		token_buffer[i++] = char_code(c);
	}
	token->st.st_fillp = i;
	token->st.st_self[i] = '\0';
}

/*
	Read_constituent(in) reads
	a sequence of constituent characters from stream in
	and places it in token_buffer.
*/
read_constituent(object in)
{
	int i, j;
	object c;

	i = 0;
	for (;;) {
		c = read_char(in);
		if (cat(c) != cat_constituent) {
			unread_char(c, in);
			break;
		}
		j = char_code(c);
		token_buffer[i++] = j;
	}
	token->st.st_fillp = i;
}

Ldouble_quote_reader(int narg, object in, object c)
{
	check_arg(2);
	read_string('"', in);
	VALUES(0) = copy_simple_string(token);
	RETURN(1);
}

Ldispatch_reader(int narg, object in, object dc)
{
	object c, x, y;
	int i, d;

	check_arg(2);
	
	if (READtable->rt.rt_self[char_code(dc)].rte_dtab == NULL)
		FEerror("~C is not a dispatching macro character", 1, dc);

	c = read_char(in);
	d = digitp((int)char_code(c), 10);
	if (d >= 0) {
		i = 0;
		do {
			i = 10*i + d;
			c = read_char(in);
			d = digitp(char_code(c), 10);
		} while (d >= 0);
		y = MAKE_FIXNUM(i);
	} else
		y = Cnil;

	x = READtable->rt.rt_self[char_code(dc)].rte_dtab[char_code(c)];
	RETURN(funcall(4, x, in, c, y));
}

Lsingle_quote_reader(int narg, object in, object c)
{
	check_arg(2);
	VALUES(0) = CONS(Squote, CONS(read_object(in), Cnil));
	RETURN(1);
}

Lright_parenthesis_reader(int narg, object in, object c)
{
	check_arg(2);
	RETURN(0);
	/*  no result  */
}

/*
Lcomma_reader(){} in backq.c
*/

Lsemicolon_reader(int narg, object in, object c)
{
	check_arg(2);
	do
		c = read_char(in);
	while (char_code(c) != '\n');
	VALUES(0) = Cnil;
	RETURN(0);
	/*  no result  */
}

/*
Lbackquote_reader(){}
*/

/*
	sharpmacro routines
*/

Lsharp_C_reader(int narg, object in, object c, object d)
{
	object x, real, imag;

	check_arg(3);
	if (d != Cnil && !READsuppress)
		extra_argument('C', d);
	c = read_char(in);
	if (char_code(c) != '(')
		FEerror("A left parenthesis is expected.", 0);
	delimiting_char = code_char(')');
	real = read_object(in);
	if (real == OBJNULL)
		FEerror("No real part.", 0);
	delimiting_char = code_char(')');
	imag = read_object(in);
	if (imag == OBJNULL)
		FEerror("No imaginary part.", 0);
	delimiting_char = code_char(')');
	x = read_object(in);
	if (x != OBJNULL)
		FEerror("A right parenthesis is expected.", 0);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	if (contains_sharp_comma(real) ||
	    contains_sharp_comma(imag)) {
		x = alloc_object(t_complex);
		x->cmp.cmp_real = real;
		x->cmp.cmp_imag = imag;
	} else {
		check_type_number(&real);
		check_type_number(&imag);
		x = make_complex(real, imag);
	}
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_backslash_reader(int narg, object in, object c, object d)
{
	check_arg(3);
	if (d != Cnil && !READsuppress)
		if (!FIXNUMP(d) ||
		    fix(d) != 0)
			FEerror("~S is an illegal CHAR-FONT.", 1, d);
			/*  assuming that CHAR-FONT-LIMIT is 1  */
	unread_char(code_char('\\'), in);
	if (READsuppress) {
		(void)read_object(in);
		VALUES(0) = Cnil;
		RETURN(1);
	}
	READsuppress = TRUE;
	(void)read_object(in);
	READsuppress = FALSE;
	c = token;
	if (c->s.s_fillp == 1) {
		VALUES(0) = code_char(c->ust.ust_self[0]);
		RETURN(1);
	}
	if (string_equal(c, STreturn))
		VALUES(0) = code_char('\r');
	else if (string_equal(c, STspace))
		VALUES(0) = code_char(' ');
	else if (string_equal(c, STrubout))
		VALUES(0) = code_char('\177');
	else if (string_equal(c, STpage))
		VALUES(0) = code_char('\f');
	else if (string_equal(c, STtab))
		VALUES(0) = code_char('\t');
	else if (string_equal(c, STbackspace))
		VALUES(0) = code_char('\b');
	else if (string_equal(c, STlinefeed) || string_equal(c, STnewline))
		VALUES(0) = code_char('\n');
	/*	#\^x	*/
	else if (c->s.s_fillp == 2 && c->s.s_self[0] == '^')
		VALUES(0) = code_char(c->s.s_self[1] & 037);
	/*	#\C-x	*/
	else if (c->s.s_fillp == 3
		 && (c->s.s_self[0] == 'C' || c->s.s_self[0] == 'c')
		 && c->s.s_self[1] == '-')
		VALUES(0) = MAKE_CHARACTER(c->s.s_self[2], CONTROL_BIT, 0);
	/*	#\M-x	*/
	else if (c->s.s_fillp == 3
		 && (c->s.s_self[0] == 'M' || c->s.s_self[0] == 'm')
		 && c->s.s_self[1] == '-')
		VALUES(0) = MAKE_CHARACTER(c->s.s_self[2], META_BIT, 0);
	/*	#\C-M-x	*/
	else if (c->s.s_fillp == 5
		 && (c->s.s_self[0] == 'C' || c->s.s_self[0] == 'c')
		 && c->s.s_self[1] == '-'
		 && (c->s.s_self[2] == 'M' || c->s.s_self[2] == 'm')
		 && c->s.s_self[3] == '-')
		VALUES(0) = MAKE_CHARACTER(c->s.s_self[4],
					   CONTROL_BIT | META_BIT, 0);
	else if (c->s.s_self[0] =='\\' && c->s.s_fillp > 1) {
		int i, n;
		for (n = 0, i = 1;  i < c->s.s_fillp;  i++)
			if (c->s.s_self[i] < '0' || '7' < c->s.s_self[i])
				FEerror("Octal digit expected.", 0);
			else
				n = 8*n + c->s.s_self[i] - '0';
		VALUES(0) = code_char(n & 0377);
	} else
		FEerror("~S is an illegal character name.", 1, c);
	RETURN(1);
}

Lsharp_single_quote_reader(int narg, object in, object c, object d)
{
	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument('#', d);
	VALUES(0) = CONS(Sfunction, CONS(read_object(in), Cnil));
	RETURN(1);
}

#define	QUOTE	1
#define	EVAL	2
#define	LIST	3
#define	LISTA	4
#define	APPEND	5
#define	NCONC	6

object siScomma;


/*
 *----------------------------------------------------------------------
 *	Stack of unknown size
 *----------------------------------------------------------------------
 */

#define INCREMENT 64
#define ESTACK(st)	volatile int _esize = 0; object *(st), *(st ## 0);
#define ETOP(st)	(st ## 0)

#define EPUSH(st, val, count) \
    { int i; if (count == _esize) { \
      st = (object *)alloca(INCREMENT*sizeof(object)); \
      for ( i = 0; i < _esize; i++) \
	st[i] = st ## 0[i]; \
      (st ## 0) = st; st += _esize;\
      _esize += INCREMENT; \
    }; *(st)++ = (val);}


Lsharp_left_parenthesis_reader(int narg, object in, object c, object d)
{
	int dim, dimcount, i, a;
	object x, last;
	ESTACK(vsp);

	check_arg(3);
	if (Null(d) || READsuppress)
		dim = -1;
	else if (FIXNUMP(d))
		dim = fix(d);
	if (backq_level > 0) {
		unreadc_stream('(', in);
		x = read_object(in);
		a = backq_car(&x);
		if (a == APPEND || a == NCONC)
		  FEerror(",at or ,. has appeared in an illegal position.", 0);
		if (a == QUOTE) {
		  for (dimcount = 0;  !endp(x);  x = CDR(x), dimcount++)
		    EPUSH(vsp, CAR(x), dimcount);
		  goto L;
		}
		VALUES(0) =
		list(4, siScomma, Sapply, CONS(Squote, CONS(Svector, Cnil)),
			x);
		RETURN(1);
	}
	for (dimcount = 0 ;; dimcount++) {
	  delimiting_char = code_char(')');
	  x = read_object(in);
	  if (x == OBJNULL)
	    break;
	  EPUSH(vsp, x, dimcount);
	}
L:
	if (dim >= 0) {
		if (dimcount > dim)
			FEerror("Too many elements in #(...).", 0);
		if (dimcount == 0)
			FEerror("Cannot fill the vector #().", 0);
		else last = vsp[-1];
	} else
	  dim = dimcount;
	x = alloc_simple_vector(dim, aet_object);
	x->v.v_self = (object *)alloc_relblock(dim * sizeof(object),
					       sizeof(object));
	for (i = 0; i < dim; i++)
		x->v.v_self[i] = (i < dimcount) ? ETOP(vsp)[i] : last;
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_asterisk_reader(int narg, object in, object c, object d)
{
	int dim, dimcount, i;
	object x, last, elt;
	ESTACK(vsp);

	check_arg(3);
	if (READsuppress) {
		read_constituent(in);
		VALUES(0) = Cnil;
		RETURN(1);
	}
	if (Null(d))
		dim = -1;
	else if (FIXNUMP(d))
		dim = fix(d);
	for (dimcount = 0 ;; dimcount++) {
		if (stream_at_end(in))
			break;
		x = read_char(in);
		if (char_code(x) != '0' && char_code(x) != '1') {
			unread_char(x, in);
			break;
		}
		EPUSH(vsp, x, dimcount);
	}	
	if (dim >= 0) {
		if (dimcount > dim)
			FEerror("Too many elements in #*....", 0);
		if (dimcount == 0)
			FEerror("Cannot fill the bit-vector #*.", 0);
		else last = vsp[-1];
	} else {
	  dim = dimcount;	/* Beppe ? */
	  last = MAKE_FIXNUM(0);
	}
	x = alloc_simple_bitvector(dim);
	x->bv.bv_self = alloc_relblock((dim + CHAR_BIT - 1)/CHAR_BIT,
				       sizeof(char));
	for (i = 0; i < dim; i++) {
		elt = (i < dimcount) ? ETOP(vsp)[i] : last;
		if (char_code(elt) == '0')
			x->bv.bv_self[i/CHAR_BIT] &= ~(0200 >> i%CHAR_BIT);
		else
			x->bv.bv_self[i/CHAR_BIT] |= 0200 >> i%CHAR_BIT;
		}
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_colon_reader(int narg, object in, object c, object d)
{
	int length;
	enum chattrib a;

	check_arg(3);
	if (d != Cnil && !READsuppress)
		extra_argument(':', d);
	c = read_char(in);
	a = cat(c);
	escape_flag = FALSE;
	length = 0;
	goto L;
	for (;;) {
		if (length >= token->st.st_dim)
			too_long_token();
		token_buffer[length++] = char_code(c);
	K:
		if (stream_at_end(in))
			goto M;
		c = read_char(in);
		a = cat(c);
	L:
		if (a == cat_single_escape) {
			c = read_char(in);
			a = cat_constituent;
			escape_flag = TRUE;
		} else if (a == cat_multiple_escape) {
			escape_flag = TRUE;
			for (;;) {
				if (stream_at_end(in))
					end_of_stream(in);
				c = read_char(in);
				a = cat(c);
				if (a == cat_single_escape) {
					c = read_char(in);
					a = cat_constituent;
				} else if (a == cat_multiple_escape)
					break;
				if (length >= token->st.st_dim)
					too_long_token();
				token_buffer[length++] = char_code(c);
			}
			goto K;
		} else if ('a' <= char_code(c) && char_code(c) <= 'z')
			c = code_char(char_code(c) - ('a' - 'A'));
		if (a == cat_whitespace || a == cat_terminating)
			break;
	}
	if (preserving_whitespace_flag || cat(c) != cat_whitespace)
		unread_char(c, in);

M:
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	token->st.st_fillp = length;
	VALUES(0) = make_symbol(copy_simple_string(token));
	RETURN(1);
}

Lsharp_dot_reader(int narg, object in, object c, object d)
{
	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument('.', d);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	RETURN(eval(read_object(in)));
}

Lsharp_comma_reader(int narg, object in, object c, object d)
{
	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument(',', d);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	RETURN(eval(read_object(in)));
}

siLsharp_comma_reader_for_compiler(int narg, object in, object c, object d)
{
	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument(',', d);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	VALUES(0) = CONS(siSsharp_comma, read_object(in));
	RETURN(1);
}

/*
	For fasload.
*/
Lsharp_exclamation_reader(int narg, object in, object c, object d)
{
	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument('!', d);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	eval(read_object(in));
	RETURN(0);
}

Lsharp_B_reader(int narg, object in, object c, object d)
{
	int i; object x;

	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument('B', d);
	read_constituent(in);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	x = parse_number(token_buffer, token->st.st_fillp, &i, 2);
	if (x == OBJNULL || i != token->st.st_fillp)
		FEerror("Cannot parse the #B readmacro.", 0);
	if (type_of(x) == t_shortfloat ||
	    type_of(x) == t_longfloat)
		FEerror("The float ~S appeared after the #B readmacro.",
			1, x);
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_O_reader(int narg, object in, object c, object d)
{
	int i; object x;

	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument('O', d);
	read_constituent(in);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	x = parse_number(token_buffer, token->st.st_fillp, &i, 8);
	if (x == OBJNULL || i != token->st.st_fillp)
		FEerror("Cannot parse the #O readmacro.", 0);
	if (type_of(x) == t_shortfloat ||
	    type_of(x) == t_longfloat)
		FEerror("The float ~S appeared after the #O readmacro.",
			1, x);
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_X_reader(int narg, object in, object c, object d)
{
	int i; object x;

	check_arg(3);
	if(d != Cnil && !READsuppress)
		extra_argument('X', d);
	read_constituent(in);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	x = parse_number(token_buffer, token->st.st_fillp, &i, 16);
	if (x == OBJNULL || i != token->st.st_fillp)
		FEerror("Cannot parse the #X readmacro.", 0);
	if (type_of(x) == t_shortfloat ||
	    type_of(x) == t_longfloat)
		FEerror("The float ~S appeared after the #X readmacro.",
			1, x);
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_R_reader(int narg, object in, object c, object d)
{
	int radix, i;
	object x;

	check_arg(3);
	if (READsuppress)
		radix = 10;
	else if (FIXNUMP(d)) {
		radix = fix(d);
		if (radix > 36 || radix < 2)
			FEerror("~S is an illegal radix.", 1, d);
	} else
		FEerror("No radix was supplied in the #R readmacro.", 0);
	read_constituent(in);
	if (READsuppress) {
		VALUES(0) = Cnil;
		RETURN(1);
	}
	x = parse_number(token_buffer, token->st.st_fillp, &i, radix);
	if (x == OBJNULL || i != token->st.st_fillp)
		FEerror("Cannot parse the #R readmacro.", 0);
	if (type_of(x) == t_shortfloat ||
	    type_of(x) == t_longfloat)
		FEerror("The float ~S appeared after the #R readmacro.",
			1, x);
	VALUES(0) = x;
	RETURN(1);
}

Lsharp_A_reader(){}

Lsharp_S_reader(){}

Lsharp_eq_reader(int narg, object in, object c, object d)
{
	int i;

	check_arg(3);
	if (READsuppress) RETURN(0);
	if (Null(d))
		FEerror("The #= readmacro requires an argument.", 0);
	for (i = 0;  i < sharp_eq_context_max;  i++)
		if (eql(sharp_eq_context[i].sharp_index, d))
			FEerror("Duplicate definitions for #~D=.",
				1, d);
	if (sharp_eq_context_max >= SHARP_EQ_CONTEXT_SIZE)
		FEerror("Too many #= definitions.", 0);
	i = sharp_eq_context_max++;
	sharp_eq_context[i].sharp_index = d;
	sharp_eq_context[i].sharp_sharp = OBJNULL;
	VALUES(0) = sharp_eq_context[i].sharp_eq = read_object(in);
	if (sharp_eq_context[i].sharp_eq == sharp_eq_context[i].sharp_sharp)
		FEerror("#~D# is defined by itself.",
			1, sharp_eq_context[i].sharp_index);
	RETURN(1);
}

Lsharp_sharp_reader(int narg, object in, object c, object d)
{
	int i;

	check_arg(3);
	if (READsuppress) RETURN(0);
	if (Null(d))
		FEerror("The ## readmacro requires an argument.", 0);
	for (i = 0; i < sharp_eq_context_max ;  i++)
	  if (eql(sharp_eq_context[i].sharp_index, d)) {
	    if (sharp_eq_context[i].sharp_sharp == OBJNULL)
	      sharp_eq_context[i].sharp_sharp = MAKE_SPICE(spice_count++);
	    VALUES(0) = sharp_eq_context[i].sharp_sharp;
	    RETURN(1);
	  }
	FEerror("#~D# is undefined.", 1, d);
}

patch_sharp_cons(object x)
{
	for (;;) {
		CAR(x) = patch_sharp(CAR(x));
		if (type_of(CDR(x)) == t_cons)
			x = CDR(x);
		else {
			CDR(x) = patch_sharp(CDR(x));
			break;
		}
	}
}

object
patch_sharp(object x)
{
	cs_check(x);

	switch (type_of(x)) {

	case t_spice:
	{
		int i;

		for (i = 0;  i < sharp_eq_context_max;  i++)
			if (sharp_eq_context[i].sharp_sharp == x)
				return(sharp_eq_context[i].sharp_eq);
		break;
	}
	case t_cons:
	/*
		CAR(x) = patch_sharp(CAR(x));
		CDR(x) = patch_sharp(CDR(x));
	*/
		patch_sharp_cons(x);
		break;

	case t_vector:
	{
		int i;

		for (i = 0;  i < x->v.v_fillp;  i++)
			x->v.v_self[i] = patch_sharp(x->v.v_self[i]);
		break;
	}
	case t_array:
	{
		int i, j;

		for (i = 0, j = 1;  i < x->a.a_rank;  i++)
			j *= x->a.a_dims[i];
		for (i = 0;  i < j;  i++)
			x->a.a_self[i] = patch_sharp(x->a.a_self[i]);
		break;
	}
	}
	return(x);
}

Lsharp_plus_reader(){}

Lsharp_minus_reader(){}

Lsharp_less_than_reader(){}

Lsharp_whitespace_reader(){}

Lsharp_right_parenthesis_reader(){}

Lsharp_vertical_bar_reader(int narg, object in, object ch, object d)
{
	int c;
	int level = 0;

	check_arg(3);
	if (d != Cnil && !READsuppress)
		extra_argument('|', d);
	for (;;) {
		c = char_code(read_char(in));
	L:
		if (c == '#') {
			c = char_code(read_char(in));
			if (c == '|')
				level++;
		} else if (c == '|') {
			c = char_code(read_char(in));
			if (c == '#') {
				if (level == 0)
					break;
				else
					--level;
			} else
				goto L;
		}
	}
	RETURN(0);
	/*  no result  */
}

Ldefault_dispatch_macro(int narg)
{
	FEerror("Undefined dispatch macro character.", 0);
}

/*
	#" ... " returns the pathname with namestring ... .
*/
Lsharp_double_quote_reader(int narg, object in, object c, object d)
{
	check_arg(3);

	if (d != Cnil && !READsuppress)
		extra_argument('"', d);
	unread_char(c, in);
	VALUES(0) = coerce_to_pathname(read_object(in));
	RETURN(1);
}

/*
	#$ fixnum returns a random-state with the fixnum
	as its content.
*/
Lsharp_dollar_reader(int narg, object in, object c, object d)
{
	check_arg(3);
	if (d != Cnil && !READsuppress)
		extra_argument('$', d);
	c = read_object(in);
	if (!FIXNUMP(c))
		FEerror("Cannot make a random-state with the value ~S.",
			1, c);
	VALUES(0) = alloc_object(t_random);
	VALUES(0)->rnd.rnd_value = fix(c);
	RETURN(1);
}

/*
	readtable routines
*/

object
copy_readtable(object from, object to)
{
	struct rtent *rtab;
	int i, j;

	if (Null(to)) {
		to = alloc_object(t_readtable);
		to->rt.rt_self = NULL;
			/*  Saving for GC.  */
		to->rt.rt_self
		= rtab
 		= (struct rtent *)
	  		alloc_contblock(RTABSIZE * sizeof(struct rtent));
		memcpy(rtab, from->rt.rt_self,
			 RTABSIZE * sizeof(struct rtent));
/*
		for (i = 0;  i < RTABSIZE;  i++)
			rtab[i] = from->rt.rt_self[i];
*/
				/*  structure assignment  */
	} else
	  rtab=to->rt.rt_self;
	for (i = 0;  i < RTABSIZE;  i++)
		if (from->rt.rt_self[i].rte_dtab != NULL) {
			rtab[i].rte_dtab
 			= (object *)alloc_contblock(RTABSIZE * sizeof(object));
			memcpy(rtab[i].rte_dtab, from->rt.rt_self[i].rte_dtab,
			      RTABSIZE * sizeof(object *));
/*
			for (j = 0;  j < RTABSIZE;  j++)
				rtab[i].rte_dtab[j]
				= from->rt.rt_self[i].rte_dtab[j];
*/
		}
	return(to);
}

object
current_readtable()
{
	object r;

	r = symbol_value(Vreadtable);
	if (type_of(r) != t_readtable) {
	  Vreadtable->s.s_dbind = copy_readtable(standard_readtable, Cnil);
	  FEerror("The value of *READTABLE*, ~S, was not a readtable.",
		  1, r);
	}
	return(r);
}


@(defun read (&optional (strm `symbol_value(Vstandard_input)`)
			(eof_errorp Ct)
			eof_value
			recursivep
	      &aux x)
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else
	    read_ch_fun = readc;
	} else
#ifdef CLOS
	  if (type_of(strm) == t_instance)
	    read_ch_fun = interactive_readc;
	  else
#endif CLOS
	    FEwrong_type_argument(Sstream, strm);
	if (Null(recursivep))
		preserving_whitespace_flag = FALSE;
	detect_eos_flag = TRUE;
	if (Null(recursivep))
		x = read_object_non_recursive(strm);
	else
		x = read_object_recursive(strm);
	if (x == OBJNULL) {
		if (Null(eof_errorp) && Null(recursivep))
			@(return eof_value)
		end_of_stream(strm);
	}
	@(return x)
@)

@(defun read_preserving_whitespace
	(&optional (strm `symbol_value(Vstandard_input)`)
		   (eof_errorp Ct)
		   eof_value
		   recursivep
	 &aux x)
	object c;
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else
	    read_ch_fun = readc;
	} else
#ifdef CLOS
	  if (type_of(strm) == t_instance)
	    read_ch_fun = interactive_readc;
	  else
#endif CLOS
	    FEwrong_type_argument(Sstream, strm);
	while (!stream_at_end(strm)) {
		c = read_char(strm);
		if (cat(c) != cat_whitespace) {
			unread_char(c, strm);
			goto READ;
		}
	}
	if (Null(eof_errorp) && Null(recursivep))
		@(return eof_value)
	end_of_stream(strm);

READ:
	if (Null(recursivep))
		preserving_whitespace_flag = TRUE;
	if (Null(recursivep))
		x = read_object_non_recursive(strm);
	else
		x = read_object_recursive(strm);
	@(return x)
@)

@(defun read_delimited_list
	(d
	 &optional (strm `symbol_value(Vstandard_input)`)
		   recursivep
	 &aux l x)

	object *p;

	int i, nr;
	bool e;
	volatile int old_sharp_eq_context_max;
	struct sharp_eq_context_struct
		old_sharp_eq_context[SHARP_EQ_CONTEXT_SIZE];
	volatile int old_backq_level;

@

	check_type_character(&d);
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
	check_type_stream(&strm);
	if (Null(recursivep)) {
		old_sharp_eq_context_max = sharp_eq_context_max;
		for (i = 0;  i < sharp_eq_context_max;  i++)
			old_sharp_eq_context[i] = sharp_eq_context[i];
		old_backq_level = backq_level;
		setup_READ();
		sharp_eq_context_max = 0;
		backq_level = 0;
		if ((nr = frs_push(FRS_PROTECT, Cnil)) != 0) {
			e = TRUE;
			goto L;
		}
	}
	l = Cnil;
	p = &l;
	preserving_whitespace_flag = FALSE;	/*  necessary?  */
	for (;;) {
		delimiting_char = d;
		x = read_object_recursive(strm);
		if (x == OBJNULL)
			break;
		*p = CONS(x, Cnil);
		p = &(CDR((*p)));
	}
	if (Null(recursivep)) {
		if (sharp_eq_context_max > 0)
			l = patch_sharp(l);
		e = FALSE;
	L:
		frs_pop();
		sharp_eq_context_max = old_sharp_eq_context_max;
		for (i = 0;  i < sharp_eq_context_max;  i++)
			sharp_eq_context[i] = old_sharp_eq_context[i];
		backq_level = old_backq_level;
		if (e) unwind(nlj_fr, nlj_tag, nr);
	}
	@(return l)
@)

@(defun read_line (&optional (strm `symbol_value(Vstandard_input)`)
			     (eof_errorp Ct)
			     eof_value
			     recursivep
		   &aux c)
	int i;
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else {
	    if (stream_at_end(strm)) {
	      if (Null(eof_errorp) && Null(recursivep))
		@(return eof_value)
		  else
		    end_of_stream(strm);
	    }
	    i = 0;
	    for (;;) {
	      c = read_char(strm);
	      if (char_code(c) == '\n') {
		c = Cnil;
		break;
	      }
	      if (i >= token->st.st_dim)
		too_long_string();
	      token->st.st_self[i++] = char_code(c);
	      if (stream_at_end(strm)) {
		c = Ct;
		break;
	      }
	    }
#ifdef CRLF
	    if (i > 0 && token->st.st_self[i-1] == '\r') i--;
#endif CRLF
	    token->st.st_fillp = i;
	    @(return `copy_simple_string(token)` c)
	    }
	} else
#ifdef CLOS
	if (type_of(strm) == t_instance)
	     RETURN(funcall(2, Sstream_read_line, strm));
        else
#endif
	  FEerror("~S is not a stream.", 1, strm);
@)

@(defun read_char (&optional (strm `symbol_value(Vstandard_input)`)
			     (eof_errorp Ct)
			     eof_value
			     recursivep)
@
        if (Null(strm))
	    strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
	    strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else {
	    if (stream_at_end(strm)) {
	      if (Null(eof_errorp) && Null(recursivep))
		@(return eof_value)
		  else
		    end_of_stream(strm);
	    }
	    @(return `read_char(strm)`)
	    }
	} else
#ifdef CLOS
	if (type_of(strm) == t_instance)
	     RETURN(funcall(2, Sstream_read_char, strm));
	else
#endif
	  FEerror("~S is not a stream.", 1, strm);
@)

@(defun unread_char (c &optional (strm `symbol_value(Vstandard_input)`))
@
	check_type_character(&c);
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else {
	    unread_char(c, strm);
	    @(return Cnil)
	    }
	} else
#ifdef CLOS
	if (type_of(strm) == t_instance)
	     RETURN(funcall(3, Sstream_unread_char, strm, c));
	else
#endif
	  FEerror("~S is not a stream.", 1, strm);
@)

@(defun peek_char (&optional peek_type
			     (strm `symbol_value(Vstandard_input)`)
			     (eof_errorp Ct)
			     eof_value
			     recursivep)
	object c;
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else {
	    setup_READtable();
	    if (Null(peek_type)) {
	      if (stream_at_end(strm)) {
		if (Null(eof_errorp) && Null(recursivep))
		  @(return eof_value)
		    else
		      end_of_stream(strm);
	      }
	      c = read_char(strm);
	      unread_char(c, strm);
	      @(return c)
	      }
	    if (peek_type == Ct) {
	      while (!stream_at_end(strm)) {
		c = read_char(strm);
		if (cat(c) != cat_whitespace) {
		  unread_char(c, strm);
		  @(return c)
		  }
	      }
	      if (Null(eof_errorp))
		@(return eof_value)
		  else
		    end_of_stream(strm);
	    }
	    check_type_character(&peek_type);
	    while (!stream_at_end(strm)) {
	      c = read_char(strm);
	      if (char_eq(c, peek_type)) {
		unread_char(c, strm);
		@(return c)
		}
	    }
	    if (Null(eof_errorp))
	      @(return eof_value)
		else
		  end_of_stream(strm);
	  }
	} else
#ifdef CLOS
	if (type_of(strm) == t_instance)
	     RETURN(funcall(3, Sstream_peek_char, strm, peek_type));
	else
#endif
	  FEerror("~S is not a stream.", 1, strm);
@)

@(defun listen (&optional (strm `symbol_value(Vstandard_input)`))
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else {
	    if (listen_stream(strm))
	      @(return Ct)
	    else
	      @(return Cnil)
	      }
	}
	else
#ifdef CLOS
	if (type_of(strm) == t_instance)
	     RETURN(funcall(2, Sstream_listen, strm));
	else
#endif
	  FEerror("~S is not a stream.", 1, strm);
@)

@(defun read_char_no_hang (&optional (strm `symbol_value(Vstandard_input)`)
			             (eof_errorp Ct)
			             eof_value
			             recursivep)
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
	check_type_stream(&strm);
	if (!listen_stream(strm))
		/* Incomplete! */
		@(return Cnil)
	@(return `read_char(strm)`)
@)

@(defun clear_input (&optional (strm `symbol_value(Vstandard_input)`))
@
	if (Null(strm))
		strm = symbol_value(Vstandard_input);
	else if (strm == Ct)
		strm = symbol_value(Vterminal_io);
RETRY:	if (type_of(strm) == t_stream) {
          if (strm->sm.sm_mode == (short)smm_synonym) {
	    strm = symbol_value(strm->sm.sm_object0);
	    goto RETRY;
	  }
	  else {
	    clear_input_stream(strm);
	    @(return Cnil)
	     }
	    } else
#ifdef CLOS
	if (type_of(strm) == t_instance)
	     RETURN(funcall(2, Sstream_clear_input, strm));
  	else
#endif
	  FEerror("~S is not a stream.", 1, strm);
@)

@(defun parse_integer (strng
		       &key start
			    end
			    (radix `MAKE_FIXNUM(10)`)
			    junk_allowed
		       &aux x)
	int s, e, ep;
@
	check_type_string(&strng);
	get_string_start_end(strng, start, end, &s, &e);
	if (!FIXNUMP(radix) ||
	    fix(radix) < 2 || fix(radix) > 36)
		FEerror("~S is an illegal radix.", 1, radix);
	setup_READtable();
	while (READtable->rt.rt_self[strng->st.st_self[s]].rte_chattrib
	       == cat_whitespace && s < e)
		s++;
	if (s >= e) {
		if (junk_allowed != Cnil)
			@(return Cnil `MAKE_FIXNUM(s)`)
		else
			goto CANNOT_PARSE;
	}
	x = parse_integer(strng->st.st_self+s, e-s, &ep, fix(radix));
	if (x == OBJNULL) {
		if (junk_allowed != Cnil)
			@(return Cnil `MAKE_FIXNUM(ep+s)`)
		else
			goto CANNOT_PARSE;
	}
	if (junk_allowed != Cnil)
		@(return x `MAKE_FIXNUM(ep+s)`)
	for (s += ep ;  s < e;  s++)
		if (READtable->rt.rt_self[strng->st.st_self[s]].rte_chattrib
		    != cat_whitespace)
			goto CANNOT_PARSE;
	@(return x `MAKE_FIXNUM(e)`)

CANNOT_PARSE:
	FEerror("Cannot parse an integer in the string ~S.", 1, strng);
@)

@(defun read_byte (binary_input_stream
		   &optional eof_errorp eof_value)
	int c;
@
	check_type_stream(&binary_input_stream);
	if (stream_at_end(binary_input_stream)) {
		if (Null(eof_errorp))
			@(return eof_value)
		else
			end_of_stream(binary_input_stream);
	}
	c = readc_stream(binary_input_stream);
	@(return `MAKE_FIXNUM(c)`)
@)

@(defun read_bytes (stream string start end)
	int is, ie, c; FILE *fp;
@
	check_type_stream(&stream);
        is = fix(start), ie = fix(end);
	fp = stream->sm.sm_fp;
	if (fp == NULL) fp = stream->sm.sm_object0->sm.sm_fp;
	c = fread (string->ust.ust_self + is, sizeof(unsigned char),
		   ie - is,
		   fp);
	@(return `MAKE_FIXNUM(c)`)
@)



@(defun copy_readtable (&o (from `current_readtable()`) to)
@
	if (Null(from)) {
		from = standard_readtable;
		if (to != Cnil)
			check_type_readtable(&to);
		to = copy_readtable(from, to);
		to->rt.rt_self['#'].rte_dtab['!']
		= default_dispatch_macro;
		/*  We must forget #! macro.  */
		@(return to)
	}
	check_type_readtable(&from);
	if (to != Cnil)
		check_type_readtable(&to);
	@(return `copy_readtable(from, to)`)
@)

Lreadtablep(int narg, object readtable)
{
	check_arg(1);

	if (type_of(readtable) == t_readtable)
		VALUES(0) = Ct;
	else
		VALUES(0) = Cnil;
	RETURN(1);
}

@(defun set_syntax_from_char (tochr fromchr
			      &o (tordtbl `current_readtable()`)
				 fromrdtbl)
	int i;
@
	check_type_character(&tochr);
	check_type_character(&fromchr);
	check_type_readtable(&tordtbl);
	if (Null(fromrdtbl))
		fromrdtbl = standard_readtable;
	else
		check_type_readtable(&fromrdtbl);
	tordtbl->rt.rt_self[char_code(tochr)].rte_chattrib
	= fromrdtbl->rt.rt_self[char_code(fromchr)].rte_chattrib;
	tordtbl->rt.rt_self[char_code(tochr)].rte_macro
	= fromrdtbl->rt.rt_self[char_code(fromchr)].rte_macro;
	if ((tordtbl->rt.rt_self[char_code(tochr)].rte_dtab
	     = fromrdtbl->rt.rt_self[char_code(fromchr)].rte_dtab) != NULL) {
		tordtbl->rt.rt_self[char_code(tochr)].rte_dtab
		= (object *)alloc_contblock(RTABSIZE * sizeof(object));
		memcpy(tordtbl->rt.rt_self[char_code(tochr)].rte_dtab,
			fromrdtbl->rt.rt_self[char_code(fromchr)].rte_dtab,
			RTABSIZE * sizeof(object *));
/*
		for (i = 0;  i < RTABSIZE;  i++)
			tordtbl->rt.rt_self[char_code(tochr)]
			.rte_dtab[i]
			= fromrdtbl->rt.rt_self[char_code(fromchr)]
			  .rte_dtab[i];
*/
	}
	@(return Ct)
@)

@(defun set_macro_character (chr fnc
			     &optional ntp
				       (rdtbl `current_readtable()`))
	int c;
@
	check_type_character(&chr);
	check_type_readtable(&rdtbl);
	c = char_code(chr);
	if (ntp != Cnil)
		rdtbl->rt.rt_self[c].rte_chattrib
		= cat_non_terminating;
	else
		rdtbl->rt.rt_self[c].rte_chattrib
		= cat_terminating;
	rdtbl->rt.rt_self[c].rte_macro = fnc;
	@(return Ct)
@)

@(defun get_macro_character (chr &o (rdtbl `current_readtable()`))
	object m;
@
	check_type_character(&chr);
/* fix to allow NIL as readtable argument. Beppe */
	if (Null(rdtbl))
		rdtbl = standard_readtable;
	else
		check_type_readtable(&rdtbl);
	if ((m = rdtbl->rt.rt_self[char_code(chr)].rte_macro)
	    == OBJNULL)
		@(return Cnil)
	if (rdtbl->rt.rt_self[char_code(chr)].rte_chattrib
	    == cat_non_terminating)
		@(return m Ct)
	else
		@(return m Cnil)
@)

@(defun make_dispatch_macro_character (chr
	&optional ntp (rdtbl `current_readtable()`))
	int i;
@
	check_type_character(&chr);
	check_type_readtable(&rdtbl);
	if (ntp != Cnil)
		rdtbl->rt.rt_self[char_code(chr)].rte_chattrib
		= cat_non_terminating;
	else
		rdtbl->rt.rt_self[char_code(chr)].rte_chattrib
		= cat_terminating;
	rdtbl->rt.rt_self[char_code(chr)].rte_dtab
	= (object *)
	  alloc_contblock(RTABSIZE * sizeof(object));
	for (i = 0;  i < RTABSIZE;  i++)
		rdtbl->rt.rt_self[char_code(chr)].rte_dtab[i]
		= default_dispatch_macro;
	rdtbl->rt.rt_self[char_code(chr)].rte_macro = dispatch_reader;
	@(return Ct)
@)

@(defun set_dispatch_macro_character (dspchr subchr fnc
	&optional (rdtbl `current_readtable()`))
@
	check_type_character(&dspchr);
	check_type_character(&subchr);
	check_type_readtable(&rdtbl);
	if (rdtbl->rt.rt_self[char_code(dspchr)].rte_macro != dispatch_reader
	    || rdtbl->rt.rt_self[char_code(dspchr)].rte_dtab == NULL)
		FEerror("~S is not a dispatch character.", 1, dspchr);
	rdtbl->rt.rt_self[char_code(dspchr)]
	.rte_dtab[char_code(subchr)] = fnc;
	if ('a' <= char_code(subchr) && char_code(subchr) <= 'z')
		rdtbl->rt.rt_self[char_code(dspchr)]
		.rte_dtab[char_code(subchr) - ('a' - 'A')] = fnc;

	@(return Ct)
@)

@(defun get_dispatch_macro_character (dspchr subchr
	&optional (rdtbl `current_readtable()`))
@
	check_type_character(&dspchr);
	check_type_character(&subchr);
/* fix to allow NIL as readtable argument. Beppe */
	if (Null(rdtbl))
		rdtbl = standard_readtable;
	else
		check_type_readtable(&rdtbl);
	if (rdtbl->rt.rt_self[char_code(dspchr)].rte_macro != dispatch_reader
	    || rdtbl->rt.rt_self[char_code(dspchr)].rte_dtab == NULL)
		FEerror("~S is not a dispatch character.", 1, dspchr);
	if (digitp(char_code(subchr),10) >= 0) @(return Cnil)
	else @(return `rdtbl->rt.rt_self[char_code(dspchr)]
		  .rte_dtab[char_code(subchr)]`)
@)

object
string_to_object(object x)
{
	object in;

	in = make_string_input_stream(x, 0, x->st.st_fillp);
	preserving_whitespace_flag = FALSE;
	detect_eos_flag = FALSE;
	x = read_object(in);
	return(x);
}

siLstring_to_object(int narg, object str)
{
	check_arg(1);

	check_type_string(&str);
	VALUES(0) = string_to_object(str);
	RETURN(1);
}

siLstandard_readtable(int narg)
{
	check_arg(0);

	VALUES(0) = standard_readtable;
	RETURN(1);
}

too_long_token()
{
	char *q;
	int i;

	q = alloc_contblock(token->st.st_dim*2);
	memcpy(q, token->st.st_self, token->st.st_dim);
	token->st.st_self = q;
	token->st.st_dim *= 2;
}

too_long_string()
{
	char *q;

	q = alloc_contblock(token->st.st_dim*2);
	memcpy(q, token->st.st_self, token->st.st_dim);
	token->st.st_self = q;
	token->st.st_dim *= 2;
}

extra_argument(int c, object d)
{
	FEerror("~S is an extra argument for the #~C readmacro.",
		2, d, code_char(c));
}


#define	make_cf(f)	make_cfun((f), Cnil, NULL)

init_read()
{
	struct rtent *rtab;
	object *dtab;
	int i;

	standard_readtable = alloc_object(t_readtable);
	enter_mark_origin(&standard_readtable);

	standard_readtable->rt.rt_self
	= rtab
	= (struct rtent *)
	  alloc_contblock(RTABSIZE * sizeof(struct rtent));
	for (i = 0;  i < RTABSIZE;  i++) {
		rtab[i].rte_chattrib = cat_constituent;
		rtab[i].rte_macro = OBJNULL;
		rtab[i].rte_dtab = NULL;
	}

	dispatch_reader = make_cf(Ldispatch_reader);
	enter_mark_origin(&dispatch_reader);

	rtab['\t'].rte_chattrib = cat_whitespace;
	rtab['\n'].rte_chattrib = cat_whitespace;
	rtab['\f'].rte_chattrib = cat_whitespace;
	rtab['\r'].rte_chattrib = cat_whitespace;
	rtab[' '].rte_chattrib = cat_whitespace;
	rtab['"'].rte_chattrib = cat_terminating;
	rtab['"'].rte_macro = make_cf(Ldouble_quote_reader);
	rtab['#'].rte_chattrib = cat_non_terminating;
	rtab['#'].rte_macro = dispatch_reader;
	rtab['\''].rte_chattrib = cat_terminating;
	rtab['\''].rte_macro = make_cf(Lsingle_quote_reader);
	rtab['('].rte_chattrib = cat_terminating;
	rtab['('].rte_macro = make_cf(Lleft_parenthesis_reader);
	rtab[')'].rte_chattrib = cat_terminating;
	rtab[')'].rte_macro = make_cf(Lright_parenthesis_reader);
/*
	rtab[','].rte_chattrib = cat_terminating;
	rtab[','].rte_macro = make_cf(Lcomma_reader);
*/
	rtab[';'].rte_chattrib = cat_terminating;
	rtab[';'].rte_macro = make_cf(Lsemicolon_reader);
	rtab['\\'].rte_chattrib = cat_single_escape;
/*
	rtab['`'].rte_chattrib = cat_terminating;
	rtab['`'].rte_macro = make_cf(Lbackquote_reader);
*/
	rtab['|'].rte_chattrib = cat_multiple_escape;
/*
	rtab['|'].rte_macro = make_cf(Lvertical_bar_reader);
*/

	default_dispatch_macro = make_cf(Ldefault_dispatch_macro);

	rtab['#'].rte_dtab
	= dtab
	= (object *)alloc_contblock(RTABSIZE * sizeof(object));
	for (i = 0;  i < RTABSIZE;  i++)
		dtab[i] = default_dispatch_macro;
	dtab['C'] = dtab['c'] = make_cf(Lsharp_C_reader);
	dtab['\\'] = make_cf(Lsharp_backslash_reader);
	dtab['\''] = make_cf(Lsharp_single_quote_reader);
	dtab['('] = make_cf(Lsharp_left_parenthesis_reader);
	dtab['*'] = make_cf(Lsharp_asterisk_reader);
	dtab[':'] = make_cf(Lsharp_colon_reader);
	dtab['.'] = make_cf(Lsharp_dot_reader);
	dtab['!'] = make_cf(Lsharp_exclamation_reader);
	/*  Used for fasload only. */
	dtab[','] = make_cf(Lsharp_comma_reader);
	dtab['B'] = dtab['b'] = make_cf(Lsharp_B_reader);
	dtab['O'] = dtab['o'] = make_cf(Lsharp_O_reader);
	dtab['X'] = dtab['x'] = make_cf(Lsharp_X_reader);
	dtab['R'] = dtab['r'] = make_cf(Lsharp_R_reader);
/*
	dtab['A'] = dtab['a'] = make_cf(Lsharp_A_reader);
	dtab['S'] = dtab['s'] = make_cf(Lsharp_S_reader);
*/
	dtab['A'] = dtab['a'] = make_si_ordinary("SHARP-A-READER");
	dtab['S'] = dtab['s'] = make_si_ordinary("SHARP-S-READER");

	dtab['='] = make_cf(Lsharp_eq_reader);
	dtab['#'] = make_cf(Lsharp_sharp_reader);
	dtab['+'] = make_cf(Lsharp_plus_reader);
	dtab['-'] = make_cf(Lsharp_minus_reader);
/*
	dtab['<'] = make_cf(Lsharp_less_than_reader);
*/
	dtab['|'] = make_cf(Lsharp_vertical_bar_reader);
	dtab['"'] = make_cf(Lsharp_double_quote_reader);
	/*  This is specific to this implimentation  */
	dtab['$'] = make_cf(Lsharp_dollar_reader);
	/*  This is specific to this implimentation  */
/*
	dtab[' '] = dtab['\t'] = dtab['\n'] = dtab['\f']
	= make_cf(Lsharp_whitespace_reader);
	dtab[')'] = make_cf(Lsharp_right_parenthesis_reader);
*/

	init_backq();

	Vreadtable
 	= make_special("*READTABLE*",
		       copy_readtable(standard_readtable, Cnil));
	Vreadtable->s.s_dbind->rt.rt_self['#'].rte_dtab['!']
	= default_dispatch_macro; /*  We must forget #! macro.  */
	Vread_default_float_format
	= make_special("*READ-DEFAULT-FLOAT-FORMAT*",
		       Ssingle_float);
	Vread_base = make_special("*READ-BASE*", MAKE_FIXNUM(10));
	Vread_suppress = make_special("*READ-SUPPRESS*", Cnil);

	Kstart = make_keyword("START");
	Kend = make_keyword("END");
	Kradix = make_keyword("RADIX");
	Kjunk_allowed = make_keyword("JUNK-ALLOWED");

	READtable = symbol_value(Vreadtable);
	enter_mark_origin(&READtable);
	READdefault_float_format = 'S';
	READbase = 10;
	READsuppress = FALSE;

	sharp_eq_context_max = 0;

	siSsharp_comma = make_si_ordinary("#,");
	enter_mark_origin(&siSsharp_comma);

	delimiting_char = OBJNULL;
	enter_mark_origin(&delimiting_char);

	detect_eos_flag = FALSE;
	in_list_flag = FALSE;
	dot_flag = FALSE;

	big_register_0 = alloc_object(t_bignum);
	big_register_0->big.big_car = 0;
	big_register_0->big.big_cdr = NULL;
	enter_mark_origin(&big_register_0);
/*
	NOTE:

		The value of big_register_0 changes
		along the execution of the read routines.
*/

	read_ch_fun = readc;
}

init_read_function()
{
	make_function("READ", Lread);
	make_function("READ-PRESERVING-WHITESPACE",
		      Lread_preserving_whitespace);
	make_function("READ-DELIMITED-LIST", Lread_delimited_list);
	make_function("READ-LINE", Lread_line);
	make_function("READ-CHAR", Lread_char);
	make_function("UNREAD-CHAR", Lunread_char);
	make_function("PEEK-CHAR", Lpeek_char);
	make_function("LISTEN", Llisten);
	make_function("READ-CHAR-NO-HANG", Lread_char_no_hang);
	make_function("CLEAR-INPUT", Lclear_input);

	make_function("PARSE-INTEGER", Lparse_integer);

	make_function("READ-BYTE", Lread_byte);
	make_si_function("READ-BYTES", Lread_bytes);

	make_function("COPY-READTABLE", Lcopy_readtable);
	make_function("READTABLEP", Lreadtablep);
	make_function("SET-SYNTAX-FROM-CHAR", Lset_syntax_from_char);
	make_function("SET-MACRO-CHARACTER", Lset_macro_character);
	make_function("GET-MACRO-CHARACTER", Lget_macro_character);
	make_function("MAKE-DISPATCH-MACRO-CHARACTER",
		      Lmake_dispatch_macro_character);
	make_function("SET-DISPATCH-MACRO-CHARACTER",
		      Lset_dispatch_macro_character);
	make_function("GET-DISPATCH-MACRO-CHARACTER",
		      Lget_dispatch_macro_character);
	make_si_function("SHARP-COMMA-READER-FOR-COMPILER",
			 siLsharp_comma_reader_for_compiler);
	make_si_function("STRING-TO-OBJECT", siLstring_to_object);
	make_si_function("STANDARD-READTABLE", siLstandard_readtable);

#ifdef CLOS
	Sstream_read_char = make_ordinary("STREAM-READ-CHAR");
	Sstream_read_line = make_ordinary("STREAM-READ-LINE");
	Sstream_unread_char = make_ordinary("STREAM-UNREAD-CHAR");
	Sstream_peek_char = make_ordinary("STREAM-PEEK-CHAR");
	Sstream_listen = make_ordinary("STREAM-LISTEN");
	Sstream_clear_input = make_ordinary("STREAM-CLEAR-INPUT");
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * read_VV --
 *     reads the data vector from stream into vector VV
 *
 * Results:
 *	a vector.
 *
 *----------------------------------------------------------------------
 */
object
read_VV(object *VV, int len, object in)
{
	volatile object x; object v;
	int i, nr;
	bool e;
	object old_READtable;
	int old_READdefault_float_format;
	int old_READbase;
	int old_READsuppress;
	int old_sharp_eq_context_max;
	struct sharp_eq_context_struct
		old_sharp_eq_context[SHARP_EQ_CONTEXT_SIZE];
	int old_backq_level;

	old_READtable = READtable;
	old_READdefault_float_format = READdefault_float_format;
	old_READbase = READbase;
	old_READsuppress = READsuppress;
	old_sharp_eq_context_max = sharp_eq_context_max;
	for (i = 0;  i < sharp_eq_context_max;  i++)
		old_sharp_eq_context[i] = sharp_eq_context[i];
	old_backq_level = backq_level;

	setup_standard_READ();

	v = alloc_simple_vector(len, aet_object);
	/* ensure it is cleared, in case the GC is invoked while reading */
	memset((char *)VV, 0, len * sizeof(object));
	v->v.v_self = VV;

	if ((nr = frs_push(FRS_PROTECT, Cnil)) != 0)
		e = TRUE;
	else {

	  while (readc_stream(in) != '#')
	    ;
	  while (readc_stream(in) != '(')
	    ;
	  if (len == 0) {
	    /* still we must read #!(..) */
	    sharp_eq_context_max = 0;
	    backq_level = 0;
	    delimiting_char = code_char(')');
	    preserving_whitespace_flag = FALSE;
	    detect_eos_flag = FALSE;
	    read_object(in);
	  }
	  for (i = 0 ; i < len; i++) {
	    sharp_eq_context_max = 0;
	    backq_level = 0;
	    delimiting_char = code_char(')');
	    preserving_whitespace_flag = FALSE;
	    detect_eos_flag = FALSE;
	    x = read_object(in);
	    if (x == OBJNULL)
	      break;
	    if (sharp_eq_context_max > 0)
	      x = patch_sharp(x);
	    VV[i] = x;
	  }
	  e = FALSE;
	}

	frs_pop();

	READtable = old_READtable;
	READdefault_float_format = old_READdefault_float_format;
	READbase = old_READbase;
	READsuppress = old_READsuppress;
	sharp_eq_context_max = old_sharp_eq_context_max;
	for (i = 0;  i < sharp_eq_context_max;  i++)
		sharp_eq_context[i] = old_sharp_eq_context[i];
	backq_level = old_backq_level;
	if (e) unwind(nlj_fr, nlj_tag, nr);
	return(v);
}
