/*
    predicate.c -- Predicates.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

Lnull(int narg, object x)
{
	check_arg(1);
	VALUES(0) =  (Null(x)) ? Ct : Cnil;
	RETURN(1);
}

Lsymbolp(int narg, object x)
{
	check_arg(1);
	VALUES(0) =  (type_of(x) == t_symbol) ? Ct : Cnil;
	RETURN(1);
}

Latom(int narg, object x)
{
	check_arg(1);
	VALUES(0) =  (type_of(x) != t_cons) ? Ct : Cnil;
	RETURN(1);
}

Lconsp(int narg, object x)
{
	check_arg(1);
	VALUES(0) =  (type_of(x) == t_cons) ? Ct : Cnil;
	RETURN(1);
}

Llistp(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (Null(x) || type_of(x) == t_cons) ? Ct : Cnil;
	RETURN(1);
}

Lnumberp(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = NUMBER_TYPE(t) ? Ct : Cnil;
	RETURN(1);
}

/*	Used in compiled code		*/
bool numberp(object x)
{
  enum type t = type_of(x);
  return(NUMBER_TYPE(t));
}

Lintegerp(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = (t == t_fixnum || t == t_bignum) ? Ct : Cnil;
	RETURN(1);
}

Lrationalp(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = (t == t_fixnum || t == t_bignum || t == t_ratio) ? Ct : Cnil;
	RETURN(1);
}

Lfloatp(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = (t == t_longfloat || t == t_shortfloat) ? Ct : Cnil;
	RETURN(1);
}

Lcomplexp(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_complex) ? Ct : Cnil;
	RETURN(1);
}

Lcharacterp(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_character) ? Ct : Cnil;
	RETURN(1);
}

Lstringp(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_string) ? Ct : Cnil;
	RETURN(1);
}

Lbit_vector_p(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_bitvector) ? Ct : Cnil;
	RETURN(1);
}

Lvectorp(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = (t == t_vector || t == t_string || t == t_bitvector) ? Ct : Cnil;
	RETURN(1);
}

Lsimple_string_p(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_string &&
		     !x->st.st_adjustable &&
		     !x->st.st_hasfillp &&
		     Null(CAR(x->st.st_displaced))) ? Ct : Cnil;
	RETURN(1);
}

Lsimple_bit_vector_p(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_bitvector &&
		     !x->bv.bv_adjustable &&
		     !x->bv.bv_hasfillp &&
		     Null(CAR(x->bv.bv_displaced))) ? Ct : Cnil;
	RETURN(1);
}

Lsimple_vector_p(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = (t == t_vector &&
		     !x->v.v_adjustable &&
		     !x->v.v_hasfillp &&
		     Null(CAR(x->v.v_displaced)) &&
		     (enum aelttype)x->v.v_elttype == aet_object) ? Ct : Cnil;
	RETURN(1);
}

Larrayp(int narg, object x)
{
	enum type t;
	check_arg(1);
	t = type_of(x);
	VALUES(0) = ARRAY_TYPE(t) ? Ct : Cnil;
	RETURN(1);
}

Lpackagep(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_package) ? Ct : Cnil;
	RETURN(1);
}

Lfunctionp(int narg, object x)
{
	enum type t;

	check_arg(1);
	t = type_of(x);
	if (t == t_cfun || t == t_cclosure)
		VALUES(0) = Ct;
	else if (t == t_symbol) {
		VALUES(0) = (x->s.s_gfdef != OBJNULL &&
			     x->s.s_mflag == FALSE) ? Ct : Cnil;
	} else if (t == t_cons) {
		x = CAR(x);
		VALUES(0) = (x == Slambda || x == Slambda_block ||
			     x == Slambda_closure ||
			     x == Slambda_block_closure) ? Ct : Cnil;
	} else
		VALUES(0) = Cnil;
	RETURN(1);
}

Lcompiled_function_p(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_cfun ||
		     type_of(x) == t_cclosure) ? Ct : Cnil;
	RETURN(1);
}

Lcommonp(int narg, object x)
{
	check_arg(1);

	VALUES(0) = (FALSE /* type_of(x) == t_spice */
#ifdef THREADS
		     || type_of(x) == t_thread
		     || type_of(x) == t_cont
#endif THREADS
#ifdef CLOS
		     || type_of(x) == t_instance
		     || type_of(x) == t_gfun
#endif CLOS
		     ) ? Cnil : Ct;
	RETURN(1);
}

Leq(int narg, object x, object y)
{
	check_arg(2);
	VALUES(0) = (x == y) ? Ct : Cnil;
	RETURN(1);
}

bool
eql(object x, object y)
{
	enum type t;

	if (x == y)
		return(TRUE);
	if ((t = type_of(x)) != type_of(y))
		return(FALSE);
	switch (t) {

	case t_fixnum:
		if (fix(x) == fix(y))
			return(TRUE);
		else
			return(FALSE);

	case t_bignum:
		if (big_compare((struct bignum *)x,
				(struct bignum *)y) == 0)
			return(TRUE);
		else
			return(FALSE);

	case t_ratio:
		if (eql(x->rat.rat_num, y->rat.rat_num) &&
		    eql(x->rat.rat_den, y->rat.rat_den))
			return(TRUE);
		else
			return(FALSE);

	case t_shortfloat:
		if (sf(x) == sf(y))
			return(TRUE);
		else
			return(FALSE);

	case t_longfloat:
		if (lf(x) == lf(y))
			return(TRUE);
		else
			return(FALSE);

	case t_complex:
		if (eql(x->cmp.cmp_real, y->cmp.cmp_real) &&
		    eql(x->cmp.cmp_imag, y->cmp.cmp_imag))
			return(TRUE);
		else
			return(FALSE);

	case t_character:
		if (char_code(x) == char_code(y) &&
		    char_bits(x) == char_bits(y) &&
		    char_font(x) == char_font(y))
			return(TRUE);
		else
			return(FALSE);
	}
	return(FALSE);
}

Leql(int narg, object x, object y)
{
	check_arg(2);
	VALUES(0) = (eql(x, y)) ? Ct : Cnil;
	RETURN(1);
}

bool
equal(register object x, object y)
{
	register enum type t;

	cs_check(y);
BEGIN:
	if ((t = type_of(x)) != type_of(y))
		return(FALSE);
	if (x==y)
		return(TRUE);
	switch (t) {

	case t_cons:
		if (!equal(CAR(x), CAR(y)))
			return(FALSE);
		x = CDR(x);
		y = CDR(y);
		goto BEGIN;

	case t_symbol: return FALSE;
	case t_vector: case t_array: return FALSE;

	case t_fixnum:
		return(fix(x)==fix(y));
	case t_shortfloat:
		return(x->SF.SFVAL==y->SF.SFVAL);
	case t_longfloat:
		return(x->LF.LFVAL==y->LF.LFVAL);

	case t_string:
		return(string_eq(x, y));

	case t_bitvector:
	{
		int i, ox, oy;

		if (x->bv.bv_fillp != y->bv.bv_fillp)
			return(FALSE);
		ox = x->bv.bv_offset;
		oy = y->bv.bv_offset;
		for (i = 0;  i < x->bv.bv_fillp;  i++)
			if((x->bv.bv_self[(i+ox)/8] & (0200>>(i+ox)%8))
			 !=(y->bv.bv_self[(i+oy)/8] & (0200>>(i+oy)%8)))
				return(FALSE);
		return(TRUE);
	}

#ifdef CLOS
	case t_instance:
	{
		int i;

		if (x->in.in_class != y->in.in_class)
			return(FALSE);
		for (i = 0;  i < x->in.in_length;  i++)
			if (!equal(x->in.in_slots[i], y->in.in_slots[i]))
				return(FALSE);
		return(TRUE);
	}
#else
	case t_structure:
	{
		int i;

		if (x->str.str_name != y->str.str_name)
			return(FALSE);
		for (i = 0;  i < x->str.str_length;  i++)
			if (!equal(x->str.str_self[i], y->str.str_self[i]))
				return(FALSE);
		return(TRUE);
	}
#endif CLOS

	case t_pathname:
#ifdef unix
		if (equal(x->pn.pn_host, y->pn.pn_host) &&
		    equal(x->pn.pn_device, y->pn.pn_device) &&
		    equal(x->pn.pn_directory, y->pn.pn_directory) &&
		    equal(x->pn.pn_name, y->pn.pn_name) &&
		    equal(x->pn.pn_type, y->pn.pn_type) &&
		    equal(x->pn.pn_version, y->pn.pn_version))
#endif
			return(TRUE);
		else
			return(FALSE);

	}
	return(eql(x,y));
}

Lequal(int narg, object x, object y)
{
	check_arg(2);
	VALUES(0) = (equal(x, y)) ? Ct : Cnil;
	RETURN(1);
}

bool
equalp(object x, object y)
{
	enum type tx, ty;

	cs_check(x);

BEGIN:
	if (eql(x, y))
		return(TRUE);
	tx = type_of(x);
	ty = type_of(y);

	switch (tx) {
	case t_fixnum:
	case t_bignum:
	case t_ratio:
	case t_shortfloat:
	case t_longfloat:
	case t_complex:
		if (ty == t_fixnum || ty == t_bignum || ty == t_ratio ||
		    ty == t_shortfloat || ty == t_longfloat ||
		    ty == t_complex)
			return(!number_compare(x, y));
		else
			return(FALSE);

	case t_vector:
	case t_string:
	case t_bitvector:
		if (ty == t_vector || ty == t_string || ty == t_bitvector)
			goto ARRAY;
		else
			return(FALSE);

	case t_array:
		if (ty == t_array && x->a.a_rank == y->a.a_rank)
		  { if (x->a.a_rank > 1)
		      {int i = 0;
		       for (i = 0; i < x->a.a_rank; i++)
			 if (x->a.a_dims[i] != y->a.a_dims[i]) return(FALSE);
		     }
		    goto ARRAY;}
		else
			return(FALSE);
	}
	if (tx != ty)
		return(FALSE);
	switch (tx) {
	case t_character:
		return(char_equal(x, y));

	case t_cons:
		if (!equalp(CAR(x), CAR(y)))
			return(FALSE);
		x = CDR(x);
		y = CDR(y);
		goto BEGIN;

#ifdef CLOS
	case t_instance:
	{
		int i;

		if (x->in.in_class != y->in.in_class)
			return(FALSE);
		for (i = 0;  i < x->in.in_length;  i++)
			if (!equal(x->in.in_slots[i], y->in.in_slots[i]))
				return(FALSE);
		return(TRUE);
	}
#else
	case t_structure:
	{
		int i;

		if (x->str.str_name != y->str.str_name)
			return(FALSE);
		for (i = 0;  i < x->str.str_length;  i++)
			if (!equalp(x->str.str_self[i], y->str.str_self[i]))
				return(FALSE);
		return(TRUE);
	}
#endif CLOS

	case t_pathname:
		return(equal(x, y));
	}
	return(FALSE);

ARRAY:

	{
		int i, j;

		if (x->a.a_dim != y->a.a_dim)
			return(FALSE);
		j = x->a.a_dim;
		if (tx != t_array)
		  /* So these are both t_vector, t_string, or t_bitvector
		     and may have fill-pointers so limit J must be decreased */
		  {if (x->v.v_hasfillp && (j > x->v.v_fillp))
		     j = x->v.v_fillp;
		   if (y->v.v_hasfillp && (j > y->v.v_fillp))
		     j = y->v.v_hasfillp;}
		for (i = 0, j = x->a.a_dim;  i < j;  i++) {
			if (!equalp(aref(x, i), aref(y, i)))
				return(FALSE);
		}
		return(TRUE);
	}
}

Lequalp(int narg, object x, object y)
{
	check_arg(2);
	VALUES(0) = (equalp(x, y)) ? Ct : Cnil;
	RETURN(1);
}

Fand(object args)
{
	VALUES(0) = Ct;
	for (; !endp(args); args = CDR(args)) {
		eval(CAR(args));
		if (Null(VALUES(0))) RETURN(1);
	}
	RETURN(1);
}

For(object args)
{
	VALUES(0) = Cnil;
	for (; !endp(args) ; args = CDR(args)) {
		eval(CAR(args));
		if (!Null(VALUES(0))) RETURN(1);
	}
	RETURN(1);
}

/*
	Contains_sharp_comma returns TRUE, iff the argument contains
	a cons whose car is si:|#,| or a STRUCTURE.
	Refer to the compiler about this magic.
*/
bool
contains_sharp_comma(object x)
{
	enum type tx;

	cs_check(x);

BEGIN:
	tx = type_of(x);
	if (tx == t_complex)
		return(contains_sharp_comma(x->cmp.cmp_real) ||
		       contains_sharp_comma(x->cmp.cmp_imag));
	if (tx == t_vector)
	{
		int i;

		for (i = 0;  i < x->v.v_fillp;  i++)
			if (contains_sharp_comma(x->v.v_self[i]))
				return(TRUE);
		return(FALSE);
	}
	if (tx == t_cons) {
		if (CAR(x) == siSsharp_comma)
			return(TRUE);
		if (contains_sharp_comma(CAR(x)))
			return(TRUE);
		x = CDR(x);
		goto BEGIN;
	}
	if (tx == t_array)
	{
		int i, j;

		for (i = 0, j = 1;  i < x->a.a_rank;  i++)
			j *= x->a.a_dims[i];
		for (i = 0;  i < j;  i++)
			if (contains_sharp_comma(x->a.a_self[i]))
				return(TRUE);
		return(FALSE);
	}
#ifdef CLOS
	if (tx == t_instance)
		return(TRUE);		/*  Oh, my god!  */
#else
	if (tx == t_structure)
		return(TRUE);		/*  Oh, my god!  */
#endif CLOS
	return(FALSE);
}

siLcontains_sharp_comma(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (contains_sharp_comma(x)) ? Ct : Cnil;
	RETURN(1);
}
/*
siLspicep(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (type_of(x) == t_spice) ? Ct : Cnil;
	RETURN(1);
}
*/
siLfixnump(int narg, object x)
{
	check_arg(1);
	VALUES(0) = (FIXNUMP(x)) ? Ct : Cnil;
	RETURN(1);
}

init_predicate_function()
{
	make_function("NULL", Lnull);
	make_function("SYMBOLP", Lsymbolp);
	make_function("ATOM", Latom);
	make_function("CONSP", Lconsp);
	make_function("LISTP", Llistp);
	make_function("NUMBERP", Lnumberp);
	make_function("INTEGERP", Lintegerp);
	make_function("RATIONALP", Lrationalp);
	make_function("FLOATP", Lfloatp);
	make_function("COMPLEXP", Lcomplexp);
	make_function("CHARACTERP", Lcharacterp);
	make_function("STRINGP", Lstringp);
	make_function("BIT-VECTOR-P", Lbit_vector_p);
	make_function("VECTORP", Lvectorp);
	make_function("SIMPLE-STRING-P", Lsimple_string_p);
	make_function("SIMPLE-BIT-VECTOR-P", Lsimple_bit_vector_p);
	make_function("SIMPLE-VECTOR-P", Lsimple_vector_p);
	make_function("ARRAYP", Larrayp);
	make_function("PACKAGEP", Lpackagep);
	make_function("FUNCTIONP", Lfunctionp);
	make_function("COMPILED-FUNCTION-P", Lcompiled_function_p);
	make_function("COMMONP", Lcommonp);

	make_function("EQ", Leq);
	make_function("EQL", Leql);
	make_function("EQUAL", Lequal);
	make_function("EQUALP", Lequalp);

	make_function("NOT", Lnull);
	make_special_form("AND",Fand);
	make_special_form("OR",For);

	make_si_function("CONTAINS-SHARP-COMMA", siLcontains_sharp_comma);

	make_si_function("FIXNUMP", siLfixnump);
/*	make_si_function("SPICEP", siLspicep); */
}
