/*
    dpp.c -- Defun preprocessor.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


/*
	Usage:
		dpp file

	The file named file.d is preprocessed and the output will be
	written to the file whose name is file.c.


	The function definition:

	@(defun name ({var}*
		      [&optional {var | (var [initform [svar]])}*]
		      [&rest var]
		      [&key {var |
			     ({var | (keyword var)} [initform [svar]])}*
			    [&allow_other_keys]]
		      [&aux {var | (var [initform])}*])

		C-declaration

	@

		C-body

	@)

	name can be either an identifier or a full C procedure header
	enclosed in quotes (').

	&optional may be abbreviated as &o.
	&rest may be abbreviated as &r.
	&key may be abbreviated as &k.
	&allow_other_keys may be abbreviated as &aok.
	&aux may be abbreviated as &a.

	Each variable becomes a C variable.

	Each supplied-p parameter becomes a boolean C variable.

	Initforms are C expressions.
	It an expression contain non-alphanumeric characters,
	it should be surrounded by backquotes (`).


	Function return:

		@(return {form}*)

*/

#include <stdio.h>
#include <ctype.h>

#define	POOLSIZE	2048
#define	MAXREQ		16
#define	MAXOPT		16
#define	MAXKEY		16
#define	MAXAUX		16
#define	MAXRES		16

#define	TRUE		1
#define	FALSE		0

typedef int bool;

FILE *in, *out;

char filename[BUFSIZ];
int lineno;
int tab;
int tab_save;

char pool[POOLSIZE];
char *poolp;

char *function;

char *required[MAXREQ];
int nreq;

struct optional {
	char *o_var;
	char *o_init;
	char *o_svar;
} optional[MAXOPT];
int nopt;

bool rest_flag;
char *rest_var;

bool key_flag;
struct keyword {
	char *k_key;
	char *k_var;
	char *k_init;
	char *k_svar;
} keyword[MAXKEY];
int nkey;
bool allow_other_keys_flag;

struct aux {
	char *a_var;
	char *a_init;
} aux[MAXAUX];
int naux;

char *result[MAXRES];
int nres;

error(char *s)
{
	printf("Error in line %d: %s.\n", lineno, s);
	exit(0);
}

readc()
{
	int c;

	c = getc(in);
	if (feof(in)) {
		if (function != NULL)
			error("unexpected end of file");
		exit(0);
	}
	if (c == '\n') {
		lineno++;
		tab = 0;
	} else if (c == '\t')
		tab++;
	return(c);
}

nextc()
{
	int c;

	while (isspace(c = readc()))
		;
	return(c);
}

unreadc(int c)
{
	if (c == '\n')
		--lineno;
	else if (c == '\t')
		--tab;
	ungetc(c, in);
}

put_tabs(int n)
{
	int i;

	for (i = 0;  i < n;  i++)
		putc('\t', out);
}

pushc(int c)
{
	if (poolp >= &pool[POOLSIZE])
		error("buffer pool overflow");
	*poolp++ = c;
}

char *
read_token()
{
	int c;
	char *p;

	p = poolp;
	if ((c = nextc()) == '`') {
		while ((c = readc()) != '`')
			pushc(c);
	} else {
	  do
		pushc(c);
	  while (isalnum(c = readc()) || c == '_');
	  unreadc(c);
	}
	pushc('\0');
	return(p);
}

reset()
{
	int i;

	poolp = pool;
	function = NULL;
	nreq = 0;
	for (i = 0;  i < MAXREQ;  i++)
		required[i] = NULL;
	nopt = 0;
	for (i = 0;  i < MAXOPT;  i++)
		optional[i].o_var
		= optional[i].o_init
		= optional[i].o_svar
		= NULL;
	rest_flag = FALSE;
	rest_var = "ARGS";
	key_flag = FALSE;
	nkey = 0;
	for (i = 0;  i < MAXKEY;  i++)
		keyword[i].k_key
		= keyword[i].k_var
		= keyword[i].k_init
		= keyword[i].k_svar
		= NULL;
	allow_other_keys_flag = FALSE;
	naux = 0;
	for (i = 0;  i < MAXAUX;  i++)
		aux[i].a_var
		= aux[i].a_init
		= NULL;
}

get_function()
{
	function = read_token();
}

get_lambda_list()
{
	int c;
	char *p;

	if ((c = nextc()) != '(')
		error("( expected");
	for (;;) {
		if ((c = nextc()) == ')')
			return;
		if (c == '&') {
			p = read_token();
			goto OPTIONAL;
		}
		unreadc(c);
		p = read_token();
		if (nreq >= MAXREQ)
			error("too many required variables");
		required[nreq++] = p;
	}

OPTIONAL:
	if (strcmp(p, "optional") != 0 && strcmp(p, "o") != 0)
		goto REST;
	for (;;  nopt++) {
		if ((c = nextc()) == ')')
			return;
		if (c == '&') {
			p = read_token();
			goto REST;
		}
		if (nopt >= MAXOPT)
			error("too many optional argument");
		if (c == '(') {
			optional[nopt].o_var = read_token();
			if ((c = nextc()) == ')')
				continue;
			unreadc(c);
			optional[nopt].o_init = read_token();
			if ((c = nextc()) == ')')
				continue;
			unreadc(c);
			optional[nopt].o_svar = read_token();
			if (nextc() != ')')
				error(") expected");
		} else {
			unreadc(c);
			optional[nopt].o_var = read_token();
		}
	}

REST:
	if (strcmp(p, "rest") != 0 && strcmp(p, "r") != 0)
		goto KEYWORD;
	rest_flag = TRUE;
	if ((c = nextc()) == ')' || c == '&')
		error("&rest var missing");
	unreadc(c);
	rest_var = read_token();
	if ((c = nextc()) == ')')
		return;
	if (c != '&')
		error("& expected");
	p = read_token();
	goto KEYWORD;

KEYWORD:
	if (strcmp(p, "key") != 0 && strcmp(p, "k") != 0)
		goto AUX;
	key_flag = TRUE;
	for (;;  nkey++) {
		if ((c = nextc()) == ')')
			return;
		if (c == '&') {
			p = read_token();
			if (strcmp(p, "allow_other_keys") == 0 ||
			    strcmp(p, "aok") == 0) {
				allow_other_keys_flag = TRUE;
				if ((c = nextc()) == ')')
					return;
				if (c != '&')
					error("& expected");
				p = read_token();
			}
			goto AUX;
		}
		if (nkey >= MAXKEY)
			error("too many optional argument");
		if (c == '(') {
			if ((c = nextc()) == '(') {
				p = read_token();
				if (p[0] != ':' || p[1] == '\0')
					error("keyword expected");
				keyword[nkey].k_key = p + 1;
				keyword[nkey].k_var = read_token();
				if (nextc() != ')')
					error(") expected");
			} else {
				unreadc(c);
				keyword[nkey].k_key
				= keyword[nkey].k_var
				= read_token();
			}
			if ((c = nextc()) == ')')
				continue;
			unreadc(c);
			keyword[nkey].k_init = read_token();
			if ((c = nextc()) == ')')
				continue;
			unreadc(c);
			keyword[nkey].k_svar = read_token();
			if (nextc() != ')')
				error(") expected");
		} else {
			unreadc(c);
			keyword[nkey].k_key
			= keyword[nkey].k_var
			= read_token();
		}
	}

AUX:
	if (strcmp(p, "aux") != 0 && strcmp(p, "a") != 0)
		error("illegal lambda-list keyword");
	for (;;) {
		if ((c = nextc()) == ')')
			return;
		if (c == '&')
			error("illegal lambda-list keyword");
		if (naux >= MAXAUX)
			error("too many auxiliary variable");
		if (c == '(') {
			aux[naux].a_var = read_token();
			if ((c = nextc()) == ')')
				continue;
			unreadc(c);
			aux[naux].a_init = read_token();
			if (nextc() != ')')
				error(") expected");
		} else {
			unreadc(c);
			aux[naux].a_var = read_token();
		}
		naux++;
	}
}

get_return()
{
	int c;

	nres = 0;
	for (;;) {
		if ((c = nextc()) == ')')
			return;
		unreadc(c);
		result[nres++] = read_token();
	}
}

put_fhead()
{
  bool b = FALSE; char *p = function;
  int i;
  fputc('L', out);
  while (!b && *p != '\0') {
    fputc(*p, out);
    b = (*p++ == '(');
  }
  if (b) {
  	/*
	@(defun `assoc_or_rassoc(object (*car_or_cdr)())`
	     (item a_list &key test test_not key)
	must become:
	Lassoc_or_rassoc(int narg, object (*car_or_cdr)(),
	      object item, object a_list, ...)
	*/
	fprintf(out, "int narg, ");
	while (*p != ')' || p[1] != '\0')
		fputc(*p++, out);
	}
  else
    fprintf(out, "(int narg", function);
	
  for (i = 0; i < nreq; i++)
		fprintf(out, ", object %s", required[i]);
  if (nopt > 0 || rest_flag || key_flag)
  		fprintf(out, ", ...");
  fprintf(out, ")\n");
  if (b) {
	while (*p++ != ')') ;
  	fprintf(out, "%s", p);	/* declaration of extra first arg */
	}
  fprintf(out, "{");
}

put_declaration()
{
  int i;

  for (i = 0;  i < nopt;  i++)
    fprintf(out, "\tobject %s;\n", optional[i].o_var);
  for (i = 0;  i < nopt;  i++)
    if (optional[i].o_svar != NULL)
      fprintf(out, "\tbool %s;\n", optional[i].o_svar);
  if (nkey > 0)
    fprintf(out, "\tobject KEYS[%d];\n", nkey);
  for (i = 0;  i < nkey;  i++)
      fprintf(out, "#define %s KEY_VARS[%d]\n", keyword[i].k_var, i);
  for (i = 0;  i < nkey;  i++)
    if (keyword[i].k_svar != NULL) {
      fprintf(out, "#define %s (bool)KEY_VARS[%d]\n", keyword[i].k_svar, nkey+i);
    }
  for (i = 0;  i < naux;  i++)
    fprintf(out, "\tobject %s;\n", aux[i].a_var);
  if (nopt == 0 && !rest_flag && !key_flag)
    fprintf(out, "\tcheck_arg(%d);\n", nreq);
  else {
    if (key_flag)
      fprintf(out, "\tobject KEY_VARS[%d];\n", 2*nkey);
    fprintf(out, "\tva_list %s;\n\tva_start(%s, %s);\n", rest_var, rest_var,
	    ((nreq > 0) ? required[nreq-1] : "narg"));
    fprintf(out, "\tif (narg < %d) FEtoo_few_arguments(&narg);\n", nreq);
    if (nopt > 0 && !rest_flag && !key_flag)
      fprintf(out, "\tif (narg > %d) FEtoo_many_arguments(&narg);\n", nreq + nopt);
    for (i = 0;  i < nopt;  i++) {
      fprintf(out, "\tif (narg > %d) {\n", nreq+i, optional[i].o_var);
      fprintf(out, "\t\t%s = va_arg(%s, object);\n",
	      optional[i].o_var, rest_var);
      if (optional[i].o_svar)
	fprintf(out, "\t\t%s = TRUE;\n", optional[i].o_svar);
      fprintf(out, "\t} else {\n");
      fprintf(out, "\t\t%s = %s;\n",
	      optional[i].o_var,
	      optional[i].o_init == NULL ? "Cnil" : optional[i].o_init);
      if (optional[i].o_svar)
	fprintf(out, "\t\t%s = FALSE;\n", optional[i].o_svar);
      fprintf(out, "\t}\n");
    }
    if (key_flag) {
      for (i = 0; i < nkey; i++)
	fprintf(out, "\tKEYS[%d]=K%s;\n", i, keyword[i].k_key);
      fprintf(out, "\tparse_key(narg-%d, ARGS, %d, KEYS, KEY_VARS, %s, %d);\n",
	      nreq+nopt, nkey, rest_flag ? rest_var : "OBJNULL", allow_other_keys_flag);
      for (i = 0;  i < nkey;  i++) {
	if (keyword[i].k_init == NULL)
	  continue;
	fprintf(out, "\tif (KEY_VARS[%d]==Cnil) %s = %s;\n",
		nkey+i, keyword[i].k_var, keyword[i].k_init);
	if (keyword[i].k_svar != NULL)
	  fprintf(out, "\t%s = (object)%s != Cnil;\n",
		  keyword[i].k_svar, keyword[i].k_svar);
      }
    }
  }
  for (i = 0;  i < naux;  i++)
    fprintf(out, "\t%s = %s;\n", aux[i].a_var,
	    aux[i].a_init == NULL ? "Cnil" : aux[i].a_init);
}

put_ftail()
{
	int i;

/*	for (i = 0;  i < nopt;  i++)
		fprintf(out, "#undef %s\n", optional[i].o_var);
	for (i = 0;  i < naux;  i++)
		fprintf(out, "#undef %s\n", aux[i].a_var);
 */
	for (i = 0;  i < nkey;  i++) {
		fprintf(out, "#undef %s\n", keyword[i].k_var);
		if (keyword[i].k_svar != NULL)
			fprintf(out, "#undef %s\n", keyword[i].k_svar);
	}
	fprintf(out, "}");
}

put_return()
{
	int i, t;

	t = tab_save+1;
	if (nres == 0)
		fprintf(out, "VALUES(0) = Cnil;\n");
	else {
	  fprintf(out, "{\n");
	  for (i = 0;  i < nres;  i++) {
		put_tabs(t);
		fprintf(out, "VALUES(%d) = %s;\n", i, result[i]);
		}
	  put_tabs(t);
	  fprintf(out, "RETURN(%d);\n", nres);
	  put_tabs(tab_save);
	  fprintf(out, "}\n");
	}
	put_tabs(tab_save);
}

main_loop()
{
	int c;
	char *p;

	lineno = 1;

LOOP:
	reset();
#ifdef unix
	fprintf(out, "\n#line %d \"%s\"\n", lineno, filename);
#endif
	while ((c = readc()) != '@')
		putc(c, out);
	if (readc() != '(')
		error("@( expected");
	p = read_token();
	if (strcmp(p, "defun") == 0) {
		get_function();
		get_lambda_list();
		put_fhead();
#ifdef unix
		fprintf(out, "\n#line %d \"%s\"\n", lineno, filename);
#endif
		while ((c = readc()) != '@')
			putc(c, out);
		put_declaration();

	BODY:
#ifdef unix
		fprintf(out, "\n#line %d \"%s\"\n", lineno, filename);
#endif
		while ((c = readc()) != '@')
			putc(c, out);
		if ((c = readc()) == ')') {
			put_ftail();
			goto LOOP;
		} else if (c != '(')
			error("@( expected");
		p = read_token();
		if (strcmp(p, "return") == 0) {
			tab_save = tab;
			get_return();
			put_return();
			goto BODY;
		} else
			error("illegal symbol");
	} else
		error("illegal symbol");
}

main(int argc, char **argv)
{
	char *p, *q;

	if (argc != 2)
		error("arg count");
	for (p = argv[1], q = filename;  *p != '\0';  p++, q++)
		if (q >= &filename[BUFSIZ-3])
			error("too long file name");
		else
			*q = *p;
	q[0] = '.';
	q[1] = 'd';
	q[2] = '\0';
	in = fopen(filename, "r");
	if (in == NULL)
		error("can't open input file");
	q[1] = 'c';
	out = fopen(filename, "w");
	if (out == NULL)
		error("can't open output file");
	q[1] = 'd';
	printf("dpp: %s -> ", filename);
	q[1] = 'c';
	printf("%s\n", filename);
	q[1] = 'd';
	main_loop();
}
