/*
    unixtime.c -- Time routines.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

#include <sys/times.h>

#ifndef HZ			/* usually from <sys/param.h> */
#define HZ 60
#endif

#ifdef sgi
#undef BSD
#define SYSV
#endif sgi

#ifdef BSD
#include <sys/timeb.h>
static struct timeb beginning;
#endif BSD

#ifdef SYSV
long beginning;
#endif SYSV

runtime()
/*
   tms_utime is the CPU time used while executing instructions in the
   user space of the calling process, measured in 1/HZ seconds.
*/
{
	struct tms buf;

	times(&buf);
	return(buf.tms_utime);
}

object
unix_time_to_universal_time(int i)
{
	object x;

	x = number_times(MAKE_FIXNUM(70*365+17), MAKE_FIXNUM(24*60*60));
	x = number_plus(bignum1(i), x);
	return(x);
}

Lget_universal_time(int narg)
{
	check_arg(0);
	VALUES(0) = unix_time_to_universal_time(time(0));
	RETURN(1);
}

Lsleep(int narg, object z)
{
	check_arg(1);
	check_type_or_rational_float(&z);
	if (number_minusp(z))
		FEerror("~S is not a non-negative number.", 1, z);
	Lround(1, z);
	z = VALUES(0);
	if (FIXNUMP(z))
		sleep(fix(z));
	else
		for(;;)
			sleep(1000);
	VALUES(0) = Cnil;
	RETURN(1);
}

Lget_internal_run_time(int narg)
{
	struct tms buf;

	check_arg(0);
	times(&buf);
	VALUES(0) = MAKE_FIXNUM(buf.tms_utime);
	RETURN(1);
}

Lget_internal_real_time(int narg)
{
#ifdef BSD
	static struct timeval begin_tzp;
	struct timeval tzp;

	check_arg(0);
	if (begin_tzp.tv_sec==0)
	  gettimeofday(&begin_tzp,0);
	gettimeofday(&tzp,0);
/* the value returned will be relative to the first time this is called,
   plus the fraction of a second.  We must make it relative, so this
   will only wrap if the process lasts longer than 818 days
   */
	VALUES(0) = MAKE_FIXNUM((tzp.tv_sec-begin_tzp.tv_sec)*HZ
				+ ((tzp.tv_usec)*HZ)/1000000);
#endif BSD

#ifdef SYSV
	VALUES(0) = MAKE_FIXNUM((time(0) - beginning)*HZ);
#endif SYSV
	RETURN(1);
}

Lget_local_time_zone(int narg)
{
#ifdef BSD
	struct timeb tb;
	check_arg(0);
	ftime(&tb);
	VALUES(0) = make_longfloat((longfloat)tb.timezone/60);
#else
	VALUES(0) = MAKE_FIXNUM(TIME_ZONE);
#endif BSD
	RETURN(1);
}

Ldaylight_saving_timep(int narg)
{
#ifdef BSD
	struct timeb tb;
	check_arg(0);
	ftime(&tb);
	VALUES(0) = tb.dstflag ? Ct : Cnil;
#else
	VALUES(0) = Cnil;
#endif BSD
	RETURN(1);
}

init_unixtime()
{
#ifdef BSD
	ftime(&beginning);
#endif BSD
#ifdef SYSV
	beginning = time(0);
#endif SYSV

	make_si_special("*DEFAULT-TIME-ZONE*", MAKE_FIXNUM(TIME_ZONE));
	make_constant("INTERNAL-TIME-UNITS-PER-SECOND", MAKE_FIXNUM(HZ));

	make_function("GET-UNIVERSAL-TIME", Lget_universal_time);
	make_function("SLEEP", Lsleep);
	make_function("GET-INTERNAL-RUN-TIME", Lget_internal_run_time);
	make_function("GET-INTERNAL-REAL-TIME", Lget_internal_real_time);
	make_si_function("GET-LOCAL-TIME-ZONE", Lget_local_time_zone);
	make_si_function("DAYLIGHT-SAVING-TIME-P", Ldaylight_saving_timep);
}
