/*
    load.d -- Binary loader (contains also open_fasl_data).
*/
/*
    Copyright (c) 1990, Giuseppe Attardi and William F. Schelter.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

#ifdef __mips
#include <sys/cachectl.h>
#endif __mips

/******************************* EXPORTS ******************************/

object FASL_string;

/******************************* IMPORTS ******************************/

extern object Kerror;
extern object Kif_does_not_exist;
extern object Knewest;
extern object Kwild;
extern object Vdefault_pathname_defaults;
extern object Vpackage;
extern object Vstandard_output;
extern object readc();

/******************************* ------- ******************************/

object Kverbose;
object Vload_verbose;
#ifdef PDE
object siVsource_pathname;
#endif PDE      

object LSP_string;

object open_fasl_data(object);

@(defun load (pathname
	      &key (verbose `symbol_value(Vload_verbose)`)
		    print
		    (if_does_not_exist Kerror)
	      &aux pntype fasl_filename lsp_filename filename
		   defaults strm stdoutput x
		   package)
	volatile bds_ptr old_bds_top;
	int nr;
	object strm1;
	object (*old_read_ch_fun)() = read_ch_fun;
@
	check_type_or_pathname_string_symbol_stream(&pathname);
	pathname = coerce_to_pathname(pathname);
	defaults = symbol_value(Vdefault_pathname_defaults);
	defaults = coerce_to_pathname(defaults);
	pathname = merge_pathnames(pathname, defaults, Knewest);
	pntype = pathname->pn.pn_type;
	if (Null(pntype) || pntype == Kwild) {
	  pathname->pn.pn_type = FASL_string;
	}
	if (type_of(pathname->pn.pn_type) == t_string &&
#ifdef unix
	    string_eq(pathname->pn.pn_type, FASL_string) &&
#endif
	    file_exists(fasl_filename = coerce_to_namestring(pathname))) {

	  /* binary file */
	  struct codeblock Cblock;
	  object temp_cfun, stream;

	  if (verbose != Cnil) {
	    setupPRINT(fasl_filename, symbol_value(Vstandard_output));
	    if (file_column(PRINTstream) != 0)
	      write_str("\n");
	    write_str(";;; Loading ");
	    PRINTescape = FALSE;
	    write_object(fasl_filename, 0);
	    write_str("\n");
	    cleanupPRINT();
	    flush_stream(PRINTstream);
	  }
	  package = symbol_value(Vpackage);
	  old_bds_top = bds_top;
	  bds_bind(Vpackage, package);

	  /* To protect the block from GC, until the init_code is executed,
	     we create a cfun */
	  Cblock.cd_start = heap_end;
	  Cblock.cd_size = 0;
	  Cblock.cd_data = OBJNULL;
	  temp_cfun = alloc_object(t_cfun);
	  temp_cfun->cf.cf_name = OBJNULL;
	  temp_cfun->cf.cf_block = &Cblock;

	  /* load and link the code */
	  dld(fasl_filename->st.st_self, &Cblock);

	  /* open stream to read data vector */
	  stream = open_fasl_data(fasl_filename);

	  if (nr = frs_push(FRS_PROTECT, Cnil)) {
	    close_stream(stream, TRUE);
	    frs_pop();
	    unwind(nlj_fr, nlj_tag, nr);
	  } else {

	    /* call the init_code function */
#ifdef __mips
	    cacheflush(Cblock.cd_start, Cblock.cd_size, BCACHE);
#endif __mips
	    (*(int (*)())(Cblock.cd_start))(Cblock.cd_size, stream);
	    close_stream(stream, TRUE);
	    frs_pop();
	  }

	  if (print != Cnil) {
	    setupPRINT(Cnil, symbol_value(Vstandard_output));
	    if (file_column(PRINTstream) != 0)
	      write_str("\n");
	    write_str(";;; Fasload successfully ended.\n");
	    cleanupPRINT();
	    flush_stream(PRINTstream);
	  }
	  bds_unwind(old_bds_top);
	  if (verbose != Cnil) {
	    setupPRINT(fasl_filename, symbol_value(Vstandard_output));
	    if (file_column(PRINTstream) != 0)
	      write_str("\n");
	    write_str(";;; Finished loading ");
	    PRINTescape = FALSE;
	    write_object(fasl_filename, 0);
	    write_str("\n");
	    cleanupPRINT();
	    flush_stream(PRINTstream);
	  }
	  @(return `pathname`)
	  }

	/* not a binary file */
	if (Null(pntype) || pntype == Kwild) {
	  pathname->pn.pn_type = LSP_string;
	  filename = coerce_to_namestring(pathname);
	}
	if (!file_exists(filename)) {
	  /* use the supplied pathname */
	  pathname->pn.pn_type = pntype;
	  filename = coerce_to_namestring(pathname);
	}
	if (if_does_not_exist != Cnil)
		if_does_not_exist = Kerror;
	strm1 = strm
	= open_stream(filename, smm_input, Cnil, if_does_not_exist);
	if (Null(strm))
		@(return Cnil)
	if (verbose != Cnil) {
		setupPRINT(filename, symbol_value(Vstandard_output));
		if (file_column(PRINTstream) != 0)
			write_str("\n");
		write_str(";;; Loading ");
		PRINTescape = FALSE;
		write_object(filename, 0);
		write_str("\n");
		cleanupPRINT();
		flush_stream(PRINTstream);
	}
	package = symbol_value(Vpackage);
	old_bds_top = bds_top;
	bds_bind(Vpackage, package);
	bds_bind(Vstandard_input, strm);
#ifdef PDE
	bds_bind(siVsource_pathname, pathname);
#endif PDE      
	if (nr = frs_push(FRS_PROTECT, Cnil)) {
		close_stream(strm1, TRUE);
		frs_pop();
		bds_unwind(old_bds_top); /* necessary even if unwind does it */
		unwind(nlj_fr, nlj_tag, nr);
	}
	for (;;) {
		preserving_whitespace_flag = FALSE;
		detect_eos_flag = TRUE;
		read_ch_fun = readc; /* setup for read. Beppe */
		x = read_object_non_recursive(strm);
		read_ch_fun = old_read_ch_fun;
		if (x == OBJNULL)
			break;
		{
			object *lex_old = lex_env; lex_dcl;

			lex_new();
			eval(x);
			lex_env = lex_old;
		}
		if (print != Cnil) {
			setupPRINT(x, symbol_value(Vstandard_output));
			write_object(x, 0);
			write_str("\n");
			cleanupPRINT();
			flush_stream(PRINTstream);
		}
	}
	close_stream(strm, TRUE);
	frs_pop();
	bds_unwind(old_bds_top);
	if (verbose != Cnil) {
		setupPRINT(filename, symbol_value(Vstandard_output));
		if (file_column(PRINTstream) != 0)
			write_str("\n");
		write_str(";;; Finished loading ");
		PRINTescape = FALSE;
		write_object(filename, 0);
		write_str("\n");
		cleanupPRINT();
		flush_stream(PRINTstream);
	}
	@(return `pathname`)
@)

/* ----------------------------------------------------------------------
 * Binary file loader utilities
 * ----------------------------------------------------------------------
 */

siLbuild_symbol_table(int narg)
{
   char *tmpfile = "ecl_raw.sym";
 /* This is now done by the Makefile:

   static char tmpfile[50]; 
   static char command[200];

   printf("\n;;; Building symbol table for %s ..", ecl_self); fflush(stdout);
   sprintf(tmpfile, "/tmp/rsym%d", getpid());
   coerce_to_filename(symbol_value(siVsystem_directory), system_directory);
#ifdef MSDOS
   sprintf(command, "go32 %srsym %s %s", system_directory, ecl_self, tmpfile);
#else
   sprintf(command, "%srsym %s %s", system_directory, ecl_self, tmpfile);
#endif MSDOS
   {int status = system(command);
    if (WEXITSTATUS(status))
      FEerror("The rsym command ~a failed.", 1, make_simple_string(command));
  }
  */
   read_special_symbols(tmpfile);
   unlink(tmpfile);
   VALUES(0) = Cnil;
   RETURN(0);
 }

/*
 *----------------------------------------------------------------------
 *
 * open_fasl_data --
 *     opens fasl file and seeks to the end of a binary file
 *      The size of LISP data is stored as an eight-digit decimal string
 *	at the end of the file, followed by newline.
 *
 * Results:
 *	stream positioned at beginning of data
 *
 *----------------------------------------------------------------------
 */
object
open_fasl_data(object fname)
{
	object faslfile;
	FILE *fp;
#define LSPDATASZ	9
	char sizebuf[LSPDATASZ];

	faslfile = open_stream(fname, smm_input, Cnil, Kerror);
	fp = faslfile->sm.sm_fp;
	fseek(fp, -LSPDATASZ, 2);
	fread(sizebuf, LSPDATASZ, 1, fp);
	sizebuf[LSPDATASZ] = '\0';
	fseek(fp, -atoi(sizebuf), 1);
	return(faslfile);
}

/* ---------------------------------------------------------------------- */
#ifdef unix

siLfaslink(int narg, object file, object lib)
{
	bds_ptr old_bds_top;
	int i;
	object package;

	check_arg(2);
	check_type_or_pathname_string_symbol_stream(&file);
	check_type_string(&lib);
	file = coerce_to_pathname(file);
	file->pn.pn_type = FASL_string;
	file = namestring(file);
	package = symbol_value(Vpackage);
	old_bds_top = bds_top;
	bds_bind(Vpackage, package);
	faslink(file, lib);
	bds_unwind(old_bds_top);
	VALUES(0) = Cnil;
	RETURN(1);
}
#endif unix

init_load()
{
  Kprint = make_keyword("PRINT");
  Kverbose = make_keyword("VERBOSE");
  Vload_verbose = make_special("*LOAD-VERBOSE*", Ct);
#ifdef PDE
  siVsource_pathname = make_si_special("*SOURCE-PATHNAME*", Cnil);
#endif PDE      

#ifdef unix
  FASL_string = make_simple_string("o");
  LSP_string = make_simple_string("lsp");
  enter_mark_origin(&FASL_string);
  enter_mark_origin(&LSP_string);
#endif unix

  make_function("LOAD", Lload);
  make_si_function("BUILD-SYMBOL-TABLE", siLbuild_symbol_table);
#ifdef unix
  make_si_function("FASLINK", siLfaslink);
#endif unix
}
