/*
    hash.d  -- Hash tables.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/

#include "config.h"

object Seq;
object Seql;
object Sequal;

object Ktest;
object Ksize;
object Krehash_size;
object Krehash_threshold;


unsigned int
hash_eql(object x)
{
	unsigned int h;

	switch (type_of(x)) {
	case t_fixnum:
		return(fix(x));

	case t_bignum:
		h = x->big.big_car;
		while (x->big.big_cdr != NULL) {
			x = (object)x->big.big_cdr;
			h += x->big.big_car;
		}
		return(h);

	case t_ratio:
   		return(hash_eql(x->rat.rat_num) + hash_eql(x->rat.rat_den));

	case t_shortfloat:
		return((int)(sf(x)));

	case t_longfloat:
		return((int)(lf(x)) + *((int *)x + 1));

	case t_complex:
		return(hash_eql(x->cmp.cmp_real) + hash_eql(x->cmp.cmp_imag));

	case t_character:
		return(char_code(x));

	default:
		return((unsigned int)x / 4);
	}
}

int
hash_equal(object x, int depth)
{
	unsigned int h = 0; int i;
	char *s;

	cs_check(x);

BEGIN:
	if (depth++ > 3) return h;
	switch (type_of(x)) {
	case t_cons:
		h += hash_equal(CAR(x), depth);
		x = CDR(x);
		goto BEGIN;

	case t_string:
		for (i = x->st.st_fillp, s = x->st.st_self;  i > 0;  --i, s++)
			h += (*s & 0377)*12345 + 1;
		return(h);

	case t_symbol:
		/* case t_string could share this code-- */
		{int len = x->st.st_fillp;
		 s = x->st.st_self;
		 switch (len) {
		 case 0: break;
		 default:
		 case 4: h+= s[--len] << 24;
		 case 3: h+= s[--len] << 16;
		 case 2: h+= s[1] << 8;
		 case 1: h+= s[0];
		 }
		 return(h);
	       }
        case t_package:
	case t_bitvector:
		return(h);

	case t_pathname:
		h += hash_equal(x->pn.pn_host, depth+1);
		h += hash_equal(x->pn.pn_device, depth+1);
		h += hash_equal(x->pn.pn_directory, depth+1);
		h += hash_equal(x->pn.pn_name, depth+1);
		h += hash_equal(x->pn.pn_type, depth+1);
		h += hash_equal(x->pn.pn_version, depth+1);
		return(h);
#ifndef ANSI
#ifdef CLOS
	case t_instance:
		h += hash_equal(CLASS_NAME(x), depth);
		for (i = 0;  i < x->in.in_length;  i++)
			h += hash_equal(x->in.in_slots[i], depth);
		return(h);
#else
	case t_structure:
		h += hash_equal(x->str.str_name, depth);
		for (i = 0;  i < x->str.str_length;  i++)
			h += hash_equal(x->str.str_self[i], depth);
		return(h);
#endif CLOS
#endif ANSI

	default:
		return(h + hash_eql(x));
	}
}

struct htent *
gethash(object key, object hashtable)
{
	enum httest htest;
	int hsize;
	struct htent *e;
	object hkey;
	int i, j = -1, k;
	bool b;

	htest = (enum httest)hashtable->ht.ht_test;
	hsize = hashtable->ht.ht_size;
	if (htest == htt_eq)
		i = (int)key / 4;
	else if (htest == htt_eql)
		i = hash_eql(key);
	else if (htest == htt_equal)
		i = hash_equal(key, 0);
	i &= 0x7fffffff;
	for (i %= hsize, k = 0; k < hsize;  i = (i + 1) % hsize, k++) {
		e = &hashtable->ht.ht_self[i];
		hkey = e->hte_key;
		if (hkey == OBJNULL) {
			if (e->hte_value == OBJNULL)
				if (j < 0)
					return(e);
				else
					return(&hashtable->ht.ht_self[j]);
			else
				if (j < 0)
					j = i;
				else if (j == i)
				  /* this was never returning --wfs
				     but looping around with j=0 */
				  return(e);
			continue;
		}
		if (htest == htt_eq)
		    	b = key == hkey;
		else if (htest == htt_eql)
			b = eql(key, hkey);
		else if (htest == htt_equal)
			b = equal(key, hkey);
		if (b)
			return(&hashtable->ht.ht_self[i]);
	}
	return(&hashtable->ht.ht_self[j]);
}

struct htent *
gethash1(object key, object hashtable)
{
	enum httest htest;
	int i, hsize;
	bool b;

	htest = (enum httest)hashtable->ht.ht_test;
	hsize = hashtable->ht.ht_size;
	if (htest == htt_eq)
		i = (int)key / 4;
	else if (htest == htt_eql)
		i = hash_eql(key);
	else if (htest == htt_equal)
		i = hash_equal(key, 0);
	i &= 0x7fffffff;
	for (i %= hsize; ; i = (i + 1) % hsize)
		if (hashtable->ht.ht_self[i].hte_key == OBJNULL)
			return(&hashtable->ht.ht_self[i]);
}

sethash(object key, object hashtable, object value)
{
	int i;
	bool over;
	struct htent *e;
	e = gethash(key, hashtable);
	if (e->hte_key != OBJNULL) {
		e->hte_value = value;
		return;
	}
	i = hashtable->ht.ht_nent + 1;
	if (i >= hashtable->ht.ht_size)
		over = TRUE;
	else if (FIXNUMP(hashtable->ht.ht_rhthresh))
		over = i >= fix(hashtable->ht.ht_rhthresh);
	else if (type_of(hashtable->ht.ht_rhthresh) == t_shortfloat)
		over =
		i >= hashtable->ht.ht_size * sf(hashtable->ht.ht_rhthresh);
	else if (type_of(hashtable->ht.ht_rhthresh) == t_longfloat)
		over =
		i >= hashtable->ht.ht_size * lf(hashtable->ht.ht_rhthresh);
	if (over) {
		extend_hashtable(hashtable);
		e = gethash1(key, hashtable);
	}
	hashtable->ht.ht_nent++;
	e->hte_key = key;
	e->hte_value = value;
}
	
extend_hashtable(object hashtable)
{
	object old, key;
	int old_size, new_size, i;
	struct htent *e;
	old_size = hashtable->ht.ht_size;
	if (FIXNUMP(hashtable->ht.ht_rhsize))
		new_size = old_size + fix(hashtable->ht.ht_rhsize);
	else if (type_of(hashtable->ht.ht_rhsize) == t_shortfloat)
		new_size = old_size * sf(hashtable->ht.ht_rhsize);
	else if (type_of(hashtable->ht.ht_rhsize) == t_longfloat)
		new_size = old_size * lf(hashtable->ht.ht_rhsize);
	if (new_size <= old_size)
		new_size = old_size + 1;
	old = alloc_object(t_hashtable);
	old->ht = hashtable->ht;
	hashtable->ht.ht_self = NULL; /* for GC sake */
	hashtable->ht.ht_size = new_size;
	if (FIXNUMP(hashtable->ht.ht_rhthresh))
		hashtable->ht.ht_rhthresh =
		MAKE_FIXNUM(fix(hashtable->ht.ht_rhthresh) +
			    (new_size - old->ht.ht_size));
	hashtable->ht.ht_self =
	(struct htent *)alloc_relblock(new_size * sizeof(struct htent),
				       sizeof(object));
	for (i = 0;  i < new_size;  i++) {
		hashtable->ht.ht_self[i].hte_key = OBJNULL;
		hashtable->ht.ht_self[i].hte_value = OBJNULL;
	}
	for (i = 0;  i < old_size;  i++)
		if ((key = old->ht.ht_self[i].hte_key) != OBJNULL) {
			e = gethash1(key, hashtable);
			e->hte_key = key;
			e->hte_value = old->ht.ht_self[i].hte_value;
		}
}


@(defun make_hash_table (&key (test Seql)
			      (size `MAKE_FIXNUM(1024)`)
			      (rehash_size
			       `make_shortfloat((shortfloat)1.5)`)
			      (rehash_threshold
			       `make_shortfloat((shortfloat)0.7)`)
			 &aux h)
	enum httest htt;
	int i;
@
	if (test == Seq || test == Seq->s.s_gfdef)
		htt = htt_eq;
	else if (test == Seql || test == Seql->s.s_gfdef)
		htt = htt_eql;
	else if (test == Sequal || test == Sequal->s.s_gfdef)
		htt = htt_equal;
	else
		FEerror("~S is an illegal hash-table test function.",
			1, test);
  	if (!FIXNUMP(size) || 0 < fix(size))
		;
	else
		FEerror("~S is an illegal hash-table size.", 1, size);
	if (FIXNUMP(rehash_size) && 0 < fix(rehash_size) ||
	    type_of(rehash_size) == t_shortfloat && 1.0 < sf(rehash_size) ||
	    type_of(rehash_size) == t_longfloat && 1.0 < lf(rehash_size))
		;
	else
		FEerror("~S is an illegal hash-table rehash-size.",
			1, rehash_size);
	if (FIXNUMP(rehash_threshold) &&
	    0 < fix(rehash_threshold) && fix(rehash_threshold) < fix(size) ||
	    type_of(rehash_threshold) == t_shortfloat &&
	    0.0 < sf(rehash_threshold) && sf(rehash_threshold) < 1.0 ||
	    type_of(rehash_threshold) == t_longfloat &&
	    0.0 < lf(rehash_threshold) && lf(rehash_threshold) < 1.0)
		;
	else
		FEerror("~S is an illegal hash-table rehash-threshold.",
			1, rehash_threshold);
	h = alloc_object(t_hashtable);
	h->ht.ht_test = (short)htt;
	h->ht.ht_size = fix(size);
	h->ht.ht_rhsize = rehash_size;
	h->ht.ht_rhthresh = rehash_threshold;
        h->ht.ht_nent = 0;
	h->ht.ht_self = NULL;	/* for GC sake */
	h->ht.ht_self = (struct htent *)
	  alloc_relblock(fix(size) * sizeof(struct htent), sizeof(object));
	for(i = 0;  i < fix(size);  i++) {
		h->ht.ht_self[i].hte_key = OBJNULL;
		h->ht.ht_self[i].hte_value = OBJNULL;
	}
	@(return h)
@)

@(defun hash_table_p (ht)
@
	@(return `(type_of(ht) == t_hashtable) ? Ct : Cnil`)
@)

Lgethash(int narg, object key, object ht, object nv)
{
	struct htent *e;
	object no_value = Cnil;
	
	if (narg < 2)
		FEtoo_few_arguments(&narg);
	else if (narg == 3)
		no_value = nv;
	else if (narg > 3)
		FEtoo_many_arguments(&narg);
	check_type_hash_table(&ht);
	e = gethash(key, ht);
	if (e->hte_key != OBJNULL) {
		VALUES(0) = e->hte_value;
		VALUES(1) = Ct;
	} else {
		VALUES(0) = no_value;
		VALUES(1) = Cnil;
	}
	RETURN(2);
}

siLhash_set(int narg, object key, object ht, object val)
{
	check_arg(3);

	check_type_hash_table(&ht);
	sethash(key, ht, val);
	VALUES(0) = val;
	RETURN(1);
}
	
@(defun remhash (key ht)
	struct htent *e;
@
	check_type_hash_table(&ht);
	e = gethash(key, ht);
	if (e->hte_key != OBJNULL) {
		e->hte_key = OBJNULL;
		e->hte_value = Cnil;
		ht->ht.ht_nent--;
		@(return Ct)
	} else
		@(return Cnil)
@)

@(defun clrhash (ht)
	int i;
@
	check_type_hash_table(&ht);
	for(i = 0; i < ht->ht.ht_size; i++) {
		ht->ht.ht_self[i].hte_key = OBJNULL;
		ht->ht.ht_self[i].hte_value = OBJNULL;
	}
	ht->ht.ht_nent = 0;
	@(return ht)
@)

@(defun hash_table_count (ht)
@
	check_type_hash_table(&ht);
	@(return `MAKE_FIXNUM(ht->ht.ht_nent)`)
@)


@(defun sxhash (key)
@
	@(return `MAKE_FIXNUM(hash_equal(key, 0) & 0x7fffffff)`)
@)

@(defun maphash (fun ht)
	int i;
@
	check_type_hash_table(&ht);
	for (i = 0;  i < ht->ht.ht_size;  i++) {
		if(ht->ht.ht_self[i].hte_key != OBJNULL)
			funcall(3, fun,
				  ht->ht.ht_self[i].hte_key,
				  ht->ht.ht_self[i].hte_value);
	}
	@(return Cnil)
@)

init_hash()
{
	Seq = make_ordinary("EQ");
	Seql = make_ordinary("EQL");
	Sequal = make_ordinary("EQUAL");
	Ksize = make_keyword("SIZE");
	Ktest = make_keyword("TEST");
	Krehash_size = make_keyword("REHASH-SIZE");
	Krehash_threshold = make_keyword("REHASH-THRESHOLD");
	
	make_function("MAKE-HASH-TABLE", Lmake_hash_table);
	make_function("HASH-TABLE-P", Lhash_table_p);
	make_function("GETHASH", Lgethash);
	make_function("REMHASH", Lremhash);
   	make_function("MAPHASH", Lmaphash);
	make_function("CLRHASH", Lclrhash);
	make_function("HASH-TABLE-COUNT", Lhash_table_count);
   	make_function("SXHASH", Lsxhash);
	make_si_function("HASH-SET", siLhash_set);
}
