/*
    bind.c -- Lambda bindings.
*/
/*
    Copyright (c) 1984, Taiichi Yuasa and Masami Hagiya.
    Copyright (c) 1990, Giuseppe Attardi.

    ECoLisp is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    See file '../Copyright' for full details.
*/


#include "config.h"

/******************************* EXPORTS ******************************/
object ANDoptional;
object ANDrest;
object ANDkey;
object ANDallow_other_keys;
object ANDaux;

object Kallow_other_keys;
/******************************* ------- ******************************/


#define declaration(x)  ((CAR(x)) == Sdeclare)
#define CONSP(x)        (type_of(x) == t_cons)
#define SUPPLIED_VAR(x) (CONSP(temp=x) && CONSP(temp=CDR(temp)) && CONSP(CDR(temp)))
#define KEY_VAR(x)      (CONSP(x) ? (CONSP(CAR(x)) ? CADAR(x) : CAR(x)) : x)

struct required {
	object  req_var;
	object  req_spp;
};

struct optional {
	object  opt_var;
	object  opt_spp;
	object  opt_init;
	object  opt_svar;
	object  opt_svar_spp;
};

struct rest {
	object  rest_var;
	object  rest_spp;
};

struct keyword {
	object  key_word;
	object  key_var;
	object  key_spp;
	object  key_init;
	object  key_svar;
	object  key_svar_spp;
	object  key_val;
	object  key_svar_val;
};

/* In external.h
struct let {
	object  let_var;
	object  let_spp;
	object  let_init;
}; */

object
lambda_bind(int narg, object lambda, object *args)
{
	object lambda_list, body, form, x, vars, ds, vs, v;
	object reqs, opts, keys, auxs;
	int nreq, nopt = 0, nkey = 0, naux = 0;
	int i, j;
	struct rest rest;
	bool rest_flag = FALSE;
	bool key_flag = FALSE;
	bool allow_other_keys_flag = FALSE;
	bool other_keys_appeared;
	bool special_processed;

	bds_check;
	if (type_of(lambda) != t_cons)
		FEerror("No lambda list.", 0);
	lambda_list = CAR(lambda);
	body = CDR(lambda);

	/*  Count variables of each kind  */
	reqs = lambda_list;
	for (nreq = 0 ;; nreq++) {
		if (endp(lambda_list))
			goto REQUIRED_ONLY;
		x = CAR(lambda_list);
		lambda_list = CDR(lambda_list);
		check_symbol(x);
		if (x == ANDallow_other_keys)
			illegal_lambda();
		if (x == ANDoptional)
			goto OPTIONAL;
		if (x == ANDrest)
			goto REST;
		if (x == ANDkey)
			goto KEYWORD;
		if (x == ANDaux)
			goto AUX;
		if ((enum stype)x->s.s_stype == stp_constant)
			FEerror("~S is not a variable.", 1, x);
	}

OPTIONAL:
	opts = lambda_list;
	for ( ;;  nopt++) {
		if (endp(lambda_list))
			goto BINDINGS;
		x = CAR(lambda_list);
		lambda_list = CDR(lambda_list);
		if (type_of(x) == t_symbol) {
			if (x == ANDoptional ||
			    x == ANDallow_other_keys)
				illegal_lambda();
			if (x == ANDrest)
				goto REST;
			if (x == ANDkey)
				goto KEYWORD;
			if (x == ANDaux)
				goto AUX;
		}
	}

REST:
	if (endp(lambda_list))
		illegal_lambda();
	check_symbol(CAR(lambda_list));
	check_var(CAR(lambda_list));
	rest_flag = TRUE;
	rest.rest_var = CAR(lambda_list);
	rest.rest_spp = Cnil;
	lambda_list = CDR(lambda_list);
	if (endp(lambda_list))
		goto BINDINGS;
	x = CAR(lambda_list);
	lambda_list = CDR(lambda_list);
	check_symbol(x);
	if (x == ANDoptional || x == ANDrest || x == ANDallow_other_keys)
		illegal_lambda();
	if (x == ANDkey)
		goto KEYWORD;
	if (x == ANDaux)
		goto AUX;
	illegal_lambda();

KEYWORD:
	keys = lambda_list;
	key_flag = TRUE;
	for ( ;;  nkey++) {
		if (endp(lambda_list))
			goto BINDINGS;
		x = CAR(lambda_list);
		lambda_list = CDR(lambda_list);
		if (type_of(x) == t_symbol) {
			if (x == ANDallow_other_keys) {
				allow_other_keys_flag = TRUE;
				if (endp(lambda_list))
					goto BINDINGS;
				x = CAR(lambda_list);
				lambda_list = CDR(lambda_list);
			}
			if (x == ANDoptional || x == ANDrest ||
			    x == ANDkey || x == ANDallow_other_keys)
				illegal_lambda();
			if (x == ANDaux)
				goto AUX;
		}
	}

AUX:
	auxs = lambda_list;
	for ( ;;  naux++) {
		if (endp(lambda_list))
			goto BINDINGS;
		x = CAR(lambda_list);
		lambda_list = CDR(lambda_list);
		if (type_of(x) == t_symbol) {
			if (x == ANDoptional || x == ANDrest ||
			    x == ANDkey || x == ANDallow_other_keys ||
			    x == ANDaux)
				illegal_lambda();
		}
	}

      BINDINGS:
	{
	  struct required required[nreq]; /*  __GNUC__  */
	  struct optional optional[nopt];
	  struct keyword keyword[nkey];
	  struct let aux[naux];

	  for (i = 0, vars = reqs; i < nreq; i++, vars = CDR(vars)) {
	    x = CAR(vars);
	    required[i].req_var = x;
	    required[i].req_spp = Cnil;
	  }
	  for (i = 0, vars = opts; i < nopt; i++, vars = CDR(vars)) {
	    x = CAR(vars);
	    if (CONSP(x)) {
	      check_symbol(CAR(x));
	      check_var(CAR(x));
	      optional[i].opt_var = CAR(x);
	      x = CDR(x);
	      optional[i].opt_spp = Cnil;
	      if (endp(x)) {
		optional[i].opt_init = Cnil;
		optional[i].opt_svar = Cnil;
		optional[i].opt_svar_spp = Cnil;
		continue;
	      }
	      optional[i].opt_init = CAR(x);
	      x = CDR(x);
	      if (endp(x)) {
		optional[i].opt_svar = Cnil;
		optional[i].opt_svar_spp = Cnil;
		continue;
	      }
	      check_symbol(CAR(x));
	      check_var(CAR(x));
	      optional[i].opt_svar = CAR(x);
	      optional[i].opt_svar_spp = Cnil;
	      if (!endp(CDR(x)))
		illegal_lambda();
	    } else {
	      check_var(x);
	      optional[i].opt_var = x;
	      optional[i].opt_spp = Cnil;
	      optional[i].opt_init = Cnil;
	      optional[i].opt_svar = Cnil;
	      optional[i].opt_svar_spp = Cnil;
	    }
	  }
	  for (i = 0, vars = keys; i < nkey; i++, vars = CDR(vars)) {
	    x = CAR(vars);
	    keyword[i].key_spp = Cnil;
	    keyword[i].key_init = Cnil;
	    keyword[i].key_svar = Cnil;
	    keyword[i].key_svar_spp = Cnil;
	    keyword[i].key_val = Cnil;
	    keyword[i].key_svar_val = Cnil;
	    if (CONSP(x)) {
	      if (CONSP(CAR(x))) {
		if (!keywordp(CAAR(x)))
		  FEerror("~S is not a keyword.", 1, CAAR(x));
		keyword[i].key_word = CAAR(x);
		if (endp(CDAR(x)))
		  illegal_lambda();
		check_symbol(CADAR(x));
		keyword[i].key_var = CADAR(x);
		if (!endp(CDDAR(x)))
		  illegal_lambda();
	      } else {
		check_symbol(CAR(x));
		check_var(CAR(x));
		keyword[i].key_word = intern(CAR(x)->st.st_self, keyword_package);
		keyword[i].key_var = CAR(x);
	      }
	      x = CDR(x);
	      if (endp(x)) continue;
	      keyword[i].key_init = CAR(x);
	      x = CDR(x);
	      if (endp(x)) continue;
	      check_symbol(CAR(x));
	      check_var(CAR(x));
	      keyword[i].key_svar = CAR(x);
	      if (!endp(CDR(x)))
		illegal_lambda();
	    } else {
	      check_var(x);
	      keyword[i].key_word = intern(x->st.st_self, keyword_package);
	      keyword[i].key_var = x;
	    }
	  }
	  for (i = 0, vars = auxs; i < naux; i++, vars = CDR(vars)) {
	    x = CAR(vars);
	    aux[i].let_spp = Cnil;
	    aux[i].let_init = Cnil;
	    if (CONSP(x)) {
	      check_symbol(CAR(x));
	      check_var(CAR(x));
	      aux[i].let_var = CAR(x);
	      x = CDR(x);
	      if (endp(x)) continue;
	      aux[i].let_init = CAR(x);
	      if (!endp(CDR(x)))
		illegal_lambda();
	    } else {
	      check_var(x);
	      aux[i].let_var = x;
	    }
	  }
	SEARCH_DECLARE:
	  for (;  !endp(body);  body = CDR(body)) {
	    form = CAR(body);
	    /*  MACRO EXPANSION  */
	    form = macro_expand(form);

	    if (type_of(form) == t_string) {
	      if (endp(CDR(body)))
		break;
	      continue;
	    }
	    if (type_of(form) != t_cons || !declaration(form))
	      break;
	    for (ds = CDR(form); !endp(ds); ds = CDR(ds)) {
	      if (type_of(CAR(ds)) != t_cons)
		illegal_declare(form);
	      if (CAAR(ds) == Sspecial) {
		for (vs = CDAR(ds);  !endp(vs);  vs = CDR(vs)) {
		  v = CAR(vs);
		  check_symbol(v);
		  special_processed = FALSE;
		  for (i = 0;  i < nreq;  i++)
		    if (required[i].req_var == v) {
		      required[i].req_spp = Ct;
		      special_processed = TRUE;
		    }
		  for (i = 0;  i < nopt;  i++)
		    if (optional[i].opt_var == v) {
		      optional[i].opt_spp = Ct;
		      special_processed = TRUE;
		    } else if (optional[i].opt_svar == v) {
		      optional[i].opt_svar_spp = Ct;
		      special_processed = TRUE;
		    }
		  if (rest_flag && rest.rest_var == v) {
		    rest.rest_spp = Ct;
		    special_processed = TRUE;
		  }
		  for (i = 0;  i < nkey;  i++)
		    if (keyword[i].key_var == v) {
		      keyword[i].key_spp = Ct;
		      special_processed = TRUE;
		    } else if (keyword[i].key_svar == v) {
		      keyword[i].key_svar_spp = Ct;
		      special_processed = TRUE;
		    }
		  for (i = 0;  i < naux;  i++)
		    if (aux[i].let_var == v) {
		      aux[i].let_spp = Ct;
		      special_processed = TRUE;
		    }
		  if (special_processed)
		    continue;
		  /*  lex_special_bind(v);  */
		  lex_env[0] = CONS(CONS(v, Cnil), lex_env[0]);
		}
	      }
	    }
	  }

	  if (narg < nreq) {
	    if (nopt == 0 && !rest_flag && !key_flag)
	      check_arg_failed(narg, nreq);
	    FEtoo_few_arguments(&narg);
	  }
	  if (!rest_flag && !key_flag && narg > nreq+nopt) {
	    if (nopt == 0)
	      check_arg_failed(narg, nreq);
	    FEtoo_many_arguments(&narg);
	  }

	  for (i = 0;  i < nreq;  i++)
	    bind_var(required[i].req_var, args[i], required[i].req_spp);
	  for (i = 0;  i < nopt;  i++)
	    if (nreq+i < narg) {
	      bind_var(optional[i].opt_var,
		       args[nreq+i],
		       optional[i].opt_spp);
	      if (optional[i].opt_svar != Cnil)
		bind_var(optional[i].opt_svar,
			 Ct,
			 optional[i].opt_svar_spp);
	    } else {
	      eval(optional[i].opt_init);
	      bind_var(optional[i].opt_var,
		       VALUES(0),
		       optional[i].opt_spp);
	      if (!Null(optional[i].opt_svar))
		bind_var(optional[i].opt_svar,
			 Cnil,
			 optional[i].opt_svar_spp);
	    }
	  if (rest_flag) {
	    object l = Cnil;
	    for (i = narg, j = nreq+nopt;  --i >= j;  )
	      l = CONS(args[i], l);
	    bind_var(rest.rest_var, l, rest.rest_spp);
	  }
	  if (key_flag) {
	    i = narg - nreq - nopt;
	    if (i >= 0 && i%2 != 0)
	      FEerror("Keyword values are missing.", 0);
	    other_keys_appeared = FALSE;
	    for (i = nreq + nopt;  i < narg;  i += 2) {
	      if (!keywordp(args[i]))
		FEerror("~S is not a keyword.", 1, args[i]);
	      if (args[i] == Kallow_other_keys &&
		  args[i+1] != Cnil)
		allow_other_keys_flag = TRUE;
	      for (j = 0;  j < nkey;  j++) {
		if (keyword[j].key_word == args[i]) {
		  if (keyword[j].key_svar_val
		      != Cnil)
		    goto NEXT_ARG;
		  keyword[j].key_val
		    = args[i+1];
		  keyword[j].key_svar_val
		    = Ct;
		  goto NEXT_ARG;
		}
	      }
	      other_keys_appeared = TRUE;

	    NEXT_ARG:
	      continue;
	    }
	    if (other_keys_appeared && !allow_other_keys_flag)
	      FEerror("Other-keys are not allowed.", 0);
	  }
	  for (i = 0;  i < nkey;  i++)
	    if (keyword[i].key_svar_val != Cnil) {
	      bind_var(keyword[i].key_var,
		       keyword[i].key_val,
		       keyword[i].key_spp);
	      if (keyword[i].key_svar != Cnil)
		bind_var(keyword[i].key_svar,
			 keyword[i].key_svar_val,
			 keyword[i].key_svar_spp);
	    } else {
	      eval(keyword[i].key_init);
	      bind_var(keyword[i].key_var,
		       VALUES(0),
		       keyword[i].key_spp);
	      if (keyword[i].key_svar != Cnil)
		bind_var(keyword[i].key_svar,
			 keyword[i].key_svar_val,
			 keyword[i].key_svar_spp);
	    }
	  for (i = 0;  i < naux;  i++) {
	    eval(aux[i].let_init);
	    bind_var(aux[i].let_var, VALUES(0), aux[i].let_spp);
	  }

	  if (body != Cnil && CAR(body) != form)
	    body = CONS(form, CDR(body));
	  return(body);
	}

REQUIRED_ONLY:
	{ struct required required[nreq]; /*  __GNUC__ */

	  for (i = 0, vars = reqs; i < nreq; i++, vars = CDR(vars)) {
	    x = CAR(vars);
	    required[i].req_var = x;
	    required[i].req_spp = Cnil;
	  }
	  body = process_decl(body, required, &required[nreq],
			      sizeof (struct required));

	  if (narg != nreq)
	    check_arg_failed(narg, nreq);
	  for (i = 0;  i < nreq;  i++)
	    bind_var(required[i].req_var, args[i], required[i].req_spp);
	  return(body);
	}
}

bind_var(object var, object val, object special)
{
	switch ((enum stype)var->s.s_stype) {
	case stp_constant:
		FEerror("Cannot bind the constant ~S.", 1, var);

	case stp_special:
		bds_bind(var, val);
		break;

	default:
		if (special != Cnil) {
		  lex_env[0] = CONS(CONS(var, Cnil), lex_env[0]);
		  bds_bind(var, val);
		} else
		  lex_env[0] = CONS(CONS(var, CONS(val, Cnil)), lex_env[0]);
	}
}

illegal_lambda()
{
	FEerror("Illegal lambda expression.", 0);
}

object let_bind(object body, struct let *start, struct let *end, int size)
{
	struct let *let;

	/*  process_decl fills let_spp  */
	body = process_decl(body, start, end, size);
	for (let = start; let < end; (int)let += size) {
	  eval(let->let_init);
	  let->let_init = VALUES(0);
	}
	for (let = start; let < end; (int)let += size)
	  bind_var(let->let_var, let->let_init, let->let_spp);
	return(body);
}

object letA_bind(object body, struct let *start, struct let *end, int size)
{
	volatile struct let *let;

	body = process_decl(body, start, end, size);
	for (let = start; let < end; (int)let += size) {
	  eval(let->let_init);
	  let->let_init = VALUES(0);
	  bind_var(let->let_var, let->let_init, let->let_spp);
	}
	return(body);
}

/*
  Handles special declarations, removes declarations from body
 */
object
process_decl(object body, struct let *start, struct let *end, int size)
{
	object form, ds, v, vs;
	struct let *let;
	bool not_found;

	for (;  !endp(body);  body = CDR(body)) {
		form = CAR(body);

		/*  MACRO EXPANSION  */
		form = macro_expand(form);

		if (type_of(form) == t_string) {
			if (endp(CDR(body)))
				break;
			else continue;
		      }
		if (type_of(form)!=t_cons || !declaration(form))
			break;
		for (ds = CDR(form); !endp(ds); ds = CDR(ds)) {
		  if (type_of(CAR(ds)) != t_cons)
		    illegal_declare(form);
		  if (CAAR(ds) == Sspecial) {
		    for ( vs = CDAR(ds);  !endp(vs);  vs = CDR(vs)) {
		      v = CAR(vs);
		      check_symbol(v);
		      not_found = TRUE;
		      for (let = start; not_found && let < end; (int)let += size)
			if (let->let_var == v) {
			  let->let_spp = Ct;
			  not_found = FALSE;
			}
		      /* declaring special a non-local variable */
		      if (not_found)
			lex_env[0] = CONS(CONS(v, Cnil), lex_env[0]);
		    }
		  }
		}
	      }

	if (body != Cnil && CAR(body) != form)
		body = CONS(form, CDR(body));
	return(body);
}

#define NOT_YET         10
#define FOUND           11
#define NOT_KEYWORD     1

parse_key(
     int narg,                  /* number of actual args */
     object *args,              /* actual args */
     int nkey,                  /* number of keywords */
     object *keys,              /* keywords for the function */
     object *vars,              /* where to put values (vars[0..nkey-1])
				   and suppliedp (vars[nkey..2*nkey-1]) */
     object rest,               /* rest variable or NULL */
     bool allow_other_keys)     /* whether other key are allowed */
{ object *p;
  int i;
  object k;

  /* fill in the rest arg list */
  if (rest != OBJNULL)
    for (i = narg, p = args; i > 0; i--) {
      CAR(rest) = *p++;
      rest = CDR(rest);
    }

  for (i = 0; i < 2*nkey; i++)
    vars[i] = Cnil;             /* default values: NIL, supplied: NIL */
  if (narg <= 0) return;

  /* scan backwards, so that if a keyword is duplicated, first one is used */
  args = args + narg;
  top:
    while (narg >= 2) {
     args = args - 2;
     k = args[0];
     for (i = 0; i < nkey; i++) {
       if (keys[i] == k) {
	 vars[i] = args[1];
	 vars[nkey+i] = Ct;
	 narg = narg-2;
	 goto top;
       }
     }
     /* the key is a new one */
     if (allow_other_keys)
	 narg = narg-2;
     else {
       /* look for :allow-other-keys t */
       for ( i = narg-2, p = args; i >= 0; i -= 2, p -=2)
	 if (*p == Kallow_other_keys) {
	   allow_other_keys = (p[1] != Cnil); break;
	 }
       if (allow_other_keys) narg = narg-2;
       else FEerror("Unrecognized key ~a", 1, k);
     }
   }
  if (narg != 0) FEerror("Odd number of keys", 0);
}

/* Used in compiled macros */
check_other_key(object l, int n, ...)
{
	object other_key = OBJNULL;
	object k;
	int i;
	bool allow_other_keys = FALSE;
	va_list ap;
	va_start(ap, n);                /* extracting arguments */

	for (;  !endp(l);  l = CDDR(l)) {
		k = CAR(l);
		if (!keywordp(k))
			FEerror("~S is not a keyword.", 1, k);
		if (endp(CDR(l)))
			FEerror("Odd number of arguments for keywords.", 0);
		if (k == Kallow_other_keys && CADR(l) != Cnil) {
			allow_other_keys = TRUE;
		} else {
#ifndef NO_ARG_ARRAY
		        object *ktab = (object *)ap;
			for (i = 0;  i < n;  i++)
				if (ktab[i] == k) {
				  ktab[i] = Cnil; /* remember seen */
				  break;
				}
			if (i >= n) other_key = k;
#else
			Rewrite this!
#endif NO_ARG_ARRAY
		}
	}
	if (other_key != OBJNULL && !allow_other_keys)
		FEerror("The keyword ~S is not allowed or is duplicated.",
			1, other_key);
}

init_bind()
{
	ANDoptional = make_ordinary("&OPTIONAL");
	enter_mark_origin(&ANDoptional);
	ANDrest = make_ordinary("&REST");
	enter_mark_origin(&ANDrest);
	ANDkey = make_ordinary("&KEY");
	enter_mark_origin(&ANDkey);
	ANDallow_other_keys = make_ordinary("&ALLOW-OTHER-KEYS");
	enter_mark_origin(&ANDallow_other_keys);
	ANDaux = make_ordinary("&AUX");
	enter_mark_origin(&ANDaux);

	make_constant("LAMBDA-LIST-KEYWORDS",
	list(8, ANDoptional, ANDrest, ANDkey, ANDallow_other_keys, ANDaux,
		make_ordinary("&WHOLE"), make_ordinary("&ENVIRONMENT"), make_ordinary("&BODY")));

	make_constant("LAMBDA-PARAMETERS-LIMIT", MAKE_FIXNUM(64));

	Kallow_other_keys = make_keyword("ALLOW-OTHER-KEYS");

}
