;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Funktion : Funktionen zur Optimierungen, die auf den Typinformationen
;;;            aufbauen.
;;;            Optimierung eines einzelnen Ausdrucks oder rekursiv eines
;;;            Ausdrucks und aller seiner Komponenten.
;;;
;;; $Revision: 1.22 $
;;; $Log: tomain.lisp,v $
;;; Revision 1.22  1993/07/23  09:40:08  hk
;;; Neues Macro eliminating-msg fuer Meldung, dass Ausdrucke eliminiert
;;; werden. Benutzt *optimize-verbosity*.
;;;
;;; Revision 1.21  1993/06/30  13:46:41  jh
;;; An optimain.lisp angepasst.
;;;
;;; Revision 1.20  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.19  1993/06/10  09:54:18  kl
;;; Bindung der Variable *current-fun* korrigiert.
;;;
;;; Revision 1.18  1993/06/08  13:13:37  jh
;;; Variable *no-to* zum Ausschalten der Optimierungen eingebaut.
;;;
;;; Revision 1.17  1993/05/27  12:49:40  jh
;;; Optimierung fuer bottom-getypte Form einer Applikation eingebaut.
;;;
;;; Revision 1.16  1993/05/25  15:12:52  jh
;;; bottom-Optimierung fuer mv-lambda eingebaut.
;;;
;;; Revision 1.15  1993/05/19  14:40:12  jh
;;; Einige bottom-Optimierungen eingebaut.
;;;
;;; Revision 1.14  1993/05/13  13:19:26  jh
;;; Durch Optimierung entstandene forms werden jetzt ebenfalls optimiert.
;;;
;;; Revision 1.13  1993/05/13  10:15:23  jh
;;; In switch-forms wird jetzt auch otherwise optimiert.
;;;
;;; Revision 1.12  1993/05/12  08:47:43  kl
;;; Schreibfehler korrigiert.
;;;
;;; Revision 1.11  1993/05/09  16:26:42  kl
;;; Ausgabe erweitert.
;;;
;;; Revision 1.10  1993/05/09  13:05:40  kl
;;; Fehler in to-1form (if-form) behoben.
;;;
;;; Revision 1.9  1993/05/07  10:06:26  hk
;;; Schreibfehler in (to-parts switch-form) behoben.
;;;
;;; Revision 1.8  1993/05/06  13:31:06  kl
;;; Verwendung von has-no-side-effect eingebaut und Ausgabe erweitert.
;;;
;;; Revision 1.7  1993/05/05  11:04:22  jh
;;; Spezialbehandlung fuer bottom-t bei if-Optimierung eingebaut.
;;;
;;; Revision 1.6  1993/04/30  14:32:02  hk
;;; do-to deaktiviert, da Fehler in Typinferenz oder Typoptimierungen.
;;;
;;; Revision 1.5  1993/04/30  13:26:32  hk
;;; Hier wurde bisher JEDES (mv-lambda args body) zu args optimiert.
;;;
;;; Revision 1.4  1993/04/21  11:35:30  jh
;;; mv-lambda eingebaut.
;;;
;;; Revision 1.3  1993/02/16  16:09:13  hk
;;; Revision Keyword eingefuegt.
;;;
;;; Revision 1.2  1993/01/07  12:59:31  jh
;;; Statistik fuer die if-Optimierungen eingebaut.
;;;-----------------------------------------------------------------------------

(in-package "CLICC")

;;------------------------------------------------------------------------------
;; to-1form optimiert einen Zwischensprachausdruck, ohne seine Teilausdruecke
;; zu optimieren.
;;------------------------------------------------------------------------------

;;------------------------------------------------------------------------------
;; Ausgabe der Meldung, dass Ausdrucke eliminiert werden
;;------------------------------------------------------------------------------
(defmacro eliminating-msg (&rest args)
  `(when (> *optimize-verbosity* 1)
    (clicc-message ,@args)))

;;------------------------------------------------------------------------------
;; An var-refs, named-consts etc. kann nichts optimiert werden.
;;------------------------------------------------------------------------------

(defmethod to-1form ((a-form form))
  a-form)

;;------------------------------------------------------------------------------

(defun is-bottom-typed (a-form)
  (is-bottom-t (?type a-form)))

(defmethod to-1form ((a-setq-form setq-form))
  (if (is-bottom-typed (?form a-setq-form))
      (progn
        (eliminating-msg "(in ~A) Eliminating bottom-typed setq." *current-fun*)
        (let ((location (?location a-setq-form)))
          (when (var-ref-p location)
            (decf (?write (?var location)))))
        (?form a-setq-form))
      a-setq-form))

;;------------------------------------------------------------------------------

(defmethod to-1form ((a-progn-form progn-form))
  (let* ((form-list (?form-list a-progn-form))
         (bottom-form-pos (position-if #'is-bottom-typed form-list)))
    (setf (?form-list a-progn-form)
          (if (and bottom-form-pos (< (1+ bottom-form-pos) (length form-list)))
              (progn
                (eliminating-msg
                 "(in ~A) Eliminating unreachable code in a progn-form"
                 *current-fun*)
                (let ((new-form-list (subseq form-list 0 (1+ bottom-form-pos))))
                  (setf (?type a-progn-form)
                        (?type (first (last new-form-list))))
                  new-form-list))
              form-list)))
  a-progn-form)

;;------------------------------------------------------------------------------

(defmethod to-1form ((an-if-form if-form))
  (let* ((pred (?pred an-if-form))
         (pred-type (?type pred))
         (then (?then an-if-form))
         (else (?else an-if-form))
         eliminated-case
         (result 
          (cond ((is-bottom-t pred-type)
                 (inc-stats 'then-optis)
                 (inc-stats 'else-optis)
                 (setq eliminated-case "then- and else")
                 pred)
                ((zs-subtypep pred-type null-t)
                 (inc-stats 'then-optis)
                 (setq eliminated-case "then")
                 (if (has-no-side-effect pred)
                     else
                     (optimize-1form
                      (make-instance 'progn-form 
                                     :form-list (list pred else)
                                     :type (?type else)))))
                ((not (zs-subtypep null-t pred-type))
                 (inc-stats 'else-optis)
                 (setq eliminated-case "else")
                 (if (has-no-side-effect pred)
                     then
                     (optimize-1form
                      (make-instance 'progn-form
                                     :form-list (list pred then)
                                     :type (?type then)))))
                (T an-if-form))))
    (when eliminated-case
      (eliminating-msg "(in ~A) Eliminating unreachable ~A-case."
                       *current-fun*
                       eliminated-case))
    result))

;;------------------------------------------------------------------------------

(defmethod to-1form ((an-app app))
  (if (or (some #'is-bottom-typed (?arg-list an-app))
          (is-bottom-typed (?form an-app)))
      (progn
        (eliminating-msg "(in ~A) Eliminating unreachable app-form"
                         *current-fun*)
        (optimize-1form
         (make-instance 'progn-form
                        :form-list (append (?arg-list an-app)
                                           (list (?form an-app)))
                        :type (?type (first (last (?arg-list an-app)))))))
      an-app))

;;------------------------------------------------------------------------------

(defmethod to-1form ((a-let*-form let*-form))
  (let* ((init-list (?init-list a-let*-form))
         (bottom-form-pos (position-if #'is-bottom-typed init-list)))
    (when bottom-form-pos
      (eliminating-msg "(in ~A) Eliminating unreachable code in a let*-form"
                       *current-fun*)
      (setf (?init-list a-let*-form) (subseq init-list 0 bottom-form-pos)
            (?var-list a-let*-form) (subseq (?var-list a-let*-form)
                                            0 bottom-form-pos)
            (?body a-let*-form) (nth bottom-form-pos init-list)
            (?type a-let*-form) (?type (?body a-let*-form)))))
  a-let*-form)

;;------------------------------------------------------------------------------

(defmethod to1-form ((a-mv-lambda mv-lambda))
  (if (is-bottom-typed (?arg a-mv-lambda))
      (progn
        (eliminating-msg "(in ~A) Eliminating unreachable mv-lambda"
                         *current-fun*)
        (?arg a-mv-lambda))
      a-mv-lambda))

;;------------------------------------------------------------------------------
(provide "tomain")
