;;;-----------------------------------------------------------------------------
;;; Copyright (C) 1993 Christian-Albrechts-Universitaet zu Kiel, Germany
;;;-----------------------------------------------------------------------------
;;; Projekt  : APPLY - A Practicable And Portable Lisp Implementation
;;;            ------------------------------------------------------
;;; Funktion : Methoden zum Traversieren der Zwischensprache, die die 
;;;            Zwischensprachausdruecke analysieren, Typen inferieren 
;;;            und die entsprechenden Ausdruecke mit Typinformationen 
;;;            versehen.
;;;
;;; $Revision: 1.29 $
;;; $Log: tipass2.lisp,v $
;;; Revision 1.29  1993/06/24  14:35:06  kl
;;; Verhalten bei destruktiven Seiteneffekten verbessert.
;;;
;;; Revision 1.28  1993/06/17  08:00:09  hk
;;; Copright Notiz eingefuegt
;;;
;;; Revision 1.27  1993/06/16  07:51:14  kl
;;; Bei Applikationen wird nun die Komponente data-effects beachtet.
;;;
;;; Revision 1.26  1993/06/10  10:32:53  kl
;;; Neuen ti-level eingefuehrt. Binden von Funktionsparameter vereinheitlicht.
;;;
;;; Revision 1.25  1993/06/07  10:09:52  kl
;;; Binden der aktuellen Parameter vereinheitlicht.
;;;
;;; Revision 1.24  1993/06/05  21:46:27  hk
;;; Bei mv-lambda suppliedp Parameter an bool-t gebunden.
;;;
;;; Revision 1.23  1993/05/27  13:41:01  kl
;;; Weil cont keine LZS-form ist, wurde eine besondere around-Methode
;;; der gen. Funktion analyse-type fuer Continuations noetig.
;;;
;;; Revision 1.22  1993/05/27  13:17:33  kl
;;; Typisierung der Continuations auf tomain.lisp abgestimmt.
;;;
;;; Revision 1.21  1993/05/23  15:59:45  kl
;;; Anpassungen an den neuen Typverband.
;;;
;;; Revision 1.20  1993/05/21  12:27:01  kl
;;; Fehler im Umgang mit Typumgebungen behoben.
;;;
;;; Revision 1.19  1993/05/18  16:16:04  kl
;;; Umstellung auf die neue Implementierung des Typverbands. Fehler in der
;;; Behandlung von tagbody-Konstrukten behoben. Analyse-Methode fuer
;;; foreign-funs eingefuehrt.
;;;
;;; Revision 1.18  1993/05/15  14:07:52  kl
;;; Behandlung der Initialisierungsausdruecke umgestellt.
;;;
;;; Revision 1.17  1993/05/09  16:57:27  kl
;;; Argumenttypen der importierten Funktionen koennen jetzt genutzt werden.
;;;
;;; Revision 1.16  1993/04/30  09:23:34  kl
;;; Hauptfunktion nach timain.lisp verlagert.
;;;
;;; Revision 1.15  1993/04/20  15:06:57  kl
;;; Es werden jetzt auch die Klassendefinitionen analysiert.
;;;
;;; Revision 1.14  1993/04/20  11:24:55  kl
;;; Fehler in der Seiteneffektanalyse werden besser aufgefangen.
;;; Typisierung der mapping-Funktionen und der Funktion coerce verbessert.
;;;
;;; Revision 1.13  1993/04/19  12:25:59  kl
;;; Die durch appfuns gelieferten Informationen werden jetzt besser genutzt.
;;;
;;; Revision 1.12  1993/04/15  08:29:12  kl
;;; nunion durch union ersetzt und appfuns benutzt.
;;;
;;; Revision 1.11  1993/03/22  17:32:32  hk
;;; Keywords in LZS Slots.
;;;
;;; Revision 1.10  1993/03/18  13:46:49  kl
;;; Umstellung auf die neue Bedeutung des analysed-Slots.
;;;
;;; Revision 1.9  1993/03/10  08:50:01  kl
;;; Bei fehlender Hauptfunkion wird kein Fehler mehr erzeugt.
;;;
;;; Revision 1.8  1993/03/09  07:21:25  ft
;;; Erweiterung von analyse-types um eine Methode fuer Klassen.
;;;
;;; Revision 1.7  1993/03/05  15:50:48  kl
;;; Die used-Annotation wird nicht mehr verwendet.
;;;
;;; Revision 1.6  1993/03/04  10:44:59  kl
;;; Anpassung an die eingefuehrten Typinferenzlevel.
;;;
;;; Revision 1.5  1993/02/16  16:10:20  hk
;;; Revision Keyword eingefuegt.
;;;
;;; Revision 1.4  1993/02/15  14:46:19  kl
;;; Ist eine Variable nicht (Typ-)gebunden, dann hat sie Typ bottom-t.
;;; Das Ruecksetzen der Typumgebungen auf bottom-t geschieht
;;; durch Loeschen aller Eintraege.
;;;
;;; Revision 1.3  1993/02/02  09:58:54  kl
;;; Initialisierungen der Vorgaengertypumgebungen nach tipass1 verlegt.
;;;
;;; Revision 1.2  1993/01/26  18:34:51  kl
;;; Umbenennung der globalen Variablen vorgenommen. Einige Variablen
;;; nach tidef.lisp verlegt.
;;;
;;; Revision 1.1  1993/01/26  17:10:49  kl
;;; Initial revision
;;;-----------------------------------------------------------------------------

(in-package "CLICC")

;;------------------------------------------------------------------------------

(require "titypes")
(require "tidef")
(require "tidecl")
(require "timisc")


;;------------------------------------------------------------------------------
;; Es folgen die Methoden zu der generischen Funktion `analyse-types', die zu
;; jedem Zwischensprachkonstrukt die entsprechenden Sprachteile analysiert.
;; Waehrend des Traversierens werden die entsprechenden Typannotationen gesetzt.
;;------------------------------------------------------------------------------
(defgeneric analyse-types (analysable-object))


;;------------------------------------------------------------------------------
;; Methode zum Auffangen der Fehlerfaelle.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((anything T))
  (internal-error 'analyse-types "type analysis can't analyse ~S." anything))


;;------------------------------------------------------------------------------
;; analyse-functions analysiert die in a-function-list angegebenen Funktionen
;; bis zum Erreichen des Typfixpunktes.
;;------------------------------------------------------------------------------
(defun analyse-objects (list)
  (do ((*ti-workset*  list)
       (iterations 0 (1+ iterations)))
      ((endp *ti-workset*))
      
    (write-size-of-workset *ti-workset* iterations)
    (analyse-object (pop *ti-workset*))))


;;------------------------------------------------------------------------------
;; Die Typanalyse einer im Modul definierten Funktion bindet die Argumenttypen
;; an die Parameter und analysiert den Rumpf.
;;------------------------------------------------------------------------------
(defmethod analyse-object ((a-defined-fun defined-fun))
  (let* ((*type-environment* (copy-type-env (?pred-type-env a-defined-fun))))
    
    ;; Binden der Parameter:
    ;;----------------------
    (bind-parameter-types (?params a-defined-fun))

    ;; Analyse des Funktionsrumpfes
    ;;-----------------------------
    (let* ((old-result-type     (?result-type a-defined-fun))
           (old-result-type-env (?result-type-env a-defined-fun))
           (new-result-type     (analyse-types (?body a-defined-fun)))
           (types-are-unchanged (type-eq old-result-type new-result-type)))
      
      (when (> *ti-level* 3)
        (multiple-value-bind (new-result-type-env env-changed)
            (join-pred-type-env-with-type-env old-result-type-env 
                                              *type-environment*)
          (when env-changed
            (setf (?result-type-env a-defined-fun) new-result-type-env
                  types-are-unchanged nil))))

      (unless types-are-unchanged
        (setf (?result-type a-defined-fun) new-result-type)
        (setf *ti-workset* (nunion (union (?called-by a-defined-fun)
                                          (list a-defined-fun))
                                   *ti-workset* ))))))


;;------------------------------------------------------------------------------
(defmethod analyse-object ((a-class-def class-def))
  (mapc #'analyse-types (mapcar #'?initform (?slot-descr-list a-class-def))))


;;------------------------------------------------------------------------------
;; Die Typanalyse von bennanten Konstanten liefert den Wert der bei der 
;; einmaligen Zuweisung gesetzten Typkomponente.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-named-const named-const))
  (?type a-named-const))


;;------------------------------------------------------------------------------
;; Bei Variablenreferenzen wird die zugehoerige Variable analysiert.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-var-ref var-ref))
  (analyse-types (?var a-var-ref)))


;;------------------------------------------------------------------------------
;; Bei statischen Variablen wird der Wert der Typbindung geliefert.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-static static))
  (get-type a-static))


;;------------------------------------------------------------------------------
;; Die Analyse dynamischer Variablen liefert entweder die Typbindung oder, falls
;; sie eine Common Lisp-Konstante mit bekanntem Wert repraesentiert, den Typ der
;; Konstanten.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-dynamic dynamic))
  (let ((constant-value (?constant-value (?sym a-dynamic))))
    (case constant-value
      (:no-const
       (get-type a-dynamic))
      (:unknown
       (?type a-dynamic))
      (otherwise
       (analyse-types constant-value)))))


;;------------------------------------------------------------------------------
;; Continuations als funktionales Objekt liefern den Bottom-Typ.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-cont cont))
  function-t)


;;------------------------------------------------------------------------------
;; Typanalyse der Literale liefert den im tiPass1 ermittelten (konstanten) Typ.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-literal literal))
  (?type a-literal))

(defmethod analyse-types ((a-fun fun))
  function-t)


;;------------------------------------------------------------------------------
;; Wird eine Continuation appliziert, dann wird das Typ-Bottomelement geliefert.
;; Ausserdem wird in die Annotation `type' der Argumenttyp der Applikation mit
;; aufgenommen und die aktuelle Typumgebung wird auf die leere Umgebung gesetzt.
;;------------------------------------------------------------------------------
(defmethod analyse-function-app ((a-cont cont) argument-types)
  (update-type-f (?result-type a-cont) (first argument-types))
  (setf (?type-env a-cont) 
        (join-type-environments (?type-env a-cont) *type-environment*)
        *type-environment*
        (empty-environment))
  bottom-t)


;;------------------------------------------------------------------------------
;; Die Applikation einer benutzerdefinierten Funktion liefert den Wert der
;; Komponente ?result-type.
;;------------------------------------------------------------------------------
(defun analyse-defined-function-app (a-fun argument-types &optional union-type)
  (let* ((old-type-env    
          *type-environment*)
         (old-pred-type-env   
          (?pred-type-env a-fun))
         (param-types-changed 
          (not (if union-type
                   (simple-set-parameter-types (?params a-fun) union-type)
                   (set-parameter-types (?params a-fun) argument-types)))))

    (multiple-value-bind (new-pred-type-env env-changed)
      (join-pred-type-env-with-type-env old-pred-type-env *type-environment*)

      (cond (env-changed 
             (setf (?pred-type-env a-fun) new-pred-type-env)
             (pushnew a-fun *ti-workset*))
            (param-types-changed
             (pushnew a-fun *ti-workset*))
            (T nil)))
      
    (setf *type-environment* old-type-env)
    (?result-type a-fun)))


;;------------------------------------------------------------------------------
(defmethod analyse-function-app ((a-fun fun) argument-types)
  (analyse-defined-function-app a-fun argument-types))


;;------------------------------------------------------------------------------
;; Die Typanalyse einer importierten Funktion liefert wenn die Funktion bekannt
;; ist den entsprechenden Resultattyp, das Top-Element des Typverbandes sonst.
;;------------------------------------------------------------------------------
(defmethod analyse-function-app ((an-imp-fun imported-fun) argument-types)
  (let ((type-abstraction-function (?type-abstraction-function an-imp-fun)))

    (if (null type-abstraction-function)
        (?result-type an-imp-fun)
        (apply type-abstraction-function argument-types))))



;;------------------------------------------------------------------------------
;; Der Typ einer Applikation ist der Ergebnistyp des applizierten Ausdrucks bei
;; Anwendung auf die Argumente.
;; Applikationen koennen Seiteneffekte haben. Deshalb ist die Typumgebung nach 
;; der Applikationen so zu aendern, dass die Bindungen der von Seiteneffekten
;; der applizierten Funktion abhaengenden Variablen geloescht werden.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((an-app app))
  (let ((form     (?form     an-app))
        (arg-list (?arg-list an-app)))
    (analyse-types form)
    (let* ((arg-types 
            (mapcar #'analyse-types arg-list))
           (called-funs
            (?called-funs an-app))
           (result-types 
            (mapcar (if (and (special-sys-fun-p form) (?special-caller form))
                        #'(lambda (a-fun)
                            (funcall (?special-caller form) a-fun arg-list))
                         
                        #'(lambda (a-fun)
                            (if (and (special-sys-fun-p a-fun) 
                                     (?tipass a-fun))
                                (apply (?tipass a-fun) arg-list)
                                (analyse-function-app a-fun arg-types))))
                                   
                    called-funs))
           (result
            (apply #'multiple-type-join result-types)))

      (when (and (imported-fun-p form)
                 (?argument-types form))
        (mapc #'(lambda (argument assertion)
                  (when (var-ref-p argument)
                    (assert-type (?var argument) assertion *type-environment*)))
              arg-list (?argument-types form)))

      (if (?other-funs an-app)
          top-t
          result))))


;;------------------------------------------------------------------------------
;; Hilfsfunktion zum Abfragen auf destruktive Seiteneffekte einer Applikation.
;;------------------------------------------------------------------------------
(defun has-destructive-effects (an-app)
  (let ((data-effects (?data-effects an-app)))
    (or (eq data-effects :dest)
        (eq data-effects :alloc-dest))))

;;------------------------------------------------------------------------------
;; Aendere die Typbindungen entsprechend der moeglichen Seiteneffekte der
;; Applikation.
;;------------------------------------------------------------------------------
(defmethod analyse-types :after ((an-app app))

  ;; Wenn der *ti-level* genuegend gross ist und eine definierte Funktion
  ;; appliziert worden ist, dann aktualisiere die Typumgebung um die in
  ;; der applizierten Funktion erfolgten Typaenderungen.
  ;; Die geaenderten Typbindungen liegen in der Komponente `result-type-env' 
  ;; der applizierten Funktion.
  ;;
  ;; Ansonsten setze die Typbindungen der von der Seiteneffektanalyse als
  ;; schreibend veraendert angezeigten Variablen auf das Topelement.
  ;; Wenn die Menge der schreibend veraenderten Variablen unbekannt ist, 
  ;; dann setze alle Typbindungen auf das Topelement. 
  ;; Wenn durch die vorliegende Funktionsanwendung destruktive Seiteneffekte 
  ;; auf Variablen ausgeloest worden sind, setze die Typbindungen geeignet
  ;; zurueck.
  ;;------------------------------------------------------------------------
  (cond ((and (> *ti-level* 3) (defined-fun-p (?form an-app)))
         (setf *type-environment* 
               (update-type-env *type-environment* 
                                (?result-type-env (?form an-app))))
         (when (has-destructive-effects an-app)
             (reset-type-bindings-after-destructive-side-effect)))
        (T 
         (let ((write-list (?write-list an-app)))
           (cond ((listp write-list) 
                  (when (has-destructive-effects an-app) 
                    (reset-type-bindings-after-destructive-side-effect))
                  (reset-type-bindings write-list))
                 (T
                  (set-all-type-bindings-to top-t)))))))


;;------------------------------------------------------------------------------
;; Die Applikationen der folgenden Funktionen werden in `appfuns',
;; `static-effect' und in der Typinferenz gesondert behandelt.
;;------------------------------------------------------------------------------
(p0-special-funs
 (?special-caller "SC")
 clicc-lisp::funcall
 clicc-lisp::apply
 clicc-lisp::mapcar
 clicc-lisp::maplist
 clicc-lisp::mapcon
 clicc-lisp::mapcan
 clicc-lisp::mapc
 clicc-lisp::mapl)
 

;;------------------------------------------------------------------------------
;; 
;;------------------------------------------------------------------------------
(defun SC-funcall (a-fun argument-list)
  (analyse-function-app a-fun (mapcar #'?type (rest argument-list))))

;;------------------------------------------------------------------------------
(defun SC-apply (a-fun argument-list)
  (if (imported-fun-p a-fun)
      (?result-type a-fun)
      (let* ((raw-args      (rest argument-list))
             (raw-arg-types (mapcar #'?type raw-args))
             (applied-types (if raw-arg-types
                                (cons (list-component 
                                       (first (last raw-arg-types)))
                                      (butlast raw-arg-types))
                                top-t))
             (union-type    (apply #'multiple-type-join applied-types)))
        (analyse-defined-function-app a-fun nil union-type))))
      

;;------------------------------------------------------------------------------
(defun SC-mapcar (a-fun argument-list)
  (let* ((list-types (mapcar #'?type (rest argument-list)))
         (arg-types  (mapcar #'list-component list-types))
         (result     (analyse-function-app a-fun arg-types)))
    (if (some #'(lambda (list-type) 
                  (types-are-conform list-type null-t))
              list-types)
        (list-of result)
        (list-cons-of result))))


(defun SC-maplist (a-fun argument-list)
  (let* ((list-types (mapcar #'?type (rest argument-list)))
         (arg-types  (mapcar #'(lambda (lt) (type-join null-t lt)) list-types))
         (result     (analyse-function-app a-fun arg-types)))
    (if (some #'(lambda (list-type) 
                  (types-are-conform list-type null-t))
              list-types)
        (list-of result)
        (list-cons-of result))))


(defun SC-mapcan (a-fun argument-list)
  (SC-mapcar a-fun argument-list))

(defun SC-mapcon (a-fun argument-list)
  (SC-maplist a-fun argument-list))

;;------------------------------------------------------------------------------
(defun SC-mapc (a-fun argument-list)
  (SC-mapcar a-fun argument-list)
  (?type (second argument-list)))

(defun SC-mapl (a-fun argument-list)
  (SC-maplist a-fun argument-list)
  (?type (second argument-list)))

                 
;;------------------------------------------------------------------------------
;; Die Applikationen der folgenden Funktionen werden in `appfuns',
;; `static-effect' und in der Typinferenz gesondert behandelt.
;;------------------------------------------------------------------------------
(p0-special-funs
 (?tipass "TI")
 clicc-lisp::coerce
 clicc-lisp::concatenate)

;;------------------------------------------------------------------------------
(defun ti-coerce (object result-type)
  (declare (ignore object))
  (get-intern-type result-type))

(defun ti-concatenate (sequence-type &rest arguments)
  (declare (ignore arguments))
  (type-meet sequence-t (get-intern-type sequence-type)))
 

;;------------------------------------------------------------------------------
;; Der Typ eines setq ist der Typ der zugehoerigen Form. Als Seiteneffekt 
;; wird bei Zuweisungen an Variablen die entsprechende Typbindung aktualisiert.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-setq-form setq-form))
  (let ((location (?location a-setq-form))
        (result   (analyse-types (?form a-setq-form))))
    
    ;; Bei Variablenreferenzen wird noch deren Typbindung aktualisiert.
    ;;-----------------------------------------------------------------
    (unless (named-const-p location)
      (let ((var (?var location)))
        (when (or (static-p var)
              (and (dynamic-p var)
                   (eq (?constant-value (?sym var)) :no-const))))
        (let ((type-binding (get-type-binding var)))
          (if type-binding
              (setf (rest type-binding) result)
              (push-type var result *type-environment*))
          (update-type-f (?type var) result))))
      
    result))


;;------------------------------------------------------------------------------
;; Alle Elemente der form-list werden analysiert. Analysierte Typbindungen 
;; werden von Ausdruck zu Nachfolgeausdruck propagiert. Geliefert wird der 
;; Typ des letzten Ausdrucks.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-progn-form progn-form))
  (mapc-progn-form-list (?form-list a-progn-form)
                        #'analyse-types
                        #'analyse-types))


;;------------------------------------------------------------------------------
;; Der Typ eines if-Konstruktes ist der Vereinigungstyp der Typen der then- und
;; der else-Ausdruecke.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((an-if-form if-form))

  (analyse-types (?pred an-if-form))
  
  ;; Merke die Typumgebung vor der Analyse der then- und else-Teile.
  ;;----------------------------------------------------------------
  (multiple-value-bind (then-entry-env else-entry-env)
      (get-type-assertions-from-predicate-position (?pred an-if-form) 
                                                   *type-environment*)
    
    (let (then-exit-type 
          then-exit-env 
          else-exit-type 
          else-exit-env)
      
      ;; Analyse des then-Teils:
      ;;------------------------
      (setf *type-environment* then-entry-env
            then-exit-type     (analyse-types (?then an-if-form))
            then-exit-env      *type-environment*)
      
      ;; Analyse des else-Teils:
      ;;------------------------
      (setf *type-environment* else-entry-env
            else-exit-type     (analyse-types (?else an-if-form))
            else-exit-env      *type-environment*)
      
      ;; Setzen der Ergebnisumgebung und Rueckgabe des Resultates:
      ;;----------------------------------------------------------
      (setf *type-environment* (join-type-environments then-exit-env
                                                       else-exit-env))
      (type-join then-exit-type else-exit-type))))


;;------------------------------------------------------------------------------
;; Die Typanalyse einer switch-Form analysiert die Form zur Bestimmung des
;; Schluessels und liefert dann den Vereinigungstypen der Analysen der
;; case-Liste.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-switch-form switch-form))
  (analyse-types (?form a-switch-form))
  
  ;; Merke die Typumgebung vor der Analyse des switch-Konstruktes.
  ;;-------------------------------------------------------------- 
  (let* ((old-environment    *type-environment*)
         (result-type        (analyse-types (?otherwise a-switch-form)))
         (result-environment *type-environment*))
    
    (dolist (a-case (?case-list a-switch-form))
      (setf *type-environment* (copy-type-env old-environment)
            result-type        (type-join (analyse-types a-case) result-type)
            result-environment (join-type-environments *type-environment*
                                                       result-environment)))
    
    (setf *type-environment* result-environment)
    result-type))


;;------------------------------------------------------------------------------
;; Analysiere den Wert und den Ausdruck der labeled-form.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-labeled-form labeled-form))
  (analyse-types (?value a-labeled-form))
  (analyse-types (?form a-labeled-form)))


;;------------------------------------------------------------------------------
;; Der Typ einer labels-form ist der Typ des Rumpfes.
;; Die Funktionsbeschreibungen zu den hier lokal definierten Funktionen sind
;; zu Beginn der Analyse bereits angelegt worden.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-labels-form labels-form))
  (analyse-types (?body a-labels-form)))


;;------------------------------------------------------------------------------
;; Die Typanalyse einer let*-Form bindet die Typen der Initialisierungsformen an
;; die zu bindenden Variablen. Als Ergebnis wird das Analyseergebnis des Rumpfes
;; geliefert.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-let*-form let*-form))
  (let ((old-type-env (copy-type-env *type-environment*)))
    
    (mapc #'(lambda (variable init-form)
              (bind-and-update-type variable (analyse-types init-form)))
          (?var-list a-let*-form)
          (?init-list a-let*-form))
    
    (let ((result (analyse-types (?body a-let*-form))))
      
      ;; Ruecksetzen der Typumgebung
      ;;----------------------------
      (setf *type-environment* 
            (update-type-env old-type-env *type-environment*))
      
      result)))


;;------------------------------------------------------------------------------
;; Bei let/cc-Ausdruecken wird der Rumpf analysiert. Waehrend der Analyse des
;; Rumpfes werden jeweils die Typen und die aktuellen Typumgebungen der 
;; Ruecksprungstellen in Annotationen der Continuation abgelegt.
;; Das Analyseergebnis ist davon abhaengig, ob die Continuation (auch) als
;; funktionales Objekt verwendet wird oder nicht.
;; Wenn die Continuation als funktionales Objekt verwendet wird, dann kann mit
;; irgendeiner Typumgebung und einem unbekannten Typ zurueckgesprungen werden.
;; Deshalb wird die Typumgebung zurueckgesetzt und als Ergebnistyp das Top-
;; element geliefert.
;; Wenn die Continuation nicht als funktionales Objekt verwendet wird, dann
;; werden die Ergebnistypumgebung des Rumpfes und die an den Ruecksprungstellen
;; gesammelten Typumgebungen vereinigt. Der Ergebnistyp ist die Typvereinigung
;; aus dem Ergebnistyp des Rumpfes und den an den Ruecksprungstellen gesammelten
;; Typen.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-let/cc-form let/cc-form))
  (let ((cont (?cont a-let/cc-form)))

    (let ((body-result-type (analyse-types (?body a-let/cc-form)))
          (body-result-env  *type-environment*))
      
      (cond ((?unknown-caller cont)
             (set-all-type-bindings-to top-t)
             top-t)
            (T 
             (setf *type-environment* 
                   (join-type-environments body-result-env (?type-env cont)))
             (type-join body-result-type (?result-type cont)))))))


;;------------------------------------------------------------------------------
;; Bei der Analyse der tagbody-Konstrukte wird das folgende Fixpunktverfahren
;; angewendet. Fuer alle tagged-forms werden die Typumgebungen saemtlicher 
;; Vorgaenger im Datenfluss gesammelt. In einem tagbody-Konstrukt mit den 
;; tagged-forms {T0, T1, ..., Tn} sind die Vorgaenger der tagged-form Ti
;; alle Ausdruecke der Form (GO Ti) und Ti-1.
;;------------------------------------------------------------------------------
(defvar *type-env-changed*)

(defmethod analyse-types ((a-tagbody-form tagbody-form))
  (let ((result (analyse-types (?first-form a-tagbody-form)))
        (tagged-form-list (?tagged-form-list a-tagbody-form)))

    (case (length tagged-form-list)
      (0 result)

      ;; Der Spezialfall einer Schleife laesst sich einfach handhaben.
      ;;--------------------------------------------------------------
      (1 (let ((tagged-form (first tagged-form-list)))
           (add-predecessor-type-env tagged-form 
                                     (copy-type-env *type-environment*))
           
           (loop 
            (let* ((*type-env-changed* nil)
                   (*type-environment* 
                    (copy-type-env (?pred-type-env tagged-form)))
                   (result (analyse-types (?form tagged-form))))
              (unless *type-env-changed* 
                (return result))))))
            
      ;; Behandlung der Tagbody-Konstrukte mit mehr als einer tagged-form:
      ;;------------------------------------------------------------------
      (T (add-predecessor-type-env (first tagged-form-list) 
                                   (copy-type-env *type-environment*))
         (let ((entry-env *type-environment*))
           
           (loop 
            (let* ((*type-env-changed* nil)
                   (*type-environment* entry-env)
                   result)
              
              (dolist (tagged-form tagged-form-list)
                (add-predecessor-type-env tagged-form *type-environment*)
                (setf *type-environment* 
                      (copy-type-env (?pred-type-env tagged-form))
                      result 
                      (analyse-types (?form tagged-form))))
              
              (unless *type-env-changed*
                (return result)))))))))


;;------------------------------------------------------------------------------
;; Wird eine tagged-form analysiert, dann handelt es sich um ein angewandtes
;; Vorkommen. Alle definierenden Vorkommen werden schon im zugehoerigen 
;; tagbody-Konstrukt analysiert.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-tagged-form tagged-form))
  (add-predecessor-type-env a-tagged-form *type-environment*)
  (setf *type-environment* (empty-environment))
  bottom-t)


;;------------------------------------------------------------------------------
;; add-predecessor-type-env vereinigt eine Typumgebung mit der bisherigen 
;; Vereinigungsumgebung aller Vorgaenger.
;;------------------------------------------------------------------------------
(defun add-predecessor-type-env (a-tagged-form type-env)
  (multiple-value-bind (result-env type-env-changed)
      (join-type-environments (?pred-type-env a-tagged-form) type-env)
    (setf (?pred-type-env a-tagged-form) result-env)
    (when type-env-changed
      (setf *type-env-changed* T))))


;;------------------------------------------------------------------------------
;; Bei mv-lambda wird zur Zeit noch an alle Parameter das Top-Element des 
;; Typverbandes gebunden.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((an-mv-lambda mv-lambda))
  ;; Analysiere zuerst den Ausdruck, der eventuell multiple Werte liefert.
  (analyse-types (?arg an-mv-lambda))
  
  (let* ((old-type-env (copy-type-env *type-environment*))
         (params       (?params an-mv-lambda)))
    
    ;; Setze alle Parametertypen auf das Topelement.
    ;;----------------------------------------------
    (simple-set-parameter-types params)
    
    ;; Binden der Parameter
    ;;---------------------
    (bind-parameter-types params)
    
    ;; Analyse des Rumpfs
    ;;-------------------
    (let ((result (analyse-types (?body an-mv-lambda))))
      
      ;; Ruecksetzen der Typumgebung
      ;;----------------------------
      (setf *type-environment* 
            (update-type-env old-type-env *type-environment*))
      
      result)))


;;------------------------------------------------------------------------------
;; Alle Klassen haben zunaechst den Typ class-t
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-class class-def))
  class-t)


;;------------------------------------------------------------------------------
;; Analyse der foreign-fun-Konstrukte liefert zur Zeit noch das Topelement des
;; Typverbands.
;;------------------------------------------------------------------------------
(defmethod analyse-types ((a-foreign-fun foreign-fun))
  top-t)


;;------------------------------------------------------------------------------
;; In der around-Methode zu analyse-types wird die Typkomponente der analysier-
;; ten Form gesetzt.
;;------------------------------------------------------------------------------
(defmethod analyse-types :around ((a-form form))
  (update-type-f (?type a-form) (call-next-method)))

(defmethod analyse-types :around ((a-cont cont))
  (update-type-f (?type a-cont) (call-next-method)))

;;------------------------------------------------------------------------------
(provide "tipass2")



