#ifndef lint
static char *RcsId="$Id: com_territory.c,v 2.22 91/03/13 21:46:35 keisuke Exp $";
#endif

/* Functions for Handling Territories
 * This file is part of YY-server of YYonX (1.3 Distribution)
 * $Id: com_territory.c,v 2.22 91/03/13 21:46:35 keisuke Exp $
 */

/****************************************************************************
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   Version 1.0 90/02/26 by Keisuke 'Keiko' Tanaka
;;;				(keisuke@csrl.aoyama.ac.jp)
;;;   Version 2.0 90/08/27 by Keisuke 'Keiko' Tanaka
;;;			Page Mode Territory is supported
;;;   Version 2.15 90/11/05 by Keisuke 'Keiko' Tanaka
;;;			Copyright Notice is rewritten
;;;
****************************************************************************/

/****************************************************************************
  $Revision: 2.22 $ Written by Keisuke 'Keiko' Tanaka
  $Date: 91/03/13 21:46:35 $
****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <X11/cursorfont.h>
#include "yydefs.h"
#include "yypacket.h"
#include "xwindow.h"
#include "territory.h"

static Window create_window_for_root_territory();
static Window create_window_for_territory();
static TERRITORY *territory_alloc();
static void territory_free();
static void territory_unlink();
static void list_territory_for_debug();

/*
 * Territory Control
 */

extern void clear_page_territory();
CTRLFUNCS CtrlFuncTab[] = {
{ TR_NONE,	NULL, NULL, NULL, NULL, NULL },
{ TR_PAGE,	NULL, NULL, NULL, NULL, clear_page_territory },
{ TR_VIEWPORT,	NULL, NULL, NULL, NULL, NULL },
{ TR_FENCE,	NULL, NULL, NULL, NULL, NULL },
{ 0,		NULL, NULL, NULL, NULL, NULL },
} ;

static CTRLFUNCS *get_control_functab(type)
    int type;
{
    register CTRLFUNCS *ft = CtrlFuncTab;
    for (ft = CtrlFuncTab; ft->TerritoryType > 0; ft++)
	if (ft->TerritoryType == type)
	    return ft;
    return ft;
}

/*
 * TERRITORY *get_specified_territory(pkt, err)
 *
 * read one integer value from packet
 * and get the entry for the specified territory
 */

TERRITORY *get_specified_territory(pkt, err)
    yy_packet *pkt;
    yy_packet **err;
{
    register territory_entry *tr;
    tr = search_territory(read_packet_entry_integer(pkt));
    if (tr != (TERRITORY *)NULL)
	return tr;
    *err = create_error_packet(pkt, YYERROR_NOTERRITORY);
    return (TERRITORY *)NULL;
}

TERRITORY *catch_territory(pkt, type, err)
    yy_packet *pkt;
    register int type;		/* Territory Type */
    yy_packet **err;
{
    register territory_entry *tr;
    tr = search_territory(read_packet_entry_integer(pkt));
    if (tr == (TERRITORY *)NULL || !(tr->teTRType & type)) {
	*err = create_error_packet(pkt, (tr? YYERROR_TYPEMISMATCH:
					 YYERROR_NOTERRITORY));
	return (TERRITORY *)NULL;
    }
    return tr;
}


yy_packet *yycom_create_territory(CH, pkt)
    yy_comm_channel *CH;
    yy_packet *pkt;
{
    TERRITORY *TR;
    int x, y, w, h;
    TRID parent_id;	/* ID for the Parent Territory */
    bool visible, drawable;
    int draw_flag;
    int type;
    yy_packet *repl;
    int err_code;

    debug_setfunc("territory", "yycom_create_territory");
    /* Size of Territory */
    x = read_packet_entry_integer(pkt);
    y = read_packet_entry_integer(pkt);
    w = read_packet_entry_integer(pkt);
    h = read_packet_entry_integer(pkt);
    /* Parent ID */
    parent_id = (TRID)read_packet_entry_integer(pkt);
    /* Visible or Not */
    visible = ((read_packet_entry_integer(pkt)&1)? TRUE: FALSE);
    /* Type of Territory */
    draw_flag = read_packet_entry_integer(pkt);
    drawable = ((draw_flag&0400)? TRUE: FALSE);
    switch (draw_flag) {
    case 0:	type = TR_NONE;		break;
    case 01:	type = TR_FENCE;	break;
    case 0401:	type = TR_VIEWPORT;	break;
    case 0400:	type = TR_PAGE;		break;
    default:
	debug_print(1, " Unknown Territory Type: 0%o\n", draw_flag);
	repl = ALLOC_ERRPACKET(pkt);
	SETYYERRORCODE(repl, 0);
	SETYYERRORCODE(repl, YYERROR_TYPEMISMATCH);
	debug_endfunc("yycom_create_territory");
	return repl;
    }
    /* Create Territory */
    TR = create_territory(CH, x, y, w, h, parent_id,
			  visible, drawable, type, &err_code);
    if (TR != (TERRITORY *)NULL) {
	debug_print(1, " Create new Territory (ID:%d)\n", TR->teID);
	repl = ALLOC_ACKPACKET(pkt);
	append_packet_entry_integer(repl, TR->teID);
	if (debug_on(6))
	    list_territory_for_debug();
    } else {
	debug_print(1, " Can't Create Territory, Error Code=%d\n", err_code);
	repl = ALLOC_ERRPACKET(pkt);
	SETYYERRORCODE(repl, 0);
	SETYYERRORCODE(repl, err_code);
    }
    debug_endfunc("yycom_create_territory");
    return repl;
}


yy_packet *yycom_display_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    register int visible;
    TERRITORY *tr;
    yy_packet *err;

    debug_setfunc("territory", "yycom_display_territory");
    if ((tr = catch_territory(pkt, TR_VISIBLE, &err)) == (TERRITORY *)NULL)
	return err;
    visible = (read_packet_entry_integer(pkt)&1);
    (void) display_territory(ch, tr, visible);
    debug_endfunc("yycom_display_territory");
    return (yy_packet *)NULL;
}

yy_packet *yycom_move_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    TERRITORY *tr;
    yy_packet *err;
    int new_x, new_y;

    debug_setfunc("territory", "yycom_move_territory");
    if ((tr = catch_territory(pkt, TR_ALL, &err)) == (TERRITORY *)NULL)
	return err;
    new_x = read_packet_entry_integer(pkt);
    new_y = read_packet_entry_integer(pkt);
    (void) move_territory(ch, tr, new_x, new_y);
    debug_endfunc("yycom_move_territory");
    return (yy_packet *)NULL;
}


yy_packet *yycom_resize_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    TERRITORY *tr;
    yy_packet *err;
    int nx, ny, nw, nh; /* New Size */
    int origX, origY;
    debug_setfunc("territory", "yycom_resize_territory");
    if ((tr = catch_territory(pkt, TR_ALL, &err)) == (TERRITORY *)NULL)
	return err;
    nx = read_packet_entry_integer(pkt);	/* new Origin */
    ny = read_packet_entry_integer(pkt);
    nw = read_packet_entry_integer(pkt);	/* new Size */
    nh = read_packet_entry_integer(pkt);
    origX = read_packet_entry_integer(pkt);	/* */
    origY = read_packet_entry_integer(pkt);
    (void) resize_territory(ch, tr, nx, ny, nw, nh, origX, origY);
    debug_endfunc("yycom_resize_territory");
    return (yy_packet *)NULL;
}


yy_packet *yycom_destroy_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *repl;
    int id;

    debug_setfunc("territory", "yycom_destroy_territory");
    if ((tr = catch_territory(pkt, TR_ALL, &repl)) == (TERRITORY *)NULL)
	return repl;
    /* Save Territtory ID, because destroy_territory() removes
     * all data on *tr */
    id = tr->teID;
    debug_print(5, " -- destroy territory #%d\n", id);
    ch->ccXNeedFlush = TRUE;
    if (destroy_territory(xp->xDisp, tr, TRUE)) {
	repl = ALLOC_ACKPACKET(pkt);
	ch->ccXNeedFlush = TRUE;
	append_packet_entry_integer(pkt, id);
    } else {
	/* Error.. */
	repl = ALLOC_NACKPACKET(pkt);
	append_packet_entry_integer(pkt, 0);
    }
    debug_endfunc("yycom_destroy_territory");
    return repl;
}


yy_packet *yycom_reparent_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    territory_entry *tr, *parent;
    yy_packet *err;
    int nx, ny;

    debug_setfunc("territory", "yycom_reparent_territory");
    /* Target Territory */
    if ((tr = catch_territory(pkt, TR_ALL, &err)) == (TERRITORY *)NULL)
	return err;
    /* New Parent */
    if ((parent = catch_territory(pkt, TR_VISIBLE, &err)) == (TERRITORY *)NULL)
	return err;
    if (tr->teParent == (TERRITORY *)NULL ||
	parent->teID == tr->teParent->teID || parent->teID == tr->teID)
	return create_error_packet(pkt, YYERROR_NOTERRITORY);
    nx = read_packet_entry_integer(pkt);
    ny = read_packet_entry_integer(pkt);
    (void) reparent_territory(ch, tr, parent, nx, ny);
    debug_endfunc("yycom_reparent_territory");
    return (yy_packet *)NULL;
}

yy_packet *yycom_raise_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    territory_entry *tr;
    yy_packet *err;
    debug_setfunc("territory", "yycom_raise_territory");
    if (NULLTERRITORY((tr = catch_territory(pkt, TR_ALL, &err))))
	return err;
    if (tr->teParent != (TERRITORY *)NULL)
	(void) raise_territory(ch, tr);
    debug_endfunc("yycom_raise_territory");
    return (yy_packet *)NULL;
}

yy_packet *yycom_lower_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    territory_entry *tr;
    yy_packet *err;
    debug_setfunc("territory", "yycom_lower_territory");
    if (NULLTERRITORY((tr = catch_territory(pkt, TR_ALL, &err))))
	return err;
    if (tr->teParent != (TERRITORY *)NULL)
	(void) lower_territory(ch, tr);
    debug_endfunc("yycom_lower_territory");
    return (yy_packet *)NULL;
}

yy_packet *yycom_clear_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    territory_entry *tr;
    yy_packet *err;
    x_color color;
    debug_setfunc("territory", "yycom_clear_territory");
    if ((tr = catch_territory(pkt, TR_ALL, &err)) == (TERRITORY *)NULL)
	return err;
    color = read_packet_entry_color(pkt);
    clear_territory(ch, tr, color);
    debug_endfunc("yycom_clear_territory");
    return (yy_packet *)NULL;
}


yy_packet *yycom_draw_background(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    territory_entry *tr, *p_tr;
    yy_packet *err;
    debug_setfunc("territory", "yycom_draw_background");
    if ((tr = catch_territory(pkt, TR_ALL, &err)) != (TERRITORY *)NULL ||
	(p_tr = catch_territory(pkt, TR_DRAWABLE, &err)) == (TERRITORY *)NULL)
	return err;
    (void) draw_background(ch, tr, p_tr);
    debug_endfunc("yycom_draw_background");
    return (yy_packet *)NULL;
}


/************************************************************************
 ******
 ***********************************************************************/

void display_territory(ch, tr, visible)
    yy_comm_channel *ch;
    TERRITORY *tr;
    bool visible;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    debug_setfunc("territory", "display_territory");
    if (XOR(tr->teVisible, visible)) {
	tr->teVisible = visible;
	debug_print(5, " -- %s window for Territory#%d\n",
		    (visible? "Map": "Unmap"), tr->teID);
	if (visible)
	    XMapWindow(xp->xDisp, tx->txWindow);
	else
	    XUnmapWindow(xp->xDisp, tx->txWindow);
	ch->ccXNeedFlush = TRUE;
    }
    debug_endfunc("display_territory");
}

void move_territory(ch, tr, nx, ny)
    yy_comm_channel *ch;
    TERRITORY *tr;
    int nx, ny;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY *parent;
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    int lt_x = nx;
    int lt_y = ny;
    debug_setfunc("territory", "move_territory");
    if (debug_on(5)) {
	Window root; int x, y, w, h, b, d;
	XGetGeometry(xp->xDisp, tx->txWindow, &root, &x, &y, &w, &h, &b, &d);
	debug_print(5, " move Window (%d,%d)[%d,%d] -> (%d,%d)\n",
		    x, y, w, h, nx, ny);
    }
    if ((parent = GetParentTerritory(tr)) != (TERRITORY *)NULL) {
	lt_x = parent->teXOrigin + nx;
	lt_y = parent->teYOrigin + ny;
    }
    tr->teX = lt_x;
    tr->teY = lt_y;
    XMoveWindow(xp->xDisp, tx->txWindow, lt_x, lt_y);
    if (IsVisibleTerritory(tr))
	ch->ccXNeedFlush = TRUE;
    debug_endfunc("move_territory");
}

void clear_pixmap(xp, pix, gc, w, h, cp)
    x_private *xp;
    Pixmap pix;
    GC gc;
    int w, h;
    x_color_entry *cp;
{
    GC wgc;
    u_long mask;
    XGCValues gcv;
    gcv.function = GXcopy;
    if (cp->xcUsePixel) {
	gcv.fill_style = FillSolid;
	gcv.foreground = cp->xcPixel;
	mask = (GCFunction|GCFillStyle|GCForeground);
    } else {
	gcv.fill_style = FillTiled;
	gcv.tile = cp->xcPixmap;
	mask = (GCFunction|GCFillStyle|GCTile);
    }
    wgc = XCreateGC(xp->xDisp, pix, mask, &gcv);
    XFillRectangle(xp->xDisp, pix, wgc, 0, 0, w, h);
    XFreeGC(xp->xDisp, wgc);
}
    
void resize_territory(ch, tr, new_x, new_y, new_w, new_h, orig_x, orig_y)
    yy_comm_channel *ch;
    TERRITORY *tr;
    int new_x, new_y, new_w, new_h;
    int orig_x, orig_y;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY *parent;
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    int new_lt_x, new_lt_y;
    debug_setfunc("territory", "resize_territory");
    if (debug_on(5)) {
	Window root; int x, y, w, h, b, d;
	XGetGeometry(xp->xDisp, tx->txWindow, &root, &x, &y, &w, &h, &b, &d);
	debug_print(5, " resize (%d,%d)[%d,%d] -> (%d,%d)[%d,%d]\n",
		    x, y, w, h, new_x, new_y, new_w, new_h); 
    }
    new_lt_x = new_x; new_lt_y = new_y;
    if ((parent = GetParentTerritory(tr)) != (TERRITORY *)NULL) {
	new_lt_x += parent->teXOrigin; new_lt_y += parent->teYOrigin;
    }
    if (tx->txPixmap != (Pixmap)NULL
	&& (new_w != tr->teWidth || new_h != tr->teHeight)) {
	Pixmap newpix;
	int copy_src_x, copy_src_y;
	int copy_dst_x, copy_dst_y;
	int copy_w, copy_h;
	newpix = XCreatePixmap(xp->xDisp, tx->txWindow, new_w, new_h,
			       DefaultDepth(xp->xDisp, xp->xScreen));
	clear_pixmap(xp, newpix, tx->txGC, new_w, new_h,
		     &tx->txBackGroundColor);
	copy_src_x = MAX((-orig_x), 0); copy_src_y = MAX((-orig_y), 0);
	copy_dst_x = MAX(orig_x, 0); copy_dst_y = MAX(orig_y, 0);
	copy_w = MIN((copy_dst_x+(tr->teWidth-copy_src_x)),new_w)-copy_dst_x;
	copy_h = MIN((copy_dst_y+(tr->teHeight-copy_src_y)),new_h)-copy_dst_y;
	if (copy_w > 0 && copy_h > 0) {
	    debug_print(4, " CopyArea in resize_territory()\n");
	    debug_print(4, "  from (%d,%d)[%d,%d] to (%d,%d)\n",
			copy_src_x, copy_src_y, copy_w, copy_h,
			copy_dst_x, copy_dst_y);
	    XCopyArea(xp->xDisp, tx->txPixmap, newpix, tx->txGC,
		      copy_src_x, copy_src_y, copy_w, copy_h,
		      copy_dst_x, copy_dst_y);
	}
	XFreePixmap(xp->xDisp, tx->txPixmap);
	tx->txPixmap = newpix;
	tr->teWidth = new_w;
	tr->teHeight = new_h;
	XMoveResizeWindow(xp->xDisp, tx->txWindow,
			  new_lt_x, new_lt_y, new_w, new_h);
    } else if (new_w != tr->teWidth || new_h != tr->teHeight)
	XMoveResizeWindow(xp->xDisp, tx->txWindow,
			  new_lt_x, new_lt_y, new_w, new_h);
    else
	XMoveWindow(xp->xDisp, tx->txWindow, new_lt_x, new_lt_y);
    if (IsVisibleTerritory(tr))
	ch->ccXNeedFlush = TRUE;
    tr->teX = new_lt_x;
    tr->teY = new_lt_y;
    debug_endfunc("resize_territory");
}

bool destroy_territory(ch, tr, nowin)
    yy_comm_channel *ch;
    territory_entry *tr;
    bool nowin;	/* parent territory has no window */
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    territory_entry *child, *next;
    debug_setfunc("territory", "destroy_territory");
    for (child = tr->teChildren; child != (TERRITORY *)NULL; child = next) {
	next = child->teNextGen;
	if (!destroy_territory(ch, child))
	    return FALSE;
    }
    /* destroy parent */
    if (tx->txGC != (GC)NULL)
	XFreeGC(xp->xDisp, tx->txGC);
    if (tx->txPixmap != (Pixmap)NULL)
	XFreePixmap(xp->xDisp, tx->txPixmap);
    if (tx->txWindow != (Window)NULL)
	XDestroyWindow(xp->xDisp, tx->txWindow);
    territory_unlink(tr);
    debug_endfunc("destroy_territory");
    return TRUE;
}

void reparent_territory(ch, tr, parent, nx, ny)
    yy_comm_channel *ch;
    TERRITORY *tr, *parent;
    int nx, ny;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    TERRITORY_X_ENTRY *ptx = GetTerritoryXEntry(parent);
    debug_setfunc("territory", "reparent_territory");
    debug_print(5, " -- chnge parent for %d (#%d -> #%d)\n",
		tr->teID, tr->teParent->teID, parent->teID);
    /* change link */
    if (tr->tePrevGen == (TERRITORY *)NULL)
	tr->teParent->teChildren = tr->teNextGen;
    else
	tr->tePrevGen->teNextGen = tr->teNextGen;
    if (tr->teNextGen != (TERRITORY *)NULL)
	tr->teNextGen->tePrevGen = tr->tePrevGen;
    tr->tePrevGen = (TERRITORY *)NULL;
    tr->teNextGen = parent->teChildren;
    parent->teChildren = tr;
    /* Change Window Parent */
    XReparentWindow(xp->xDisp, tx->txWindow, ptx->txWindow, nx, ny);
    if (IsVisibleTerritory(tr)) {
	XMapWindow(xp->xDisp, tx->txWindow);
	ch->ccXNeedFlush = TRUE;
    }
    debug_endfunc("reparent_territory");
}

void raise_territory(ch, tr)
    yy_comm_channel *ch;
    TERRITORY *tr;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    debug_setfunc("territory", "raise_territory");
    debug_print(5, "Raise for #%d\n", tr->teID);
    if (tr->tePrevGen == (TERRITORY *)NULL) {
	debug_print(5, "Territory has been raised\n");
	debug_endfunc("raise_territory");
	return;
    }
    /* unlink */
    tr->tePrevGen->teNextGen = tr->teNextGen;
    if (tr->teNextGen != (TERRITORY *)NULL)
	tr->teNextGen->tePrevGen = tr->tePrevGen;
    /* link */
    tr->tePrevGen = (territory_entry *)NULL;
    tr->teNextGen = tr->teParent->teChildren;
    tr->teNextGen->tePrevGen = tr;
    tr->teParent->teChildren = tr;
    XRaiseWindow(xp->xDisp, tx->txWindow);
    if (IsVisibleTerritory(tr))
	ch->ccXNeedFlush = TRUE;
    debug_endfunc("raise_territory");
}

void lower_territory(ch, tr)
    yy_comm_channel *ch;
    TERRITORY *tr;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY *ltr;
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    debug_setfunc("territory", "lower_territory");
    debug_print(5, "Lower for #%d\n", tr->teID);
    if (tr->teNextGen == (TERRITORY *)NULL) {
	debug_print(5, "Territory has been lowerd\n");
	debug_endfunc("lower_territory");
	return;
    }
    /* unlink */
    tr->teNextGen->tePrevGen = tr->tePrevGen;
    if (!NULLTERRITORY(tr->tePrevGen))
	tr->tePrevGen->teNextGen = tr->teNextGen;
    else
	tr->teParent->teChildren = tr->teNextGen;
    /* link */
    for (ltr = tr->teNextGen; !NULLTERRITORY(ltr->teNextGen); )
	ltr = ltr->teNextGen;
    ltr->teNextGen = tr;
    tr->tePrevGen = ltr;
    tr->teNextGen = (territory_entry *)NULL;
    XLowerWindow(xp->xDisp, tx->txWindow);
    if (IsVisibleTerritory(tr))
	ch->ccXNeedFlush = TRUE;
    debug_endfunc("lower_territory");
}

static u_long set_gc_for_territory_foreground(cp, gcvp)
    x_color_entry *cp;
    XGCValues *gcvp;
{
    if (cp->xcUsePixel) {
	gcvp->fill_style = FillSolid;
	gcvp->foreground = cp->xcPixel;
	return (GCFillStyle|GCForeground);
    }
    gcvp->fill_style = FillTiled;
    gcvp->tile = cp->xcPixmap;
    return (GCFillStyle|GCTile);
}

static u_long set_gc_for_territory_background(cp, gcvp)
    x_color_entry *cp;
    XGCValues *gcvp;
{
    if (cp->xcUsePixel) {
	gcvp->background = cp->xcPixel;
	return (GCForeground);
    }
    gcvp->fill_style = FillTiled;
    gcvp->tile = cp->xcPixmap;
    return (GCFillStyle|GCTile);
}

void clear_territory(ch, tr, color)
    yy_comm_channel *ch;
    TERRITORY *tr;
    x_color color;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    XGCValues gcv;
    debug_setfunc("territory", "clear_territory");
    if (IsDrawableTerritory(tr) || tr->teTRType == TR_NONE) {
	debug_print(5, "Clear #%d\n", tr->teID);
	gcv.function = GXcopy;
	gcv.fill_style = FillSolid;
	gcv.foreground = color;
	gcv.background = color;
	XChangeGC(xp->xDisp, tx->txGC,
		  (GCFunction|GCFillStyle|GCForeground|GCBackground), &gcv);
	/* Set BackGround Color */
	tx->txBackGroundColor.xcPixel = color;
	if (IsDrawableTerritory(tr)) {
	    XFillRectangle(xp->xDisp, tx->txPixmap, tx->txGC,
			   0, 0, tr->teWidth, tr->teHeight);
	}
	if (IsVisibleTerritory(tr) && tx->txWindow != (Window)NULL) {
	    debug_print(8, "Clear Window with color#%lu\n", color);
	    XClearArea(xp->xDisp, tx->txWindow, 0, 0, 0, 0, True);
	    ch->ccXNeedFlush = TRUE;
	}
	if (tr->teTRType == TR_PAGE)
	    clear_page_territory(ch, tr, color);
    }
    debug_endfunc("clear_territory");
}

void draw_background(ch, tr, pattern_tr)
    yy_comm_channel *ch;
    TERRITORY *tr;
    TERRITORY *pattern_tr;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    XGCValues gcv;
    debug_setfunc("territory", "draw_background");
    debug_print(5, "Draw background #%d on #%d\n", pattern_tr->teID, tr->teID);
    gcv.function = GXcopy;
    gcv.tile = XEntry(pattern_tr)->txPixmap;
    XChangeGC(xp->xDisp, tx->txGC, (GCFunction|GCTile), &gcv);
    XFillRectangle(xp->xDisp, tx->txPixmap, tx->txGC,
		   0, 0, tr->teWidth, tr->teHeight);
    if (IsVisibleTerritory(tr)) {
	debug_print(8, "Clear Window with tile\n");
	XClearWindow(xp->xDisp, tx->txWindow);
	ch->ccXNeedFlush = TRUE;
    }
    debug_endfunc("draw_background");
}

#define ABS(i)		((i)>0? (i): (-(i)))

void scroll_territory(ch, tr, x, y, w, h, x2, y2, do_resize)
    yy_comm_channel *ch;
    TERRITORY *tr;
    int x, y;
    int w, h;
    int x2, y2;
    bool do_resize;
{
    int n_w, n_h;
    if (!IsDrawableTerritory(tr) || x <= 0 || y <= 0 || w <= 0 || h <= 0)
	return;
    /* New Origin */
    n_w = w + ABS(x);
    n_h = h + ABS(y);
    
}
    





/************************************************************************
 ******
 ***********************************************************************/

static void wait_for_mapping(disp, win)
    Display *disp;
    Window win;
{
    XEvent event;
    XWindowEvent(disp, win, ExposureMask, &event);
}

/************************************************************************
 ******
 ***********************************************************************/

static void set_default_foreground(xp, cp)
    x_private *xp;
    x_color_entry *cp;
{
    cp->xcUsePixel = TRUE;
    cp->xcPixel = BlackPixel(xp->xDisp, xp->xScreen);
    cp->xcPixmap = (Pixmap)NULL;
}

static void set_default_background(xp, cp)
    x_private *xp;
    x_color_entry *cp;
{
    cp->xcUsePixel = TRUE;
    cp->xcPixel = WhitePixel(xp->xDisp, xp->xScreen);
    cp->xcPixmap = (Pixmap)NULL;
}

u_long set_window_background_attribute(cp, attr)
    x_color_entry *cp;
    XSetWindowAttributes *attr;
{
    if (cp->xcUsePixel) {
	attr->background_pixel = cp->xcPixel;
	return CWBackPixel;
    }
    attr->background_pixmap = cp->xcPixmap;
    return CWBackPixmap;
}

u_long set_window_border_attribute(xp, attr)
    x_private *xp;
    XSetWindowAttributes *attr;
{
    attr->border_pixel = BlackPixel(xp->xDisp, xp->xScreen);
    return CWBorderPixel;
}




/***************************************************************************
 * CREATE TERRITORY
 ***************************************************************************/

TERRITORY *create_territory(ch, x, y, w, h, parent_id,
			    visible, drawable, type, err_code)
    yy_comm_channel *ch;
    int x, y, w, h;
    TRID parent_id;
    bool visible, drawable;
    int type;
    int *err_code;
{
    TERRITORY *TR;
    debug_setfunc("territory", "create_territory");
    /* Allocate New Entry and Set Infomation of Territory */
    if ((TR = territory_alloc()) == (TERRITORY *)NULL) {
	debug_print(1, " can't alloc TERRITORY ENTRY - No enough memory\n");
	debug_endfunc("create_territory");
	*err_code = YYERROR_SYSERR;
	return (TERRITORY *)NULL;
    }
    /* Size of Territory */
    debug_print(1, "Create New Territory SIZE[%d,%d] at (%d,%d), type#0%o\n",
		w, h, x, y);
    TR->teX = x;	TR->teY = y;
    TR->teWidth = w;	TR->teHeight = h;
    TR->teVisible = visible;
    TR->teDrawable = drawable;
    TR->teTRType = type;
    TR->teCtrlFuncs = get_control_functab(type);
    *err_code = (parent_id == 0? create_root_territory(ch, TR):
		 create_normal_territory(ch, TR, parent_id));
    if (*err_code != YYNOERROR) {
	territory_unlink(TR); TR = (TERRITORY *)NULL;
    }
    debug_endfunc("create_territory");
    return TR;
}

static u_long set_attr_for_territory(xp, tr, attr, root)
    x_private *xp;
    register TERRITORY *tr;
    XSetWindowAttributes *attr;
    bool root;
{
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    u_long mask = 0;
    if (root) {
	/* Set Default Cursor */
	tx->txDefCursor = XCreateFontCursor(xp->xDisp, YYDEFAULTCURSORFONT);
	attr->cursor = tx->txDefCursor;
	mask |= CWCursor;
    }
    /* Default ForeGround Color */
    set_default_foreground(xp, &tx->txForeGroundColor);
    /* Default BackGround Color */
    set_default_background(xp, &tx->txBackGroundColor);
    mask |= set_window_background_attribute(&tx->txBackGroundColor, attr);
#ifdef KEISUKEENV
    /* Border */
    if (YYWINDEBUGBORDERWIDTH > 0)
	mask |=set_window_border_attribute(xp, attr);
#endif
    return mask;
}

static void set_default_gc(disp, tx, gc)
    Display *disp;
    TERRITORY_X_ENTRY *tx;
    GC gc;
{
    XGCValues gcv;
    gcv.function = GXcopy;
    gcv.foreground = tx->txForeGroundColor.xcPixel;
    gcv.background = tx->txBackGroundColor.xcPixel;
    XChangeGC(disp, gc, (GCFunction|GCForeground|GCBackground), &gcv);
}


static void set_hints_for_root_territory(xp, tr, win)
    x_private *xp;
    territory_entry *tr;
    Window win;
{
    XSizeHints shint;
    XClassHint chint;
    /* Send hints to Window Manager */
    shint.x = tr->teX; shint.y = tr->teY;
    shint.width = shint.min_width = tr->teWidth;
    shint.height = shint.min_height = tr->teHeight;
    shint.max_width = DisplayWidth(xp->xDisp, xp->xScreen);
    shint.max_height = DisplayHeight(xp->xDisp, xp->xScreen);
    shint.flags = (PPosition|PSize|PMinSize|PMaxSize);
    XSetStandardProperties(xp->xDisp, win, xp->xWindowName, xp->xIconName,
			   NULL, NULL, 0, &shint);
    chint.res_name = YYRESOURCENAME;
    chint.res_class = YYCLASSNAME;
    XSetClassHint(xp->xDisp, win, &chint);
}

static int create_root_territory(ch, tr)
    yy_comm_channel *ch;
    TERRITORY *tr;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    Window w;
    GC gc;
    XSetWindowAttributes attr;
    u_long mask;
    debug_setfunc("territory", "create_root_territory");
    if (selected_territory() != (TERRITORY *)NULL) {
	/* We alread have another Root Territory */
	debug_print(9, "Duplicated ROOT\n");
	debug_endfunc("create_root_territory");
	return YYERROR_DUPROOT;
    }
    /* Create new window as a Root Window of YY */
    mask = set_attr_for_territory(xp, tr, &attr, TRUE);
    debug_print(5, "Create Window for ROOT Territory#%d at (%d,%d)[%d,%d]\n",
		tr->teID, tr->teX, tr->teY, tr->teWidth, tr->teHeight);
    w = XCreateWindow(xp->xDisp, DefaultRootWindow(xp->xDisp),
		      tr->teX, tr->teY, tr->teWidth, tr->teHeight,
		      YYWINDEFAULTBORDERWIDTH,
		      DefaultDepth(xp->xDisp, xp->xScreen), InputOutput,
		      DefaultVisual(xp->xDisp, xp->xScreen), mask, &attr);
    /* Set GC for root territory */
    gc = XCreateGC(xp->xDisp, w, 0, 0);
    XCopyGC(xp->xDisp, DefaultGC(xp->xDisp, xp->xScreen), ~0L, gc);
    (void) set_default_gc(xp->xDisp, tx, gc);
    /* Set Hints */
    set_hints_for_root_territory(xp, tr, w);
    /* Map YY Root Window */
    XSelectInput(xp->xDisp, w, (VisibilityChangeMask|ExposureMask));
    if (tr->teVisible) {
	debug_print(5, "Map Root Window on the Screen\n");
	XMapWindow(xp->xDisp, w);
	wait_for_mapping(xp->xDisp, w);
    }
    XSelectInput(xp->xDisp, w, (ExposureMask|KeyPressMask));
    tx->txWindow = w;
    tx->txPixmap = (Pixmap)NULL;
    tx->txGC = gc;
    debug_endfunc("create_root_territory");
    return YYNOERROR;
}

int create_cursor_territory(xp, tr, w, gc, pix)
    x_private *xp;
    TERRITORY *tr;
    Window w;
    GC gc;
    Pixmap pix;
{
    debug_setfunc("territory", "create_cursor_territory");
    debug_endfunc("create_cursor_territory");
    return YYNOERROR;
}

int create_page_territory(xp, tr, w, gc, pix)
    x_private *xp;
    TERRITORY *tr;
    Window w;
    GC gc;
    Pixmap pix;
{
    debug_setfunc("territory", "create_page_territory");
    debug_endfunc("create_page_territory");
    return YYNOERROR;
}

int create_viewport_territory(xp, tr, w, gc, pix)
    x_private *xp;
    TERRITORY *tr;
    Window w;
    GC gc;
    Pixmap pix;
{
    debug_setfunc("territory", "create_viewport_territory");
    debug_endfunc("create_viewport_territory");
    return YYNOERROR;
}

static int create_normal_territory(ch, tr, parent_id)
    yy_comm_channel *ch;
    TERRITORY *tr;
    TRID parent_id;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY *parent;
    TERRITORY_X_ENTRY *tx = GetTerritoryXEntry(tr);
    Window w = (Window)NULL;
    GC gc = (GC)NULL;
    Pixmap p = (Pixmap)NULL;

    debug_setfunc("territory", "create_normal_territory");
    if ((parent = search_territory(parent_id)) == (TERRITORY *)NULL ||
	XEntry(parent)->txWindow == (Window)NULL ||
	XEntry(parent)->txGC == (GC)NULL)
	return YYERROR_NOTERRITORY;
    switch (tr->teTRType) {
    case TR_NONE:
    case TR_PAGE:
    case TR_VIEWPORT:
    case TR_CURSOR:
    default: {
	TERRITORY_X_ENTRY *ptx = GetTerritoryXEntry(parent);
	XSetWindowAttributes attr;
	u_long mask;
	mask = set_attr_for_territory(xp, tr, &attr, FALSE);
	w = XCreateWindow(xp->xDisp, ptx->txWindow, tr->teX, tr->teY,
			  tr->teWidth, tr->teHeight, YYWINDEBUGBORDERWIDTH, 0,
			  InputOutput, CopyFromParent, mask, &attr);
	gc = XCreateGC(xp->xDisp, w, 0, 0);
	XCopyGC(xp->xDisp, ptx->txGC, ~0L, gc);
	(void) set_default_gc(xp->xDisp, tx, gc);
	if (tr->teTRType == TR_PAGE) {
	    register int ret;
	    p = XCreatePixmap(xp->xDisp, w, tr->teWidth, tr->teHeight,
			      DefaultDepth(xp->xDisp, xp->xScreen));
	    if ((ret = create_page_territory(xp, tr, w, gc, p)) != YYNOERROR)
		return ret;
	} else if (tr->teTRType == TR_CURSOR) {
	    register int ret;
	    p = XCreatePixmap(xp->xDisp, w, tr->teWidth, tr->teHeight,
			      DefaultDepth(xp->xDisp, xp->xScreen));
	    if ((ret = create_cursor_territory(xp, tr, w, gc, p)) != YYNOERROR)
		return ret;
	    tr->teVisible = FALSE;
	} else if (tr->teTRType == TR_VIEWPORT) {
	    register int ret;
	    p = XCreatePixmap(xp->xDisp, w, tr->teWidth, tr->teHeight,
			      DefaultDepth(xp->xDisp, xp->xScreen));
	    if ((ret = create_viewport_territory(xp, tr, w, gc, p)) != YYNOERROR)
		return ret;
	}
	break;
    }
    case TR_FENCE: {
	TERRITORY_X_ENTRY *ptx = GetTerritoryXEntry(parent);
	/* Fence mode Territory */
	w = XCreateWindow(xp->xDisp, ptx->txWindow,
			  tr->teX, tr->teY, tr->teWidth, tr->teHeight,
			  0, 0, InputOnly, CopyFromParent, NULL, NULL);
	/* Fence has no Pixmap and GC */
	p = (Pixmap)NULL;
	gc = (GC)NULL;
	break;
    }
    case TR_SPLITE:
	w = (Window)NULL;
	p = (Pixmap)NULL;
	gc = (GC)NULL;
	break;
    }
    tx->txWindow = w;
    tx->txPixmap = p;
    tx->txGC = gc;
    set_default_foreground(xp, &tx->txForeGroundColor);
    set_default_background(xp, &tx->txBackGroundColor);
    /* Link Info. */
    tr->teParent = parent;
    tr->teNextGen = parent->teChildren;
    if (parent->teChildren != (TERRITORY *)NULL)
	parent->teChildren->tePrevGen = tr;
    parent->teChildren = tr;
    /* Clear Territory */
    if (gc != (GC)NULL)
	clear_territory(ch, tr, tx->txBackGroundColor.xcPixel);
    /* Basic Event Handling */
    if (IsDrawableTerritory(tr))
	XSelectInput(xp->xDisp, w,
		     (ExposureMask|VisibilityChangeMask|ResizeRedirectMask));
    if (IsVisibleTerritory(tr)) {
	debug_print(5, "Map Window on the Screen\n");
	XMapWindow(xp->xDisp, w);
	ch->ccXNeedFlush = TRUE;
    }
    debug_endfunc("create_normal_territory");
    return YYNOERROR;
}

static TERRITORY
    *CurSelectedTerritory = (TERRITORY *)NULL;
static TERRITORY
    *TerritoryTop = (TERRITORY *)NULL;
static TERRITORY
    *TerritoryTail = (TERRITORY *)NULL;
static int TerritoryID = 0;

static TERRITORY *territory_alloc()
{
    register territory_entry *new;
    if ((new = (TERRITORY *)memALLOC(sizeof(TERRITORY))) == (TERRITORY *)NULL)
	return (TERRITORY *)NULL;
    bzero((char *)new, sizeof(TERRITORY));
    if (TerritoryTail != (territory_entry *)NULL) {
	TerritoryTail->teNext = new;
	new->tePrev = TerritoryTail;
	TerritoryTail = new;
    } else {
	TerritoryTop = TerritoryTail = new;
    }
    if (TerritoryID++ > 30000)
	TerritoryID = 1;
    new->teID = TerritoryID;
    return new;
}

static void territory_free(tr)
    register TERRITORY *tr;
{
    if (tr != (territory_entry *)NULL)
	free((char *)tr);
}

static void territory_unlink(tr)
    register TERRITORY *tr;
{
    /* If Specified Territory is selected.. */
    if (tr == CurSelectedTerritory)
	CurSelectedTerritory = tr->teParent;
    /* Link Info in TerritoryList */
    if (tr->tePrev != (TERRITORY *)NULL)
	tr->tePrev->teNext = tr->teNext;
    else
	TerritoryTop = tr->teNext;
    if (tr->teNext != (TERRITORY *)NULL)
	tr->teNext->tePrev = tr->tePrev;
    else
	TerritoryTail = tr->tePrev;
    /* Link Info in Generation List */
    if (tr->tePrevGen != (territory_entry *)NULL)
	tr->tePrevGen->teNextGen = tr->teNextGen;
    else if (tr->teParent != (territory_entry *)NULL)
	tr->teParent->teChildren = tr->teNextGen;
    if (tr->teNextGen != (territory_entry *)NULL)
	tr->teNextGen->tePrevGen = tr->tePrevGen;
    territory_free(tr);
}


territory_entry *change_selected_territory(id)
    register int id;
{
    register territory_entry *tr = search_territory(id);
    if (tr == (territory_entry *)NULL)
	return (territory_entry *)NULL;
    CurSelectedTerritory = tr;
    return tr;
}

territory_entry *selected_territory()
{
    return CurSelectedTerritory;
}

territory_entry *search_territory(id)
    register int id;
{
    register territory_entry *tr = TerritoryTop;
    for (tr = TerritoryTop;tr != (territory_entry *)NULL;tr = tr->teNext)
	if (tr->teID == id)
	    return tr;
    return (territory_entry *)NULL;
}

territory_entry *search_territory_from_win(w)
    Window w;
{
    register territory_entry *tr = TerritoryTop;
    for (tr = TerritoryTop;tr != (territory_entry *)NULL;tr = tr->teNext)
	if (XEntry(tr)->txWindow == w)
	    return tr;
    return (territory_entry *)NULL;
}

/***************************************************************************
 *** For Debbugging
 **************************************************************************/

static char *TerritoryType(tr)
    TERRITORY *tr;
{
    char *s;
    switch (tr->teTRType) {
    case TR_NONE:	s = "Normal";	break;
    case TR_PAGE:	s = "Page";	break;
    case TR_VIEWPORT:	s = "ViewPort";	break;
    case TR_FENCE:	s = "Fence";	break;
    case TR_SPLITE:	s = "Splite";	break;
    case TR_CURSOR:	s = "Cursor";	break;
    default:		s = "UNKNOWN";	break;
    }
    return s;
}

#define GETID(tr)	((tr)!=(TERRITORY *)NULL? (tr)->teID: 0)

static void list_territory_for_debug()
{
    territory_entry *tr;
    debug_setfunc("territory", "list_territory_for_debug");
    debug_print(1, "********** LIST OF TERRITORIES ******************\n");
    debug_print(1, "Cuurent ID = %d\n", TerritoryID);
    debug_print(1, "Cuurent Selected Territory = %d\n",
		(CurSelectedTerritory != (territory_entry *)NULL?
		 CurSelectedTerritory->teID: 0));
    debug_print(1, "********** TERRITORIES ******************\n");
    for (tr = TerritoryTop;tr != (territory_entry *)NULL;tr = tr->teNext){
	debug_print(1, "ID#%3d <%s> (%d,%d)[%d,%d]\n",
		    GETID(tr), TerritoryType(tr),
		    tr->teX, tr->teY, tr->teWidth, tr->teHeight);
	debug_print(1, "\tParent#%3d, Child#%3d, Next#%3d, Prev#%3d\n",
		    GETID(tr->teParent), GETID(tr->teChildren),
		    GETID(tr->teNextGen), GETID(tr->tePrevGen));
    }
    debug_endfunc("list_territory_for_debug");
}

yy_packet *yycom_debug_list_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    list_territory_for_debug();
    return (yy_packet *)NULL;
}

/**********************************************************************
 ****  Cursor Operation ***********************************************
 *********************************************************************/

/*
 * Cursor Control
 */

yy_packet *yycom_create_cursor_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    TERRITORY *tr;	/* Cursor Territory */
    TERRITORY *ptr;	/* Parent Terriotry for Cursor */
    TERRITORY *ctr;	/* Terriotry which has Cursor Pattern */
    yy_packet *repl;
    XGCValues gcv;
    int loop;
    u_long mask;

    debug_setfunc("cursor", "yycom_create_cursor_territory");
#ifdef NOTYET
    /* Allocate New Entry and Set Infomation of Territory */
    tr = (territory_entry *)territory_alloc();
    tr->teX = 0;
    tr->teY = 0;
    tr->teWidth = read_packet_entry_integer(pkt);
    tr->teHeight = read_packet_entry_integer(pkt);
    /* Parent Territory */
    ptr = tr->teParent = get_specified_territory(pkt, &repl);
    if (ptr == (TERRITORY *)NULL) {
	territory_unlink(tr);
	return repl;
    }
    tr->teVisible = FALSE; /* Cursor Terrioty is not visible */
    tr->teDrawable = TRUE; /* Cursor Territory has Pixmap */
    debug_print(5, " Create Cursor Territory (Parent#%d)\n", ptr->teID);
    
    /* Create Default Window */
    tr->teWindow = create_window_for_territory(xp, tr);
    if (tr->teWindow == (Window)NULL) {
	/* We can't get Window for this territory */
	territory_unlink(tr);
	return create_error_packet(pkt, YYERROR_NOTERRITORY);
    }
    /* Get Pattern Terriotry and Copy its Pattern */
    ctr = search_territory(read_packet_entry_integer(pkt));
    if (ctr != (territory_entry *)NULL && ctr->teDrawable) {
	XCopyArea(xp->xDisp, ctr->tePixmap, tr->tePixmap, tr->teGC,
		  0, 0, MIN(tr->teWidth, ctr->teWidth),
		  MIN(tr->teHeight, ctr->teHeight), 0, 0);
    }
    /* Cursor Territory has another Pixmap for Cursor Mask */
    tr->teCursorPat = XCreatePixmap(xp->xDisp, tr->teWindow,
				    tr->teWidth, tr->teHeight, 1);
    tr->teCursorMask = XCreatePixmap(xp->xDisp, tr->teWindow,
				     tr->teWidth, tr->teHeight, 1);
    gcv.function = GXcopy;
    gcv.foreground = tr->teBackGroundColor;
    XChangeGC(xp->xDisp, tr->teGC, (GCFunction|GCForeground), &gcv);
    XFillRectangle(xp->xDisp, tr->teCursorPat, tr->teGC,
		   0, 0, tr->teWidth, tr->teHeight);
    gcv.function = GXor;
    XChangeGC(xp->xDisp, tr->teGC, GCFunction, &gcv);
    for (loop = DefaultDepth(xp->xDisp, xp->xScreen), mask = 1;
	 loop > 0; loop--, mask <<= 1) {
	XCopyPlane(xp->xDisp, tr->tePixmap, tr->teCursorPat, tr->teGC,
		   0, 0, tr->teWidth, tr->teHeight, 0, 0, mask);
    }
    /* Setup Link Info */
    if (!NULLTERRITORY(tr->teNextGen = ptr->teChildren))
	ptr->teChildren->tePrevGen = tr;
    ptr->teChildren = tr;
    /* */
    repl = ALLOC_ACKPACKET(pkt);
    append_packet_entry_integer(repl, tr->teID);
#endif NOTYET
    debug_endfunc("yycom_create_cursor_territory");
    return repl;
}

yy_packet *yycom_move_cursor_hotspot(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *err;
    int x, y;

    debug_setfunc("cursor", "yycom_move_cursor_hotspot");
    tr = get_specified_territory(pkt, &err);
    if (tr == (territory_entry *)NULL)
	return err;

    debug_endfunc("yycom_move_cursor_hotspot");
    return (yy_packet *)NULL;
}

yy_packet *yycom_change_cursor_bitmap(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *err;
    yy_packet *repl;
    debug_setfunc("cursor", "yycom_change_cursor_bitmap");

    debug_endfunc("yycom_change_cursor_bitmap");
    return (yy_packet *)NULL;
}

yy_packet *yycom_move_cursor(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *err;
    yy_packet *repl;
    debug_setfunc("cursor", "yycom_move_cursor");

    debug_endfunc("yycom_move_cursor");
    return (yy_packet *)NULL;
}

yy_packet *yycom_destroy_cursor(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *err;
    yy_packet *repl;
    debug_setfunc("cursor", "yycom_destroy_cursor");

    debug_endfunc("yycom_destroy_cursor");
    return (yy_packet *)NULL;
}

static int get_current_cursor_position(xp, tr_x, tr_y, mask)
    x_private *xp;
    int *tr_x, *tr_y;
    u_int *mask;
{
    int id = 0;
    Window top = XEntry(TerritoryTop)->txWindow;
    Window r_win, c_win;
    int r_x, r_y, w_x, w_y;
    u_int x_mask;
    debug_setfunc("territory", "get_current_cursor_position");
    *mask = 0;
    if (XQueryPointer(xp->xDisp, top, &r_win, &c_win,
		      &r_x, &r_y, &w_x, &w_y, &x_mask) == True) {
	if (c_win != None) {
	    Window win_dummy;
	    TERRITORY *tr = search_territory_from_win(c_win);
	    id = tr->teID;
	    XTranslateCoordinates(xp->xDisp, top, c_win, w_x, w_y,
				  tr_x, tr_y, &win_dummy);
	} else {
	    TERRITORY *tr = search_territory_from_win(top);
	    if (w_x >= 0 && w_x <= tr->teWidth &&
		w_y >= 0 && w_y <= tr->teHeight) {
		id = tr->teID;
		*tr_x = w_x; *tr_y = w_y;
	    } else
		id = *tr_x = *tr_y = 0;
	}
	debug_print(2, "QueryPointer...(%d,%d) on TR#%d\n", *tr_x, *tr_y, id);
    } else {
	/* Cursor is not in this screen */
	*tr_x = *tr_y = *mask = 0;
	id = 0;
    }
    if (id != 0) {
	/* Parse x_mask */
	*mask = parse_button_state(YYMASK_ALL, YYMASK_BUTTON_PRESS, x_mask);
    }
    debug_endfunc("get_current_cursor_position");
    return id;
}


yy_packet *yycom_get_cursur_position(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    int cs_id;		/* TerritoryID for Cursor Territory */
    int id;		/* The cursor is in this territory */
    int tr_x, tr_y;	/* Position of cursor */
    u_int mask;
    yy_packet *repl = (yy_packet *)NULL;

    debug_setfunc("cursor", "yycom_get_cursur_position");
    if ((cs_id = read_packet_entry_integer(pkt)) == 0) {
	/* Get Position for Current Cursor */
	id = get_current_cursor_position(xp, &tr_x, &tr_y, &mask);
    } else {
	TERRITORY *tr = search_territory(cs_id);
	/* if (TypeOfTerritory(tr) != TR_CURSOR) */
	id = 0;
	tr_x = tr_y = 0;
	mask = 9;
    }
    repl = ALLOC_ACKPACKET(pkt);
    append_packet_entry_integer(repl, id);
    append_packet_entry_integer(repl, mask);
    append_packet_entry_integer(repl, tr_x);
    append_packet_entry_integer(repl, tr_y);
    debug_endfunc("yycom_get_cursur_position");
    return (yy_packet *)repl;
}


yy_packet *yycom_display_cursor(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    territory_entry *rtr;	/* Root Territory */
    yy_packet *err;
    int x, y;
    
    debug_setfunc("cursor", "yycom_display_cursor");
    tr = get_specified_territory(pkt, &err);
    if (tr == (territory_entry *)NULL)
	return err;
#ifdef NOTYET
    if (tr->teCursorMask == (Pixmap)NULL) {
	/* This territory is not cursor */
	return create_error_packet(pkt, YYERROR_NOTERRITORY);
    }
    /* Search Root Territory */
    for (rtr = tr; rtr->teParent != (TERRITORY *)NULL; )
	rtr = rtr->teParent;
#endif NOTYET
    debug_endfunc("yycom_display_cursor");
    return (yy_packet *)NULL;
}

/**********************************************************************
 ****  Event Control ***********************************************
 *********************************************************************/

/*
 * Event Control
 */

yy_packet *yycom_select_territory(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *err;
    debug_setfunc("event", "yycom_select_territory");
    tr = get_specified_territory(pkt, &err);
    if (tr == (territory_entry *)NULL)
	return err; /* Unknow Territory */
    change_selected_territory(tr->teID);
    debug_endfunc("yycom_select_territory");
    return (yy_packet *)NULL;
}

yy_packet *yycom_set_event_mask(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    TERRITORY_X_ENTRY *tx;
    register int mask;
    register u_long event_mask;
    yy_packet *err;
    
    debug_setfunc("event", "yycom_set_event_mask");
    if ((tr = catch_territory(pkt, TR_ALL, &err)) == (TERRITORY *)NULL)
	return err;
    mask = read_packet_entry_integer(pkt);
    tx = GetTerritoryXEntry(tr);
    
    /* Set Default Event Mask */
    event_mask = ExposureMask;
    if (tr->teParent == (TERRITORY *)NULL)
	event_mask |= KeyPressMask;
    if (mask & YYMASK_BUTTON_PRESS)
	event_mask |= ButtonPressMask;
    if (mask & YYMASK_BUTTON_RELEASE)
	event_mask |= ButtonReleaseMask;
    if (mask & YYMASK_MOVE)
	event_mask |= PointerMotionMask;
    if (mask & YYMASK_IN)
	event_mask |= EnterWindowMask;
    if (mask & YYMASK_OUT)
	event_mask |= LeaveWindowMask;
    if (mask & YYMASK_STAY)
	event_mask |= (EnterWindowMask|LeaveWindowMask);
    if (mask & YYMASK_DOUBLE_CLICK)
	event_mask |= (ButtonPressMask|ButtonReleaseMask);
    XSelectInput(xp->xDisp, tx->txWindow, event_mask);
    debug_print(1, "SetEventMask: Mask=0%o\n", mask);
    tr->teEventMask = mask;
    if (IsVisibleTerritory(tr))
	ch->ccXNeedFlush = TRUE;
    debug_endfunc("yycom_set_event_mask");
    return (yy_packet *)NULL;
}


yy_packet *yycom_mask_event(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    x_private *xp = XPRIVATE(ch);
    territory_entry *tr;
    yy_packet *err;
    
    debug_setfunc("event", "yycom_mask_event");

    debug_endfunc("yycom_mask_event");
    return (yy_packet *)NULL;
}
