#ifndef lint
static char *RcsId="$Id: com_font.c,v 2.16 91/03/14 12:42:55 keisuke Exp $";
#endif

/* Handling Fonts on YY-server
 * This file is part of YY-server of YYonX (1.3 Distribution)
 * $Id: com_font.c,v 2.16 91/03/14 12:42:55 keisuke Exp $
 */

/****************************************************************************
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   Version 1.0 90/02/28 by Keisuke 'Keiko' Tanaka
;;;				(keisuke@csrl.aoyama.ac.jp)
;;;   Version 2.0 90/08/27 by Keisuke 'Keiko' Tanaka
;;;			Page Mode Territory is supported
;;;   Version 2.7 90/11/05 by Keisuke 'Keiko' Tanaka
;;;			Copyright Notice is rewritten
;;;
****************************************************************************/

/****************************************************************************
  $Revision: 2.16 $ Written by Keisuke 'Keiko' Tanaka
  $Date: 91/03/14 12:42:55 $
****************************************************************************/

#include <stdio.h>
#include <math.h>
#include <sys/types.h>
#include "yydefs.h"
#include "yypacket.h"
#include "xwindow.h"
#include "territory.h"
#include "yyfont.h"

#define XFSTRUCT(xfont)		((xfont)->fxFontStruct)

/*
 * Drawing Characters on Territory
 */

YYFONT *call_font(ch, id)
    yy_comm_channel *ch;
    int id;
{
    yy_font_table *cur = ch->ccFontTable;
    for ( ; cur != (yy_font_table *)NULL; cur = cur->yfNext)
	if (cur->yfFontInfo.fiFID == id && cur->yfFontInfo.fiLoaded)
	    return &cur->yfFontInfo;
    return (YYFONT *)NULL;
}

/***************************************************************************
 ***** Command 10
 *****
 *****  Load Character Font
 *****
 *****  Functions:
 *****		static YYFONT *search_font(ch, name)
 *****		static bool load_x_font(xp, xfont)
 *****		static void dump_font_info(pkt, xfont, dump_all)
 *****		static void set_yy_char_info(ci, xci)
 ***************************************************************************/

static YYFONT *search_font(ch, name)
    yy_comm_channel *ch;
    char *name;
{
    register yy_font_table *cur = ch->ccFontTable;
    for ( ; cur != (yy_font_table *)NULL; cur = cur->yfNext)
	if (strSAME(cur->yfFontInfo.fiFontName, name))
	    return &cur->yfFontInfo;
    return (YYFONT *)NULL;
}

static bool load_x_font(xp, xfont)
    x_private *xp;
    YYXFONT *xfont;
{
    debug_setfunc("font", "load_x_font");
    if (xfont->fxFontName == (char *)NULL)
	return FALSE;
    if (xfont->fxFontStruct == (XFontStruct *)NULL) {
	xfont->fxFontStruct = XLoadQueryFont(xp->xDisp, xfont->fxFontName);
	if (xfont->fxFontStruct == (XFontStruct *)NULL) {
	    warning("%s: can't load font\n", xfont->fxFontName);
	    return FALSE;
	}
	xfont->fxFontID = xfont->fxFontStruct->fid;
    }
    debug_endfunc("load_x_font");
    return TRUE;
}

static void dump_font_info(pkt, xfont, dump_all)
    yy_packet *pkt;
    YYXFONT *xfont;
    bool dump_all;
{
    XFontStruct *fs = xfont->fxFontStruct;
    if (!dump_all) {
	/* Width, Height and BaseLine */
	if (fs == (XFontStruct *)NULL) {
	    append_packet_entry_integer(pkt, 0);
	    append_packet_entry_integer(pkt, 0);
	    append_packet_entry_integer(pkt, 0);
	} else {
	    append_packet_entry_integer(pkt, fs->max_bounds.width);
	    append_packet_entry_integer(pkt, fs->ascent + fs->descent);
	    append_packet_entry_integer(pkt, fs->ascent);
	}
	return;
    }
    /* DUMP ALL - Char Code, Width, Height and BaseLine */
    if (fs == (XFontStruct *)NULL) {
	register int c, l;
	for (c = 0, l= 256; l > 0; c++, l--) {
	    /* code, width, height, and baseline */
	    append_packet_entry_onebyte(pkt, (byte)c);
	    append_packet_entry_onebyte(pkt, (byte)0);
	    append_packet_entry_onebyte(pkt, (byte)0);
	    append_packet_entry_onebyte(pkt, (byte)0);
	}
    } else if (fs->per_char != (XCharStruct *)NULL) {
	register XCharStruct *pc = fs->per_char;
	register int c, l;
	for (c = 0, l = 256; l > 0; c++, pc++, l--) {
	    append_packet_entry_onebyte(pkt, (byte)c);
	    /* width, height, and baseline */
	    append_packet_entry_onebyte(pkt, (byte)(pc->width));
	    append_packet_entry_onebyte(pkt, (byte)(pc->ascent+pc->descent));
	    append_packet_entry_onebyte(pkt, (byte)pc->ascent);
	}
    } else {
	register int w = fs->max_bounds.width;
	register int h = fs->ascent + fs->descent;
	register int b = fs->ascent;
	register int c, l;
	for (c = 0, l= 256; l > 0; c++, l--) {
	    /* code, width, height and baseline */
	    append_packet_entry_onebyte(pkt, (byte)c);
	    append_packet_entry_onebyte(pkt, (byte)w);
	    append_packet_entry_onebyte(pkt, (byte)h);
	    append_packet_entry_onebyte(pkt, (byte)b);
	}
    }
    return;
}


static void set_yy_char_info(ci, xci)
    YYCHAR *ci;
    XCharStruct *xci;
{
    ci->ciAscent = xci->ascent;
    ci->ciDescent = xci->descent;
    ci->ciLBearing = xci->lbearing;
    ci->ciRBearing = xci->rbearing;
    ci->ciHeight = ci->ciAscent + ci->ciDescent;
    ci->ciWidth = xci->width;
}

yy_packet *yycom_load_font(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    char font_name[256];
    int leng;
    yy_packet *repl;
    x_private *xp = XPRIVATE(ch);
    YYFONT *yfont;

    debug_setfunc("font", "yycom_load_font");
    leng = read_packet_entry_integer(pkt);
    read_packet_entry_string(pkt, leng, font_name);
    debug_print(1, "Load YY Font '%s'\n", font_name);
    if ((yfont = search_font(ch, font_name)) == (YYFONT *)NULL) {
	repl = ALLOC_NACKPACKET(pkt); append_packet_entry_integer(repl, 0);
	goto send_reply;
    }
    /* Load X Font fot Alphabet */
    if (load_x_font(XPRIVATE(ch), yfont->fiSingle)) {
	(void) set_yy_char_info(&yfont->fiSingle->fxMaxBound,
				&XFSTRUCT(yfont->fiSingle)->max_bounds);
	(void) set_yy_char_info(&yfont->fiSingle->fxMinBound,
				&XFSTRUCT(yfont->fiSingle)->min_bounds);
    } else {
	/* No such Font */
	debug_print(1, "can't load Font '%s'\n", yfont->fiSingle->fxFontName);
	repl = ALLOC_NACKPACKET(pkt); append_packet_entry_integer(repl, 0);
	goto send_reply;
    }
    /* Load X Font for KANJI */
    if (load_x_font(XPRIVATE(ch), yfont->fiDouble)) {
	(void) set_yy_char_info(&yfont->fiDouble->fxMaxBound,
				&XFSTRUCT(yfont->fiDouble)->max_bounds);
	(void) set_yy_char_info(&yfont->fiDouble->fxMinBound,
				&XFSTRUCT(yfont->fiDouble)->min_bounds);
    } else {
	/* No such Font - ignore it */
	bzero((char *)&yfont->fiDouble->fxMaxBound, sizeof(YYCHAR));
	bzero((char *)&yfont->fiDouble->fxMinBound, sizeof(YYCHAR));
    }

    repl = ALLOC_ACKPACKET(pkt);
    append_packet_entry_integer(repl, yfont->fiFID);
    dump_font_info(repl, yfont->fiDouble, FALSE);
    dump_font_info(repl, yfont->fiSingle, TRUE);
    yfont->fiLoaded = TRUE;

 send_reply:
    debug_endfunc("yycom_load_font");
    return repl;
}



#define KCODE(c)		((c)&0200)


#define DRAW_VISIBLE(ctrl)	((ctrl)->tr->teVisible)
#define DRAW_DISP(ctrl)		((ctrl)->xp->xDisp)
#define DRAW_SCR(ctrl)		((ctrl)->xp->xScreen)
#define DRAW_WIN(ctrl)		((ctrl)->tx->txWindow)
#define DRAW_GC(ctrl)		((ctrl)->tx->txGC)
#define DRAW_PIX(ctrl)		((ctrl)->tx->txPixmap)
#define DRAW_WIDTH(ctrl)	((ctrl)->tr->teWidth)
#define DRAW_HEIGHT(ctrl)	((ctrl)->tr->teHeight)

#define DRAW_B1XFONT(ctrl)	((ctrl)->font->fiSingle)
#define DRAW_B2XFONT(ctrl)	((ctrl)->font->fiDouble)

typedef struct {
    x_private *xp;
    TERRITORY *tr;
    TERRITORY_X_ENTRY *tx;
    int top_x, top_y;
    int x, y;
    YYFONT *font;
    double xscale;
    double yscale;
} draw_text_control;

static void put_b1text(ctrl, str, wlen)
    register draw_text_control *ctrl;
    u_char *str;
    int wlen;
{
    XTextItem b1item;
    int width;
    debug_setfunc("font", "put_b1text");
    debug_print(5, "length = %d\n", wlen);
    b1item.delta = 0;
    b1item.font = DRAW_B1XFONT(ctrl)->fxFontID;
    b1item.chars = (char *)str;
    b1item.nchars = wlen;

    width = XTextWidth(DRAW_B1XFONT(ctrl)->fxFontStruct, str, wlen);
    debug_print(5, "Put %d Chars on (%d,%d) (W:%d)\n",
		wlen, ctrl->x, ctrl->y, width);
    XDrawText(DRAW_DISP(ctrl), DRAW_PIX(ctrl), DRAW_GC(ctrl),
	      ctrl->x, ctrl->y, &b1item, 1);
    if (DRAW_VISIBLE(ctrl))
	XDrawText(DRAW_DISP(ctrl), DRAW_WIN(ctrl), DRAW_GC(ctrl),
		  ctrl->x, ctrl->y, &b1item, 1);
    ctrl->x += width;
    debug_endfunc("put_b1text");
}

static void put_b2text(ctrl, str, wlen)
    register draw_text_control *ctrl;
    u_char *str;
    int wlen;
{
    XTextItem16 b2item;
    int width;
    debug_setfunc("font", "put_b2text");
    debug_print(5, "length = %d\n", wlen);
    b2item.delta = 0;
    b2item.font = DRAW_B2XFONT(ctrl)->fxFontID;
    b2item.chars = (XChar2b *)str;
    b2item.nchars = wlen;

    width = XTextWidth16(DRAW_B2XFONT(ctrl)->fxFontStruct, str, wlen);
    debug_print(5, "Put %d Kanjis(%s) on (%d,%d) (W:%d)\n",
		wlen, str, ctrl->x, ctrl->y, width);
    XDrawText16(DRAW_DISP(ctrl), DRAW_PIX(ctrl), DRAW_GC(ctrl),
		ctrl->x, ctrl->y, &b2item, 1);
    if (DRAW_VISIBLE(ctrl))
	XDrawText16(DRAW_DISP(ctrl), DRAW_WIN(ctrl), DRAW_GC(ctrl),
		    ctrl->x, ctrl->y, &b2item, 1);
    ctrl->x += width;
    debug_endfunc("put_b2text");
}

/*
 * +--+        +--------+
 * |30|  --->  |012     |
 * |41|  --->  |345     |
 * |52|        +--------+
 * |  |
 * |  |
 * |  |
 * |  |
 * +--+
 */

static void vtoh(src, src_w, src_h, dst)
    XImage *src, *dst;
    int src_w, src_h;
{
    register int src_y;
    debug_setfunc("font", "vtoh");
    debug_print(10, "VtoH [%d,%d]\n", src_w, src_h);
    for (src_y = 0 ; src_h-- > 0; src_y++) {
	register int src_x, dst_y;
	for (src_x = 0, dst_y = src_w - 1; dst_y >= 0; src_x++, dst_y--)
	    XPutPixel(dst, src_y, dst_y, XGetPixel(src, src_x, src_y));
    }
    debug_endfunc("vtoh");
}

/*
 * +-----------+        +--+
 * |012        |  --->  |30|
 * |345        |  --->  |41|
 * +-----------+        |52|
 *                      |  |
 *                      |  |
 *                      |  |
 *                      |  |
 *                      +--+
 */

static void htov(src, src_w, src_h, dst)
    XImage *src, *dst;
    int src_w, src_h;
{
    register int src_x;
    debug_setfunc("font", "htov");
    debug_print(10, "HtoV [%d,%d]\n", src_w, src_h);
    for (src_x = 0; src_w-- > 0; src_x++) {
	register int src_y, dst_x;
	for (src_y = 0, dst_x = src_h - 1; dst_x >= 0; src_y++, dst_x--)
	    XPutPixel(dst, dst_x, src_x, XGetPixel(src, src_x, src_y));
    }
    debug_endfunc("htov");
}

static void rotate_string(ctrl, item, start_x, start_y, string_w, string_h)
    register draw_text_control *ctrl;
    XTextItem *item;
    int start_x, start_y;
    int string_w, string_h;
{
    Pixmap wk_pix;		/* working area */
    GC wk_gc;			/* GC for copy Operation */
    int x_on_win, y_on_win, w_on_win, h_on_win;	/* Area on Territory */
    int x_on_pix, y_on_pix, w_on_pix, h_on_pix;	/* Area on Pixmap */
    XImage *image_on_pix;
    XImage *image_on_tr;

    debug_setfunc("font", "rotate_string");
    debug_print(1, "put Text[W:%d,H:%d] at (%d,%d) on Territory [%d,%d]\n",
		string_w, string_h, start_x, start_y,
		DRAW_WIDTH(ctrl), DRAW_HEIGHT(ctrl));
    /* Get Active Area on the Territory */
    x_on_win = MAX(start_x, 0); y_on_win = MAX(start_y, 0);
    w_on_win = MIN((start_x+string_h), DRAW_WIDTH(ctrl)) - x_on_win;
    h_on_win = MIN((start_y+string_w), DRAW_HEIGHT(ctrl)) - y_on_win;
    debug_print(2, " Working Area: (%d,%d)[%d,%d]\n",
		x_on_win, y_on_win, w_on_win, h_on_win);
    if (w_on_win <= 0 || h_on_win <= 0)
	return; /* Active Area is not in the Territory */

    /* Create Working Pixmap for the string */
    debug_print(2, " Create Pixmap - SIZE:[%d,%d]\n", string_w, string_h);
    wk_pix = XCreatePixmap(DRAW_DISP(ctrl), DRAW_WIN(ctrl), string_w, string_h,
			   DefaultDepth(DRAW_DISP(ctrl), DRAW_SCR(ctrl)));
    wk_gc = XCreateGC(DRAW_DISP(ctrl), DRAW_WIN(ctrl), 0, 0);
    XCopyGC(DRAW_DISP(ctrl), DRAW_GC(ctrl), ~0L, wk_gc);
    XSetFunction(DRAW_DISP(ctrl), wk_gc, GXcopy);
    /* Get Dummy Image on Working Pixmap to allocate XImage Structure - */
    x_on_pix = y_on_win - start_y;
    y_on_pix = string_h - w_on_win - (x_on_win - start_x);
    w_on_pix = h_on_win; h_on_pix = w_on_win;	/* String will be rotated */
    debug_print(3, "  GetDummyImage from Pixmap (%d,%d)[%d,%d]\n",
		/*X*/x_on_pix, /*Y*/y_on_pix, /*W*/w_on_pix, /*H*/h_on_pix);
    image_on_pix = XGetImage(DRAW_DISP(ctrl), wk_pix,
			     /*X*/x_on_pix, /*Y*/y_on_pix,
			     /*W*/w_on_pix, /*H*/h_on_pix, AllPlanes, ZPixmap);
    /* Get Background Image on the Territory */
    debug_print(3, "  GetImage from Territory (%d,%d)[%d,%d]\n",
		/*X*/(x_on_win), /*Y*/(y_on_win),
		/*W*/(w_on_win), /*H*/(h_on_win));
    image_on_tr = XGetImage(DRAW_DISP(ctrl), DRAW_PIX(ctrl),
			    /*X*/(x_on_win), /*Y*/(y_on_win),
			    /*W*/(w_on_win), /*H*/(h_on_win),
			    AllPlanes, ZPixmap);
    /* Rotate Background Imagem, then put it on working Pixmap */
    vtoh(image_on_tr, /*W*/w_on_win, /*H*/h_on_win, image_on_pix);
    XPutImage(DRAW_DISP(ctrl), wk_pix, wk_gc, image_on_pix, 0, 0,
	      /*X*/x_on_pix, /*Y*/y_on_pix, /*W*/w_on_pix, /*H*/h_on_pix);
    /* Draw Text on the Working Pixmap */
    debug_print(5, "Draw TEXT on (%d,%d)\n",
		0, string_h - DRAW_B1XFONT(ctrl)->fxMaxBound.ciDescent);
    XDrawText(DRAW_DISP(ctrl), wk_pix, DRAW_GC(ctrl),
	      0, string_h - DRAW_B1XFONT(ctrl)->fxMaxBound.ciDescent, item, 1);
    /* Get Text Image on the Working Pixmap and Put it on the Territory */
    debug_print(3, "  GetImage from Pixmap (%d,%d)[%d,%d]\n",
		/*X*/x_on_pix, /*Y*/y_on_pix, /*W*/w_on_pix, /*H*/h_on_pix);
    XGetSubImage(DRAW_DISP(ctrl), wk_pix, 
		 /*X*/x_on_pix, /*Y*/y_on_pix, /*W*/w_on_pix, /*H*/h_on_pix,
		 AllPlanes, ZPixmap, image_on_pix, 0, 0);
    htov(image_on_pix, /*W*/(h_on_win), /*H*/(w_on_win), image_on_tr);
    XPutImage(DRAW_DISP(ctrl), DRAW_PIX(ctrl), wk_gc, image_on_tr, 0, 0,
	      /*X*/(x_on_win), /*Y*/(y_on_win),
	      /*W*/(w_on_win), /*H*/(h_on_win));
    if (DRAW_VISIBLE(ctrl))
	XPutImage(DRAW_DISP(ctrl), DRAW_WIN(ctrl), wk_gc, image_on_tr, 0, 0,
		  /*X*/(x_on_win), /*Y*/(y_on_win),
		  /*W*/(w_on_win), /*H*/(h_on_win));

    XFreePixmap(DRAW_DISP(ctrl), wk_pix);
    XFreeGC(DRAW_DISP(ctrl), wk_gc);
    XDestroyImage(image_on_pix);
    XDestroyImage(image_on_tr);
    debug_endfunc("rotate_string");
}

static void put_b1vtext(ctrl, str, wlen)
    register draw_text_control *ctrl;
    u_char *str;
    int wlen;
{
    XTextItem b1item;
    int width;
    int height;
    int offset;
    debug_setfunc("font", "put_b1vtext");
    debug_print(5, "length = %d\n", wlen);
    b1item.delta = 0;
    b1item.font = DRAW_B1XFONT(ctrl)->fxFontID;
    b1item.chars = (char *)str;
    b1item.nchars = wlen;
    width = XTextWidth(DRAW_B1XFONT(ctrl)->fxFontStruct, str, wlen);
    height = DRAW_B1XFONT(ctrl)->fxMaxBound.ciHeight;
    debug_print(5, "Put %d Chars on (%d,%d) (W:%d,H:%d)\n",
		wlen, ctrl->x, ctrl->y, width, height);
    offset = DRAW_B2XFONT(ctrl)->fxMaxBound.ciAscent;
    rotate_string(ctrl, &b1item, ctrl->x, ctrl->y-offset, width, height);
    ctrl->y += width;
    debug_endfunc("put_b1vtext");
}

static void put_b2vtext(ctrl, str, wlen)
    register draw_text_control *ctrl;
    u_char *str;
    int wlen;
{
    XTextItem16 b2item;
    int width, height;
    int x = ctrl->x;
    int y = ctrl->y;
    debug_setfunc("font", "put_b2vtext");
    debug_print(5, "length = %d\n", wlen);
    b2item.delta = 0;
    b2item.font = DRAW_B2XFONT(ctrl)->fxFontID;
    for ( ; wlen > 0; str += 2, wlen--) {
	b2item.chars = (XChar2b *)str;
	b2item.nchars = 1;
	XDrawText16(DRAW_DISP(ctrl), DRAW_PIX(ctrl), DRAW_GC(ctrl),
		    x, y, &b2item, 1);
	if (DRAW_VISIBLE(ctrl))
	    XDrawText16(DRAW_DISP(ctrl), DRAW_WIN(ctrl),
			DRAW_GC(ctrl), x, y, &b2item, 1);
	y += DRAW_B2XFONT(ctrl)->fxMaxBound.ciHeight;
    }
    ctrl->y = y;
    debug_endfunc("put_b2vtext");
}

void put_b1rtext(ctrl, str, wlen)
    register draw_text_control *ctrl;
    u_char *str;
    int wlen;
{
    int x = ctrl->x - ctrl->top_x;
    int y = ctrl->y - ctrl->top_y;
    XTextItem b1item;
    debug_setfunc("font", "put_b1rtext");
    debug_print(5, "length = %d\n", wlen);
    b1item.delta = 0;
    b1item.font = DRAW_B1XFONT(ctrl)->fxFontID;
    for ( ; wlen > 0; str++, wlen--) {
	register int put_x = ctrl->top_x + (int)(ctrl->xscale * x);
	register int put_y = ctrl->top_y + (int)(ctrl->yscale * x);
	b1item.chars = (char *)str;
	b1item.nchars = 1;
	XDrawText(DRAW_DISP(ctrl), DRAW_PIX(ctrl), DRAW_GC(ctrl),
		  put_x, put_y, &b1item, 1);
	if (DRAW_VISIBLE(ctrl))
	    XDrawText(DRAW_DISP(ctrl), DRAW_WIN(ctrl),
		      DRAW_GC(ctrl), put_x, put_y, &b1item, 1);
	x += XTextWidth(DRAW_B1XFONT(ctrl)->fxFontStruct, (char *)str, 1);
    }
    ctrl->x += x;
    debug_endfunc("put_b1rtext");
}

void put_b2rtext(ctrl, str, wlen)
    register draw_text_control *ctrl;
    u_char *str;
    int wlen;
{
    int x = ctrl->x - ctrl->top_x;
    int y = ctrl->y - ctrl->top_y;
    XTextItem16 b2item;
    debug_setfunc("font", "put_b1rtext");
    debug_print(5, "length = %d\n", wlen);
    b2item.delta = 0;
    b2item.font = DRAW_B2XFONT(ctrl)->fxFontID;
    for ( ; wlen > 0; str += 2, wlen--) {
	register int put_x = ctrl->top_x + (int)(ctrl->xscale * x);
	register int put_y = ctrl->top_y + (int)(ctrl->yscale * x);
	b2item.chars = (XChar2b *)str;
	b2item.nchars = 1;
	XDrawText16(DRAW_DISP(ctrl), DRAW_PIX(ctrl), DRAW_GC(ctrl),
		    put_x, put_y, &b2item, 1);
	if (DRAW_VISIBLE(ctrl))
	    XDrawText16(DRAW_DISP(ctrl), DRAW_WIN(ctrl),
			DRAW_GC(ctrl), put_x, put_y, &b2item, 1);
	x += XTextWidth16(DRAW_B2XFONT(ctrl)->fxFontStruct, (XChar2b *)str, 1);
    }
    ctrl->x = ctrl->top_x + x;
    debug_endfunc("put_b2rtext");
}


static yy_packet *draw_text(ch, pkt, ctrl, b1func, b2func)
    yy_comm_channel *ch;
    yy_packet *pkt;
    draw_text_control *ctrl;
    void (*b1func)();
    void (*b2func)();
{
    char str[258];
    char *s;
    int size;
    int leng, rlen;
    int offset = 0;
    debug_setfunc("font", "draw_text");
    /* Get the Length of String */
    leng = read_packet_entry_integer(pkt);
    size = (sizeof(str) & ~3);
    for (s = str; leng > 0; ) {
	char *ws, *we;
	rlen = MIN(leng, (size-offset));
	read_packet_entry_string(pkt, rlen, &str[offset]);
	leng -= rlen;
	if (offset)
	    rlen++;
	/* 'str' has 'rlen+l' chars */
	offset = 0;
	for (ws = str; rlen > 0; ws = we) {
	    if (KCODE(*ws)) {
		register int wlen = 0;
		if (rlen < 2) {
		    offset = 1; str[0] = *ws; break;
		}
		for (we = ws; KCODE(*we) && rlen > 0; ) {
		    *we++ &= 0177; *we++ &= 0177;
		    wlen++, rlen -= 2;
		}
		if (DRAW_B2XFONT(ctrl) != (YYXFONT *)NULL)
		    (*b2func)(ctrl, ws, wlen);
	    } else {
		register int wlen = 0;
		for (we = ws; !KCODE(*we) && rlen > 0; ) {
		    *we++; wlen++; rlen--;
		}
		(*b1func)(ctrl, ws, wlen);
	    }
	}
    }
    if (DRAW_VISIBLE(ctrl))
	ch->ccXNeedFlush = TRUE;
    debug_endfunc("draw_text");
    return (yy_packet *)NULL;
}

static yy_packet *draw_text_init(ch, pkt, ctrl, rotate)
    yy_comm_channel *ch;
    yy_packet *pkt;
    draw_text_control *ctrl;
    bool rotate;
{
    x_private *xp = XPRIVATE(ch);
    YYFONT *yfont;
    yy_packet *err = (yy_packet *)NULL;

    debug_setfunc("font", "draw_text_init");
    ctrl->xp = xp;
    /* Get Terriotry */
    ctrl->tr = catch_territory(pkt, TR_DRAWABLE, &err);
    if (ctrl->tr == (TERRITORY *)NULL)
	goto send_reply;
    ctrl->tx = GetTerritoryXEntry(ctrl->tr);
    /* Start Position and Change GC */
    ctrl->top_x = ctrl->x = read_packet_entry_integer(pkt);
    ctrl->top_y = ctrl->y = read_packet_entry_integer(pkt);
    change_gc(xp->xDisp, DRAW_GC(ctrl), pkt,
	      (YYGC_OPERATION|YYGC_COLOR), (YYGC *)NULL, xp);
    debug_print(5, " Draw text on TR#%d(%d,%d)\n",
		ctrl->tr->teID, ctrl->x, ctrl->y);
    /* Get Font Info */
    ctrl->font = call_font(ch, read_packet_entry_integer(pkt));
    if (ctrl->font == (YYFONT *)NULL) {
	ctrl->tr = (TERRITORY *)NULL;
	debug_print(5, " Font is not availabel\n");
	goto send_reply;
    }
    if (rotate) {
	register int xscale, yscale, angle;
	double real_angle;
	xscale = read_packet_entry_integer(pkt);
	yscale = read_packet_entry_integer(pkt);
	angle = (read_packet_entry_integer(pkt) % YYPI2);
	if (angle > 0) {
	    yscale *= (-1);
	} else {
	    angle *= (-1);
	}
	if (angle >= YYPI3_2) {
	    real_angle = (YYPI2 - angle);
	    yscale *= (-1);
	} else if (angle >= YYPI) {
	    real_angle = (angle - YYPI);
	    xscale *= (-1); yscale *= (-1);
	} else if (angle >= YYPI_2) {
	    real_angle = (YYPI - angle);
	    xscale *= (-1);
	} else
	    real_angle = angle;
	ctrl->xscale = xscale * cos(M_PI*real_angle/YYPI);
	ctrl->yscale = yscale * sin(M_PI*real_angle/YYPI);
    }
 send_reply:
    debug_endfunc("draw_text_init");
    return err;
}

yy_packet *yycom_draw_text(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    yy_packet *repl;
    draw_text_control draw_ctrl;
    repl = draw_text_init(ch, pkt, &draw_ctrl, FALSE);
    if (repl != (yy_packet *)NULL || draw_ctrl.tr == (TERRITORY *)NULL)
	return repl;
    return draw_text(ch, pkt, &draw_ctrl, put_b1text, put_b2text);
}

yy_packet *yycom_draw_vtext(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    yy_packet *repl;
    draw_text_control draw_ctrl;
    repl = draw_text_init(ch, pkt, &draw_ctrl, FALSE);
    if (repl != (yy_packet *)NULL || draw_ctrl.tr == (TERRITORY *)NULL)
	return repl;
    return draw_text(ch, pkt, &draw_ctrl, put_b1vtext, put_b2vtext);
}

yy_packet *yycom_draw_rtext(ch, pkt)
    yy_comm_channel *ch;
    yy_packet *pkt;
{
    yy_packet *repl;
    draw_text_control draw_ctrl;
    repl = draw_text_init(ch, pkt, &draw_ctrl, TRUE);
    if (repl != (yy_packet *)NULL || draw_ctrl.tr == (TERRITORY *)NULL)
	return repl;
    return draw_text(ch, pkt, &draw_ctrl, put_b1rtext, put_b2rtext);
}

/***************************************************************************
 ***
 *** Font Info
 ***
 ***************************************************************************/

#define TXTDIR_H	0
#define TXTDIR_V	1

#ifdef newversion
TEXTBOX *YYTextSize(ch, fid, dir, leng, str, base_x, base_y)
    yy_comm_channel *ch;
    int fid;
    int dir;
    int leng;
    u_char *str;
    int base_x, base_y;
{
    YYFONT *yfont;
    static TEXTBOX box;
    int sum_of_width = 0;
    int sum_of_height = 0;
    int max_width = 0;
    int max_ascent = 0;
    int max_descent = 0;
    debug_setfunc("font", "YYTextSize");
    if ((yfont = call_font(ch, fid)) == (YYFONT *)NULL) {
	box.x = box.y = box.h = box.w = 0;
	return &box;
    }
    while (leng > 0) {
	int bytes;
    }
    debug_print(5, "BOX (%d,%d)[%d,%d]\n", box.x, box.y, box.w, box.h);
    debug_endfunc("YYTextSize");
    return &box;
}
#endif /*newversion*/

TEXTBOX *YYTextSize(ch, fid, dir, leng, str, x, y)
    yy_comm_channel *ch;
    int dir;		/* direction */
    int leng;		/* byte length of string */
    char *str;		/* length */
    int x, y;
{
    YYFONT *yfont;
    static TEXTBOX box;
    bool horizontal = (dir == TXTDIR_H);
    int top = 0;
    int tail = 0;
    int width = 0;
    debug_setfunc("font", "YYTextSize");
    if ((yfont = call_font(ch, fid)) == (YYFONT *)NULL) {
	box.x = box.y = box.h = box.w = 0;
	return &box;
    }
    while (leng > 0) {
	if (KCODE(*str)) {
	    leng -= 2;
	    str += 2;
	    if (horizontal) {
		if (top < yfont->fiDouble->fxMaxBound.ciAscent)
		    top = yfont->fiDouble->fxMaxBound.ciAscent;
		if (tail < yfont->fiDouble->fxMaxBound.ciDescent)
		    tail = yfont->fiDouble->fxMaxBound.ciDescent;
		width += yfont->fiDouble->fxMaxBound.ciWidth;
	    } else {
		if (top == 0 && tail == 0) {
		    top = yfont->fiDouble->fxMaxBound.ciAscent;
		    tail = yfont->fiDouble->fxMaxBound.ciHeight;
		    width = yfont->fiDouble->fxMaxBound.ciWidth;
		} else {
		    tail += yfont->fiDouble->fxMaxBound.ciHeight;
		    if (width < yfont->fiDouble->fxMaxBound.ciWidth)
			width = yfont->fiDouble->fxMaxBound.ciWidth;
		}
	    }
	} else {
	    char *s = str;
	    int l = 0;
	    int direction, ascent, descent;
	    XCharStruct cstruct;
	    while (leng > 0 && !KCODE(*s)) {
		s++, leng--, l++;
	    }
	    XTextExtents(yfont->fiSingle->fxFontStruct, str, l,
			 &direction, &ascent, &descent, &cstruct);
	    if (horizontal) {
		if (top < ascent)	top = ascent;
		if (tail < descent)	tail = descent;
		width += cstruct.width;
	    } else {
		register int b = yfont->fiSingle->fxMaxBound.ciDescent;
		if (top == 0 && tail == 0)
		    top = yfont->fiDouble->fxMaxBound.ciAscent;
		tail += cstruct.width;
		if (width < b + ascent)
		    width = b + ascent;
	    }
	    str = s;
	}
    }
    box.x = x;
    box.y = y - top;
    box.w = width;
    box.h = (horizontal? (top+tail): tail);
    debug_print(5, "BOX (%d,%d)[%d,%d]\n", box.x, box.y, box.w, box.h);
    debug_endfunc("YYTextSize");
    return &box;
}

int YYTextWidth(ch, fid, dir, leng, str)
    yy_comm_channel *ch;
    int fid;
    int dir;		/* direction */
    int leng;		/* byte length of string */
    char *str;		/* length */
{
    bool horizontal = (dir == TXTDIR_H);
    YYFONT *yfont;
    int top = 0;
    int tail = 0;
    int width = 0;

    debug_setfunc("font", "YYTextWidth");
    debug_print(9, "Length = %d (%s)\n", leng, (horizontal? "H": "V"));
    if ((yfont = call_font(ch, fid)) == (YYFONT *)NULL)
	return 0;
    while (leng > 0) {
	if (KCODE(*str)) {
	    leng -= 2;
	    str += 2;
	    if (horizontal) {
		width += yfont->fiDouble->fxMaxBound.ciWidth;
	    } else {
		if (width < yfont->fiDouble->fxMaxBound.ciWidth)
		    width = yfont->fiDouble->fxMaxBound.ciWidth;
	    }
	} else {
	    char *s = str;
	    int l = 0;
	    int direction, ascent, descent;
	    XCharStruct cstruct;
	    while (leng > 0 && !KCODE(*s)) {
		s++, leng--, l++;
	    }
	    XTextExtents(yfont->fiSingle->fxFontStruct, str, l,
			 &direction, &ascent, &descent, &cstruct);
	    if (horizontal) {
		width += cstruct.width;
	    } else {
		register int b = yfont->fiSingle->fxMaxBound.ciDescent;
		tail += cstruct.width;
		if (width < b + ascent)
		    width = b + ascent;
	    }
	    str = s;
	}
    }
    debug_print(5, "Text Width = %d\n", width);
    debug_endfunc("YYTextWidth");
    return width;
}


/***************************************************************************
 *** Draw Text on Page Mode Territory
 ***************************************************************************/

void do_draw_text_on_page(ch, tr, x, y, fid, leng, str)
    yy_comm_channel *ch;
    TERRITORY *tr;
    int x, y;
    int fid;
    int leng;
    char *str;
{
    char *ws, *we;
    draw_text_control draw_ctrl;
    debug_setfunc("font", "do_draw_text_on_page");
    debug_print(9, "Put %d Characters on (%d,%d)\n", leng, x, y);
    draw_ctrl.xp = XPRIVATE(ch);
    draw_ctrl.tr = tr;
    draw_ctrl.tx = GetTerritoryXEntry(tr);
    draw_ctrl.top_x = draw_ctrl.x = x;
    draw_ctrl.top_y = draw_ctrl.y = y;
    draw_ctrl.font = call_font(ch, fid);
    for (ws = str; leng > 0; ws = we) {
	if (KCODE(*ws)) {
	    register int wlen = 0;
	    for (we = ws; KCODE(*we) && leng > 0; ) {
		*we++ &= 0177; *we++ &= 0177;
		wlen++, leng -= 2;
	    }
	    if (DRAW_B2XFONT(&draw_ctrl) != (YYXFONT *)NULL)
		put_b2text(&draw_ctrl, ws, wlen);
	} else {
	    register int wlen = 0;
	    for (we = ws; !KCODE(*we) && leng > 0; ) {
		*we++; wlen++; leng--;
	    }
	    put_b1text(&draw_ctrl, ws, wlen);
	}
    }
    if (IsVisibleTerritory(tr))
	ch->ccXNeedFlush = TRUE;
    debug_endfunc("do_draw_text_on_page");
}


static YYXFONT *alloc_x_font_info(name)
    char *name;
{
    static yy_font_x_table *top = (yy_font_x_table *)NULL;
    register yy_font_x_table *cur;
    for (cur = top ;cur != (yy_font_x_table *)NULL; cur = cur->fxNext)
	if (strSAME(cur->fxFontInfo.fxFontName, name))
	    return &cur->fxFontInfo;
    cur = (yy_font_x_table *)memALLOC(sizeof(yy_font_x_table));
    if (cur != (yy_font_x_table *)NULL) {
	if (name != (char *)NULL && name[0] != EOS && !strSAME(name, "-"))
	    cur->fxFontInfo.fxFontName = strDUP(name);
	else
	    cur->fxFontInfo.fxFontName = (char *)NULL;
	cur->fxFontInfo.fxFontStruct = (XFontStruct *)NULL;
	cur->fxNext = top;
	top = cur;
    }
    return (cur != (yy_font_x_table *)NULL? &cur->fxFontInfo: (YYXFONT *)NULL);
}


yy_font_table *alloc_font_table(ffile)
    char *ffile;	/* Font Info File */
{
    int id = 0;
    yy_font_table *top = (yy_font_table *)NULL;
    FILE *fp;
    char line[1024];
    debug_setfunc("font", "yy_font_table");
    if ((fp = fopen(ffile, "r")) == NULL) {
	warning("Can't open font info file '%s'\n", ffile);
	return (yy_font_table *)NULL;
    }
    while (fgets(line, sizeof(line), fp)) {
	register char *s, *e;
	register yy_font_table *new;
	char *av[3];
	if (getargs(line, av, 3) != 3)
	    continue;	/* Format Error */
	new = (yy_font_table *)memALLOC(sizeof(yy_font_table));
	if (new == (yy_font_table *)NULL)
	    break;
	id++;
	debug_print(1, "Set Font Info - ID:#%d\n", id);
	debug_print(2, " Font Name : %s (%s,%s)\n", av[0], av[1], av[2]);
	new->yfFontInfo.fiLoaded = FALSE;
	new->yfFontInfo.fiFID = id;
	new->yfFontInfo.fiFontName = strDUP(av[0]);
	new->yfFontInfo.fiSingle = alloc_x_font_info(av[1]);
	new->yfFontInfo.fiDouble = alloc_x_font_info(av[2]);
	/* Link Infomation */
	new->yfNext = top;
	top = new;
    }
    fclose(fp);
    debug_print(5, "%d fonts are avairable..\n");
    debug_endfunc("yy_font_table");
    return top;
}
