;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;  graphic-primitive.lisp
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.0 90/06/01 by t.kosaka (kosaka@csrl.aoyama.ac.jp)
;;;   version 1.1 90/07/31 by t.kosaka
;;;   update 1.11 90/09/14 by t.kosaka
;;;   version 1.2 90/11/05 by t.kosaka

;;; $BIA2h5Z$SI=<($N$?$a$N4pK\%/%i%9$NDj5A(B
;;; 1/26 1990 $B8E:d(B

;;; Version 1.0   Coded by t.kosaka 1990-1-26
;;; Change log 1990-6-20 add get-new-position-xy 
;;;            1990-8-24 cursor-position is no operation
;;;            1990-10-4 Add work-region1 anf work-region2 to graphic-stream

(in-package :yy)


;;; $B:BI8JQ49%/%i%9%_%-%7%s(B
(defclass translate-coordinate-mixin () ())

;;; $B:BI8JQ49%/%i%9(Bleft bottom
;;; $B:82<$9$_MQ$N%/%i%9(B
(defclass translate-coordinate-left-bottom
  (translate-coordinate-mixin)
  ())



;;; $B:BI8JQ49%/%i%9(Bleft top$BMQ(B
;;; $B:82<$9$_:BI87O%/%i%9(B
(defclass translate-coordinate-left-top
  (translate-coordinate-mixin)
  ())

;;; $B:BI8JQ49$NAm>N4X?t(B XY$BMQ(B
(defgeneric translate-coordinate-y (object y max-height)
  (:method ((object translate-coordinate-left-bottom)
	    y max-height)
           ;;; $B:82<$9$_:BI87O$r:8>e$9$_:BI87O$KJQ49$9$k(B
	   (- max-height y))
  (:method ((object translate-coordinate-left-top)
	    y max-height)
#-:PCL
  (declare (ignore max-height))
           ;;; $B:BI87O$,:8>e$9$_$J$N$G!"$J$K$b$7$J$$(B
	   y))

;;; $B:BI8JQ49$NAm>N4X?t(B position$BMQ(B
(defgeneric translate-coordinate (object position max-height)
  (:method ((object translate-coordinate-left-bottom)
	    (position position)
	    max-height)
	   (declare (inline -))
	   (make-position :x (position-x position)
			  :y (- max-height (position-y position))))
  (:method ((object translate-coordinate-left-top)
	    (position position)
            max-height)
#-:PCL
  (declare (ignore max-height))
	   position))

;;; $B:BI8JQ49$NAm>N4X?t(B region $BMQ(B
(defgeneric translate-coordinate-region (object region max-height)
  (:method ((object translate-coordinate-left-bottom)
	    (region region)
	    max-height)
	   (declare (inline max min))
	   (let ((bottom (min (- max-height (region-bottom region))
			      (- max-height (region-top region))))
		 (top (max (- max-height (region-bottom region))
			   (- max-height (region-top region)))))
	     (make-region :left (region-left region)
			  :bottom bottom
			  :width (region-width region)
			  :top top)))
  (:method ((object translate-coordinate-left-top)
	    (region region)
	    max-height)
#-:PCL
  (declare (ignore max-height))
	   region))


;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9%_%-%7%s(B
(defclass graphic-transform-matrix-mixin ()
;;;  | cos theta -sin theta | | x-time |
;;;  | sin theta  cos theta | | y-time |
;;; $B$N2sE>9TNs$rDs6!$9$k%/%i%9(B
  ((theta :initarg :theta   :accessor matrix-theta :initform 0.0 :type number)
   (x-time :initarg :x-time :accessor matrix-x-time :initform 1.0 :type number)
   (y-time :initarg :y-time :accessor matrix-y-time :initform 1.0 :type number)))

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N<B9T%/%i%9(B
(defclass graphic-transform-matrix-exec 
  (graphic-transform-matrix-mixin)
  ((step-x1 :initarg :x-step1 :initform 1 :accessor step-x1)
   (step-x2 :initarg :x-step2 :initform 1 :accessor step-x2)
   (step-y1 :initarg :y-step1 :initform 1 :accessor step-y1)
   (step-y2 :initarg :y-step2 :initform 1 :accessor step-y2)))

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$NHs<B9T%/%i%9(B
(defclass graphic-transform-matrix-non
  (graphic-transform-matrix-mixin)
  ())

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9%$%s%9%?%s%9I=<(%a%=%C%I(B
(defmethod print-object ((matrix  graphic-transform-matrix-mixin) stream)
  (let ((tt (matrix-theta matrix)) 
	(xx (matrix-x-time matrix)) 
	(yy (matrix-y-time matrix)))
   (format stream "~%\#<Matrix \|cos ~a ~t-sin ~a\|\|~a|~%         \|sin ~a ~t cos ~a\|\|~a\|>" 
	    tt tt xx tt tt yy)))


;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N(Bsetf $B%a%=%C%I(B $B&H$NJQ99(B
(defmethod (setf matrix-theta) :around (new-theta
			       (matrix graphic-transform-matrix-mixin))
  (if (numberp new-theta)
      (call-next-method)
  (error "The aergument ~a is an illegal value." new-theta)))

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9(B
(defmethod (setf matrix-theta) :after (new-theta
				       (matrix graphic-transform-matrix-exec))
  (declare (inline sin cos / * - round))
  (let* ((sinx (sin (* pi (/ new-theta 180.0))))
	 (cosx (cos (* pi (/ new-theta 180.0)))))

    (setf (step-x1 matrix) cosx
	  (step-x2 matrix) (*  sinx -1)
	  (step-y1 matrix) sinx
	  (step-y2 matrix) cosx))
  )


;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N%"%/%;%C%5(B x-time$B$NJQ99(B
(defmethod (setf matrix-x-time) :around (new-x
				 (matrix graphic-transform-matrix-mixin))
  (if (numberp new-x)
      (call-next-method)
    (error "The argument ~a is an illegal value." new-x)))

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N%"%/%;%C%5(B y-time$B$NJQ99(B
(defmethod (setf matrix-y-time) :around  (new-y
				 (matrix graphic-transform-matrix-mixin))
  (if (numberp new-y)
      (call-next-method)
    (error "The argument ~a is an illegal value." new-y)))

;;;$B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N@8@.4X?t(B
(defun make-transform-matrix (&key (theta 0.0) (x-time 1.0) (y-time 1.0))
;;;$B%f!<%6!<$O!"$3$N4X?t$rMQ$$$F2sE>9TNs$r:n@.$9$k(B
  (let ((matrix (make-instance 'graphic-transform-matrix-exec
			       :theta theta :x-time x-time :y-time y-time)))
    (setf (matrix-theta matrix) theta)
    matrix))
  

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/$NAm>N4X?t(B ($B%]%8%7%g%sMQ(B)
(defgeneric transform-by-matrix (pos matrix &key return)
  (:method ((pos position) 
	    (matrix graphic-transform-matrix-exec)
	    &key (return :multiple))
   (with-position-destructured 
    ((x y) pos)
    (with-slots 
     (step-x1 step-x2 step-y1 step-y2 x-time y-time) matrix
     (let ((new-x (round (* (+ (* x step-x1) (* y step-x2)) x-time)))
	   (new-y (round (* (+ (* x step-y1) (* y step-y2)) y-time))))
       (case return
	 (:multiple (values new-x new-y))
	 (:position (make-position :x new-x
				   :y new-y))
	 (t (list new-x new-y)))
       ))))
  (:method ((pos position)
	    (matrix graphic-transform-matrix-non)
	    &key (return :multiple))
   (with-position-destructured ((x y) pos)
       (case return
	     (:maltiple (values x y))
	     (:position pos)
	     (t (list x y)))
       )))

;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/$NAm>N4X?t(B (XY $BMQ(B)
(defgeneric transform-by-matrix-xy (x y matrix  &key return)
  (:method ((x number) (y number)
	    (matrix graphic-transform-matrix-exec)
	    &key (return :multiple))
   (with-slots 
    (step-x1 step-x2 step-y1 step-y2 x-time y-time) matrix
    (let ((new-x (round (* (+ (* x step-x1) (* y step-x2)) x-time)))
	  (new-y (round (* (+ (* x step-y1) (* y step-y2)) y-time))))
      (case return
	(:multiple (values new-x new-y))
	(:position (make-position :x new-x :y new-y))
	(t (list new-x new-y)))
      )))
  (:method ((x number) (y number)
	    (matrix graphic-transform-matrix-non)
	    &key (return :multiple))
  (let ((new-x (round x)) (new-y (round y)))
    (case return
	  (:multiple (values new-x new-y))
	  (:position (make-position :x new-x :y new-y))
	  (t (list new-x new-y)))
    )))

#|
;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/$NAm>N4X?t(B (X $BMQ(B)
(defgeneric transform-by-matrix-x (x matrix)
  (:method ((x integer)
	    (matrix graphic-transform-matrix-exec))
   (with-slots
    (step-x1) matrix
    (* x step-x1)))
  (:method ((x integer)
	    (matrix graphic-transform-matrix-non))
     x))

 
;;; $B%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/$NAm>N4X?t(B (Y $BMQ(B)
(defgeneric transform-by-matrix-y (y matrix)
  (:method ((y integer)
	    (matrix graphic-transform-matrix-exec))
  (with-slots (step-y2) matrix
	      (- (* y step-y2))))
  (:method ((y integer)
	    (matrix graphic-transform-matrix-non))
    y))
|#

;;; $B%0%i%U%#%C%/%9%F%$%H%_%-%7%s(B
(defclass graphic-state-mixin  ()
  ((operation :initarg :operation :accessor graphic-operation
	      :initform *GCOPY*)
   (color :initarg :color :accessor graphic-color :type color
	  :initform *BLACK-COLOR*)
   (line-width :initarg :line-width :accessor line-width :type integer
	       :initform 1)
   (line-edge :initarg :line-edge :accessor line-edge 
	      :initform *SQUEAR-LINE-EDGE* :type integer)
   (joint-type :initarg :line-joint-type :accessor line-joint-type
	       :initform *SHARP-JOINT* :type integer)
   (line-dashing :initarg :line-dashing :accessor line-dashing 
		 :type simple-string :initform "")

   (arc-mode :initarg :arc-mode :accessor arc-mode :initform *ARCPIESLICE*)

   ;;; filled-type *Fillnon* *FillSolid* *FillTiled* *FillStippled* 
   ;;; *FillOpaqueStippled*
   (filled-type :initarg :filled-type :accessor filled-type 
		:initform *Fillednon*)

   ;;; graphic-stream default color or pattern
   ;;; $B%+%i!<!"%$%a!<%8!"%S%C%H%^%C%W$,F~$k!#(B
   (default-color-pattern :initarg :default-color-pattern
			 :initform *white-color*
			 :accessor stream-default-color-pattern)
   

   ;;;  fillRule *EvenOddRule* *WindingRule*
   (filled-rule :initarg :filled-rule :accessor filled-rule :initform  *EvenOddRule*)
   (filled-pattern :initarg :filled-pattern
		   :accessor filled-pattern :initform nil)))

;;; $B%3%T!<$r:n$k(B
(defmethod copy-gaphic-state ((state graphic-state-mixin))
  (make-instance 'graphic-state-mixin 
		 :operation (graphic-operation state)
		 :color (graphic-color state)
		 :line-width (line-width state)
		 :line-edge (line-edge state)
		 :line-joint-type (line-joint-type state)
		 :line-dashing (line-dashing state)
		 :arc-mode (arc-mode state)
		 :filled-type (filled-type state)
		 :filled-rule (filled-rule state)
		 :filled-pattern (filled-pattern state)))

;;; $B%*%Z%l!<%7%g%s$NJQ99(B
(defmethod (setf graphic-operation) :around  (new-value (state graphic-state-mixin))
  (if (or (eq new-value *GCLEAR*) (eq new-value *GAND*) 
	  (eq new-value *GANDREVERSE*) (eq new-value *GCOPY*)
	  (eq new-value *GANDINTEVERTED*) (eq new-value *GNOOP*)
	  (eq new-value *GXOR*) (eq new-value *GOR*)
	  (eq new-value *GNOR*) (eq new-value *GEQIV*)
	  (eq new-value *GINVERT*) (eq new-value *GORREVERSE*)
	  (eq new-value *GCOPYINVERTED*) (eq new-value *GORINVERTED*)
	  (eq new-value *GNAND*) (eq new-value *GSET*))
      (setf (slot-value state 'operation) new-value)
    (error "The Operation ~a is an illegal argument. Shuld be from #x00 to #xFF"
	   new-value)))

;;; $BIA2h?'$NJQ99(B
(defmethod (setf graphic-color) :around (new-color (state graphic-state-mixin))
  (if (eq (class-name (class-of new-color)) 'color)
      (call-next-method)
    (error "The argument ~a is not a color class instance." new-color)))

;;; $B@~$NI}$NJQ99(B
(defmethod (setf line-width) :around (new-width (state graphic-state-mixin))
  (if (integerp new-width)
      (call-next-method)
  (error "The argument ~a is an illegal value." new-width)))

;;; $B@~$NC<$N7A(B
(defmethod (setf line-edge) :around (new-edge  (state graphic-state-mixin))
  (if (or (eq *SQUEAR-LINE-EDGE* new-edge)
	  (eq *SQUEAR-LINE-EDGE-WITHOUT-END* new-edge)
	  (eq *ROUND-LINE-EDGE* new-edge))
      (call-next-method)
    (error "The line edge style ~a is an illegal value." new-edge)))

;;; $B@~$N@\B3$N7A(B
(defmethod (setf line-joint-type) :around (new-joint (state graphic-state-mixin))
  (if (or (eq *SHARP-JOINT* new-joint)
	  (eq *ROUND-JOINT* new-joint))
      (call-next-method)
    (error "The line joint type ~a is an illegal value." new-joint)))

;;; $B@~$N%@%C%7%s%0$N7A>u(B
(defmethod (setf line-dashing) :around (new-dash (state graphic-state-mixin))
  (if (stringp new-dash)
      (call-next-method)
    (error "The dashing type ~a is an illegal value" new-dash)))


;;; $B1_8L$N%b!<%I(B
(defmethod (setf arc-mode) :around (new-mode (state graphic-state-mixin))
  (if (or (eq *ARCPIESLICE* new-mode)
	  (eq *ARCCHORD* new-mode))
      (call-next-method)
    (error "The arc mode ~a is an illegal value." new-mode)))

;;; $BEI$j$D$V$7%b!<%I(B
(defmethod (setf filled-type) :around (new-type  (state graphic-state-mixin))
  (if (or (eq *FillSolid* new-type)
	  (eq *Fillednon* new-type)
	  (eq *FillTiled* new-type)
	  (eq *FillOpaqueStippled* new-type))
      (call-next-method)
    (error "The filled type ~a is an illigale value." new-type)))

;;; $BEI$j$D$V$7%k!<%k(B
(defmethod (setf filled-rule) :around (new-rule (state graphic-state-mixin))
  (if (or (eq  *EvenOddRule* new-rule)
	  (eq *WindingRule* new-rule))
      (call-next-method)
    (error "The filled rule ~a is an illigale value." new-rule)))


;;; $B%0%i%U%#%C%/%9%H%j!<%`$NIA2h%5%$%:%/%i%9(B
(defclass graphic-output-stream-size (graphic-state-mixin )
  ((world-width :initarg :width :accessor world-width
	  :type integer :initform 0)
   (world-height :initarg :height :accessor world-height
	   :type integer :initform 0)))

(defmethod world-height (stream)
  (if (null stream)
      0
    ))

;;; $B%0%i%U%#%C%/%9%H%j!<%`$N%F%-%9%H%3%s%H%m!<%k%/%i%9(B
(defclass graphic-text-control-mixin (graphic-output-stream-size)
   ;;; $B:8$O$7$N%F%-%9%H3+;O0LCV(B nil$B$J$i$P%9%H%j!<%`$N3+;O0LCV$+$i(B
   ;;; $BCM$,$"$l$P!"$=$N0LCV$+$i(B
  ((left-margin :initarg :left-margin :initform nil
		:accessor stream-left-margin)
   ;;; $BJ8;zNs=PNO$N=*N;0LCV(B nil$B$J$i$PIA2h%5%$%:$K=>$&(B
   ;;; $BCM$,$"$l$P!"$=$N0LCV$^$G(B
   (right-margin :initarg :right-margin :initform nil
		:accessor stream-right-margin)
   (top-margin :initarg :top-margin :initform nil
   	       :accessor stream-top-margin)
   (bottom-margin :initarg :bottom-margin :initform nil
   		  :accessor stream-bottom-margin)
   ;;; $B%F%-%9%HMQ%U%)%s%H(B
   (font :initarg :text-font :accessor stream-font :type text-font
	 :initform *default-font*)
   ;;; $B%F%-%9%H$N%i%$%s%U%#!<%I9b$5(B
   (line-feed :initarg :line-feed  :initform 0
	      :accessor stream-line-feed)

   ;;; $BJ8;zNs$N=P%9%?%$%k(B horizontal $B$J$i$P2#=q$-(B :vertical$B$J$i$P=D=q$-(B
   (output-direction :initarg :output-direction 
		     :accessor stream-output-direction  
		     :initform :horizontal)
   ;;; $B%^!<%8%sI}$G$NJ8;zNs:o=|(B $B2#=q$-(B $BMQ(B
   (truncate-width :initarg :truncate-width :accessor stream-truncate-width
		   :initform nil)
   ;;; $B%^!<%8%s9b$5$G$NJ8;zNs:o=|(B $B=D=q$-MQ(B
   (truncate-height :initarg :truncate-heigt :accessor stream-truncate-height
		    :initform nil)
   ))


;;; $B%F%-%9%H%3%s%H%m!<%k%/%i%9$N%3%T!<(B
(defmethod copy-graphic-text-control-mixin ((state graphic-text-control-mixin))
  (make-instance 'graphic-text-control-mixin
		 :left-margin (stream-left-margin state)
		 :right-margin (stream-right-margin state)
		 :top-margin (stream-top-margin state)
		 :bottom-margin (stream-bottom-margin state)
		 :text-font (stream-font state)
		 :line-feed (stream-line-feed state)
		 :output-direction (stream-output-direction state)))



;;; $B%"%/%;%9%a%=%C%I(B left-margin setf $BMQ(B nil $B?t;zMQ(B
(defmethod (setf stream-left-margin) :around (new-value
				      (instance graphic-text-control-mixin))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The left margin ~a is an illegal value." new-value)))

;;; $B%"%/%;%9%a%=%C%I(B right-margin setf $BMQ(B nil $B?t;zMQ(B
(defmethod (setf stream-right-margin) :around (new-value
				      (instance graphic-text-control-mixin))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The right margin ~a is an illegal value." new-value)))

;;; $B%"%/%;%9%a%=%C%I(B top-margin setf $BMQ(B nil 
(defmethod (setf stream-top-margin) :around (new-value 
				      (instance graphic-text-control-mixin))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The top margin ~a is an illegal value." new-value)))

;;; $B%"%/%;%9%a%=%C%I(B bottom-margin setf $BMQ(B nil 
(defmethod (setf stream-bottom-margin) :around (new-value 
				      (instance graphic-text-control-mixin))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The bottom margin ~a is an illegal value." new-value)))


;;;  X3J13. Stream OUTPUT
(defclass fundamental-character-output-stream ()
  ((string :initform (make-array 64.
                                :element-type
#-:EXCL
                      'string-char
#+:EXCL
                      'character
		      :fill-pointer 0 :adjustable t)
 :initarg :string )
   (endp    :initform 0 :initarg :endp :type :integer)))


;;; $B%0%i%U%#%C%/%9%H%j!<%`(B
(defclass graphic-stream
  (graphic-text-control-mixin fundamental-character-output-stream)

   ;;; $B%+!<%=%k$N%]%8%7%g%s%9%m%C%H(B
  ((cursor-position :initarg :strem-cursor-position  
		    :type position :accessor stream-cursor-position
		    :initform (make-position))
   
   ;;; $B:BI8JQ49MQ%9%m%C%H(B $B%G%U%)%k%H$O!":8>e$9$_(B
   (translate-coordinate :initarg :translate-coordinate-ob
			 :type translate-coordinate-mixin
			 :reader stream-translate-coordinate
			 :initform nil)

   ;;; $B%0%i%U%#%/%9%H%i%s%9%U%)!<%`%^%H%j%C%/%9%9%m%C%H(B
   ;;; $B%G%U%)%k%H$O!"2?$b$7$J$$(B
   (transform-by-matrix :initarg :transform-matrix
        :reader stream-transform-by-matrix
        :type graphic-transform-matrix-mixin
        :initform  (make-instance 'graphic-transform-matrix-non))

   ;;; $B%o!<%/MQ(Bregion
   (work-region1 :initform (make-region)
		 :accessor work-region1)
   (work-region2 :initform (make-region)
		 :accessor work-region2)
   (work-region3 :initform (make-region)
		 :accessor work-region3)
   (work-region4 :initform (make-region)
		 :accessor work-region4)

   ))

;;; $B7?H=Dj(B
(defun graphic-stream-p (stream)
  (typep stream 'graphic-stream))

;;; $B:BI8JQ49$r$7$?%j!<%8%g%s$rJV$9(B
(defmethod change-coodinate-region ((region  region)
				    (stream graphic-stream))
  (translate-coordinate-region (stream-translate-coordinate stream)
			       region
			       (world-height stream))
  )
			       


#|
;;; stream-line-feed $B$N%"%/%C%;%5!<(B
(defmethod stream-line-feed ((stream graphic-stream))
  (abs (slot-value stream 'line-feed)))

;;; stream-line-feed $B$N%a%C%=%I(B
(defmethod (setf stream-line-feed) ((new-v T) (stream graphic-stream))
  (if (numberp new-v)
      (if (eq (class-name (class-of (stream-translate-coordinate stream)))
	      'translate-coordinate-left-bottom)
	  (setf (slot-value stream 'line-feed) (- new-v))
	(setf (slot-value stream 'line-feed) new-v))
    (error "The argument type is not a integer.")))

;;; graphic-stream$B$N=i4|2=%a%=%C%I(B
(defmethod initialize-instance :after ((stream graphic-stream) &rest arg)
  (let ((val (nth (+ (position :line-feed arg) 1) arg)))
    (if (eq (class-name (class-of (stream-translate-coordinate stream)))
              'translate-coordinate-left-bottom)
	(setf (slot-value stream 'line-feed) (- val))
      (setf (slot-value stream 'line-feed) val))
    ))

|#

;;;$B%+!<%=%k$N(BX$B%]%8%7%g%s$N%"%/%;%C%5(B
(defmethod stream-cursor-x-position  ((stream graphic-stream))
  (position-x (slot-value stream 'cursor-position)))

;;;$B%+!<%=%k$N(BY$B%]%8%7%g%s$N%"%/%;%C%5(B
(defmethod stream-cursor-y-position ((stream graphic-stream))
  (position-y (stream-cursor-position stream)))

#|

  (translate-coordinate-y (stream-translate-coordinate stream)
			  (position-y (slot-value stream 
						  'cursor-position))
			  (world-height stream)))

;;; $B%+!<%=%k%]%8%7%g%s$N%"%/%;%C%5(B
(defmethod stream-cursor-position ((stream graphic-stream))
  (with-translate-transform ((position) stream
			     (slot-value stream 'cursor-position))
     position))

;;; $B%+!<%=%k%]%8%7%g%s$N(Bsetf$B%"%/%;%C%5(B
(defmethod (setf stream-cursor-position) ((new-position position)
					  (stream graphic-stream))
  (with-translate-transform ((position) stream  new-position)
     (setf (slot-value stream 'cursor-position) position)))
			     

|#

;;;$B%+!<%=%k$N(BX$B%]%8%7%g%s$N(Bsetf$B%"%/%;%C%5(B 
(defmethod (setf stream-cursor-x-position) ((new-x integer)
    			         	    (stream graphic-stream))
  (setf (position-x (slot-value stream 'cursor-position)) new-x))


;;;$B%+!<%=%k$N(BY$B%]%8%7%g%s$N(Bsetf$B%a%=%C%I(B
(defmethod (setf stream-cursor-y-position) ((new-y integer)
					   (stream graphic-stream))
  (setf (position-y (stream-cursor-position stream)) new-y))
	
;;; translate-coordinate$B$NJQ99%a%=%C%I(B
(defmethod (setf translate-coordinate) ((new-value 
					 translate-coordinate-mixin)
					(stream graphic-stream))

  (setf (slot-value stream 'translate-coordinate) new-value))


;;; transform-by-matrix $B$NJQ99%a%=%C%I(B
(defmethod (setf stream-transform-by-matrix) ((new-value 
					graphic-transform-matrix-mixin)
				       (stream graphic-stream))
  (setf (slot-value stream 'transform-by-matrix) new-value)
)
  	
;;; $B2sE>9TNs$NJQ49$r2C$($??7$7$$(Bposition$B$rJV$9(B
;;; $B%a%=%C%I(B
(defmethod get-new-position ((pos position) (stream graphic-stream))
  (with-transform-stream pos stream))

;;; stream graphic-stream$B0J30$O!"$=$N$^$^$rJV$9!#(B
(defmethod get-new-position ((pos position) (stream T))
  pos)


;;; $B2sE>9TNs$NJQ49$r2C$($??7$7$$(BXY$B$rJV$9(B
(defmethod get-new-position-xy ((x integer) (y integer) (stream graphic-stream))
  (transform-by-matrix-xy x y (slot-value stream 'transform-by-matrix)
			  :return :multiple))

;;; stream graphic-stream$B0J30$O!"$=$N$^$^$rJV$9(B
(defmethod get-new-position-xy ((x integer) (y integer) (stream T))
  (values x y))

;;; $B2sE>9TNs!":BI8JQ49$r$7$?CM$rJV$9(B($B%j%9%HMQ!K(B
(defmethod translate-transform-xy ((stream graphic-stream) 
				   (x integer)
				   (y integer))
  (let ((ret-val
	 (transform-by-matrix-xy x y (slot-value stream 'transform-by-matrix)
				 :return T)))

    (list (first ret-val) 
	  (translate-coordinate-y (slot-value stream 'translate-coordinate)
		    (second ret-val) (world-height stream)))
    ))

;;; $B2sE>9TNs!":BI8JQ49$r$7$?CM$rJV$9(B
(defmethod translate-transform-xy-valuse ((stream graphic-stream)
                                   (x integer)
                                   (y integer))
  (multiple-value-bind (xx yy)
       (transform-by-matrix-xy x y (slot-value stream 'transform-by-matrix)
			       :return :multiple)
       (values xx 
	       (translate-coordinate-y (slot-value stream 
						   'translate-coordinate)
		    yy (world-height stream)))))

;;;  X3J13. Stream INPUT
(defclass fundamental-character-input-stream  ()
  ((input-string :initarg input-string :type list :reader stream-input-string-internal
		 :initform nil)
   (edit-string :initarg edit-string :type list 
		:initform nil)
   (sx-position :initarg sx-position :type integer :accessor sx-position)
   (sy-position :initarg sy-position :type integer :accessor sy-position)
   (current-output-direction :accessor current-output-direction)
   (current-matrix :accessor current-matrix :type graphic-transform-matrix-mixin)
   (current-text-color :accessor current-text-color :type color)
   (current-operation :accessor current-operation)
   (current-line-feed :accessor current-line-feed :type integer)
   (current-font :accessor current-font :type text-font)
   (index :initarg :start :type fixnum :initform 0)
   (text-cursor-x :initarg text-cursor-x :accessor text-cursor-x)
   (text-cursor-y :initarg text-cursor-y :accessor text-cursor-y)
   (text-curcor-visible :initform 0 :type integer :accessor text-curcor-visible)
   (for-unread-char :initarg :for-unread-char :type character :accessor for-unread-char)
   ))

