;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; graphic-method-p.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.3 91/02/06 by t.kosaka

;;; $B%W%l%<%s%F!<%7%g%s$N$?$a$NIA2h(B

(in-package :yy)

;;; $BE@$NIA2h(B 
;;; draw-position-xy graphic-stream x y
;;; ASG. 
;;;         graphic-stream     = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;         x y                = $B0LCV(B
(defmethod p-draw-position-xy ((stream graphic-stream) (x integer) (y integer))
  (p-draw-position-xy-internal stream x y))

;;; $BE@$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-position graphic-stram position
;;; ARG.
;;;             graphci-stream   =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;             psosition        =  $B%0%i%U%#%/%9%9%H%j!<%`$G$N0LCV(B
(defmethod p-draw-position ((stream graphic-stream) (position position))
  (p-draw-position-xy-internal
       stream (position-x position) (position-y position)))


;;; draw-position$BIA2h4X?t$N%W%j%_%F%#%V(B
(defun p-draw-position-xy-internal (stream x y)
  (declare (inline + max min))
  (with-translate-transform-xy ((new-x new-y) stream x  y)
    (with-temp-region-args ((draw-ponit-region) (work-region1 stream)
			   :left new-x :right new-x
			   :top new-y :bottom new-y)
     (let* ((drawing-region (drawing-region stream))
	    ;; graphic-statement
	    (color (graphic-color stream))
	    (operation (graphic-operation stream)))

        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
       (setf (world-region stream) draw-ponit-region)

      ;;; $B0LCV$NJQ99(B
      (setf new-x (+ new-x (world-x-start stream))
	    new-y (+ new-y (world-y-start stream)))
      
      ;;; call draw primitive
      (draw-position-primitive (presentation-instance stream)
			       new-x new-y operation (color-no color)
			       x y)

      ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
      (set-drawing-region drawing-region draw-ponit-region)
      stream))))

;;; draw-position-primitive for presentation 
(defun draw-position-primitive (presentation x y op cl-no old-x old-y)
  (push #'(lambda (ptno dx dy)
	    (yy-protocol-20 ptno (+ x dx) (+ y dy) op cl-no))
	(cdr (last (after-drawing-list presentation))))

  ;;; $BIA2h%7!<%1%s%9$r5-21(B
  (push #'(lambda (new-stream)
                 (draw-position-xy new-stream old-x old-y))
             (cdr (last  (drawing-sequence presentation))))
  )


;;; $B@~$NIA2h(B
;;; draw-line-xy stream x1 y1 x2 y2
;;; ARG.
;;;               stream        =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;               x1 y1 x2 y2   =   $B@~$N:BI8(B
(defmethod p-draw-line-xy ((stream graphic-stream) (x1 integer) (y1 integer)
			 (x2 integer) (y2 integer))
  (p-draw-line-xy-internal stream x1 y1 x2 y2))

;;; $B@~$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-line stream position1 position2
;;; ARG
;;;           stream              =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           position1 position2 =   $B@~$NIA2h$9$k0LCV%$%s%9%?%s%9(B
(defmethod p-draw-line ((stream graphic-stream) (position1 position)
		      (position2 position))
	(p-draw-line-xy-internal
	        stream (position-x position1) (position-y position1)
			     (position-x position2) (position-y position2)))


;;; draw-line$BIA2h4X?t$N%W%j%_%#%F%V(B
(defun p-draw-line-xy-internal (stream x1 y1 x2 y2)
  (declare (inline + max min))
  (with-translate-transform-xy ((new-x1 new-y1) stream x1 y1)
    (with-translate-transform-xy ((new-x2 new-y2) stream x2 y2)
      (with-temp-region-args ((draw-line-region) (work-region1 stream)
			     :left (min new-x1 new-x2)
			     :top (max new-y1 new-y2)
			     :right (max new-x1 new-x2)
			     :bottom (min new-y1 new-y2))
      (let* ((drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (color-no (graphic-color stream)))
	     (operation (graphic-operation stream))
	     (line-width (line-width stream))
	     (line-edge (line-edge stream))
	     (line-dash (line-dashing stream)))


        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-line-region)

      ;;; $B0LCV$NJQ99(B
      (setf new-x1 (+ new-x1 (world-x-start stream))
	    new-y1 (+ new-y1 (world-y-start stream))
	    new-x2 (+ new-x2 (world-x-start stream))
	    new-y2 (+ new-y2 (world-y-start stream)))
      
      ;;; $BIA2h%W%j%_%F%#%V(B
      (draw-line-primitive (presentation-instance stream)
			   new-x1 new-y1 new-x2 new-y2 
			   line-width operation
			   line-edge (color-no color) line-dash
			   x1 y1 x2 y2)

        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
      (set-drawing-region drawing-region draw-line-region)
      stream)))))


;;; draw-line-primitive for presentation 
(defun draw-line-primitive (presentation x1 y1 x2 y2 line-width op 
			    line-edge cl-no line-dash
			    ox1 oy1 ox2 oy2)
  (push #'(lambda (p-tno dx dy)
	    (yy-protocol-21 p-tno (+ x1 dx) (+ y1 dy) (+ x2 dx) (+ y2 dy)
              line-width op line-edge cl-no line-dash))
	(cdr (last (after-drawing-list presentation))))

  (push #'(lambda (new-stream)
	    (draw-line-xy new-stream ox1 oy1 ox2 oy2))
	(cdr (last (drawing-sequence presentation))))
    )


;;; $B6k7A$NIA2h(B
;;; draw-region-xy stream x y width height
;;; ARG.
;;;           stream           =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           x y              =  $B6k7A$rIA2h$9$k3+;O0LCV(B
;;;           width height     =  $B6k7A$NBg$-$5(B
(defmethod p-draw-region-xy ((stream graphic-stream) (x integer) (y integer)
			 (width integer) (height integer))
  (p-draw-region-xy-internal stream x y width height))

;;; $B6k7A$NIA2h%]%8%7%g%sMQ(B
;;; draw-region stream position width height
;;; ARG.
;;;             stream            =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;             position          =   $B6k7A$rIA2h$9$k3+;O0LCV%$%s%9%?%s%9(B
;;;             width height      =   $B6k7A$NBg$-$5(B
(defmethod p-draw-region ((stream graphic-stream) (pos position)
			 (width integer) (height integer))
  (p-draw-region-xy-internal stream (position-x pos) (position-y pos)
		  width height))

;;; $B6k7A$NIA2h(B $B%j!<%8%g%sMQ(B
;;; draw-region-region stream region
;;; ARG.
;;;             stream           = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;             region           = $B6k7A$N0LCV$HBg$-$5$r;}$D%$%s%9%?%s%9(B
(defmethod p-draw-region-region ((stream graphic-stream) (region region))


  (p-draw-region-xy-internal stream (region-left region) (region-bottom region)
		  (region-width region) (region-height region)))


;;; p-draw-region-xy $BIA2h%W%j%_%F%#%V(B
(defun p-draw-region-xy-internal (stream x y width height)
  (declare (inline + min max))
  (with-translate-transform-xy ((new-x new-y) stream x y)
   (with-temp-region-args ((draw-region-region) (work-region1 stream)
                           :left new-x :width width
                           :bottom new-y :height height)
      (let* ((drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (graphic-color stream))
	     (operation (graphic-operation stream))
	     (line-width (line-width stream))
	     (filled-type (filled-type stream))
	     (filled-pattern (if (filled-pattern stream)
				 0
			       (filled-pattern stream)))
	     (line-dash (line-dashing stream)))

        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-region-region)

      ;;; $B0LCV$NJQ99(B
	(setf new-x (+ new-x (world-x-start stream))
	      new-y (+ new-y (world-y-start stream)))
      
      ;;; $BIA2h%W%j%_%F%#%V(B
	(draw-region-primitive  (presentation-instance stream)
				new-x new-y width height line-width 
				operation (color-no color)
				line-dash filled-pattern filled-type
				x y)

        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
     (set-drawing-region drawing-region draw-region-region)
     stream))))

;;; draw-region-rimitive for presentation 
(defun draw-region-primitive (presentation x y width height 
			      line-width operation cl-no
			      line-dash filled-pattern filled-type
			      ox oy)
      ;;; $BIA2h%b!<%I$r7hDj(B
  (push 
   (cond
    ;; draw region without pattern
    ((eql filled-type *fillednon*)
     #'(lambda (pt-no dx dy) (yy-protocol-26 pt-no
			     (+ dx x) (+ y dy) width height line-width
			     operation cl-no line-dash)))

    ;; draw rectabglw filled color
    ((null filled-pattern)
     #'(lambda (pt-no dx dy) (yy-protocol-28 pt-no (+ x dx) (+ y dy) width height
			     operation cl-no 0)))

    ;; draw rectangle filled pattern
    (t
     #'(lambda (pt-no dx dy) (yy-protocol-28 pt-no (+ x dx) (+ y dy)
				width height operation cl-no filled-pattern))))
   (cdr (last (after-drawing-list presentation))))

  (push #'(lambda (new-stream) 
	    (draw-region-xy new-stream ox oy width height))
	(cdr (last (drawing-sequence presentation))))
  )
   
;;; $B@^$l@~$NIA2h(B      
;;; draw-polyline-xy stream xy-list
;;; ARG.
;;;          stream            = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;          xy-list           = $B0LCV(B
(defmethod p-draw-polyline-xy ((stream graphic-stream) &rest xy-list)
  (p-draw-polyline-xy-internal stream xy-list))


	    
;;; $B@^$l@~$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-polyline-xy stream position-list
;;; ARG.
;;;          stream           =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;           position-list    =  $B%]%8%7%g%s%$%s%9%?%s%974(B
(defmethod p-draw-polyline ((stream graphic-stream) &rest position-list)
  (apply #'p-draw-polyline-xy-internal stream (reduce #'nconc 
			    (map 'list #'(lambda (x) (list (position-x x)
				 			  (position-y x)))
					position-list))))

;;; draw-polyline-xy$BIA2h%W%j%_%F%#%V(B
(defun p-draw-polyline-xy-internal (stream &rest xy-list)
  (declare (inline + < > man min / car cdr))
 (with-temp-region-args ((draw-polyline-region) (work-region1 stream))
  (let* ((drawing-region (drawing-region stream))
	 (max-x most-negative-fixnum) (max-y most-negative-fixnum)
	 (min-x most-positive-fixnum) (min-y most-positive-fixnum)
	 (x-start (world-x-start stream))
	 (y-start (world-y-start stream))
	 ;; graphic-statement
	 (color (graphic-color stream))
	 (new-xy-list (copy-seq xy-list))
	 (operation (graphic-operation stream))
	 (line-width (line-width stream))
	 (line-edge (line-edge stream))
	 (line-joint-type (line-joint-type stream))
	 (line-dash (line-dashing stream)))

    ;;; $B:BI8%j%9%H$N:BI8CMJQ99(B
    (dotimes (ii (/ (length new-xy-list) 2))
       (multiple-value-bind (nx ny)
	    (translate-transform-xy stream (car new-xy-list)
				    (car (cdr new-xy-list)))

	    (if (> nx max-x)
		(setf max-x nx))

	    (if (> ny max-y)
		(setf max-y ny))

	    (if (< nx min-x)
		(setf min-x nx))
	    
	    (if (< ny min-y)
		(setf min-y ny))

	    (setf (car new-xy-list) (+ nx x-start)
		  (car (cdr new-xy-list)) (+ ny y-start)
		  new-xy-list (cddr new-xy-list))))
	    
    ;;; $BIA2h%j!<%8%g%s$r@_Dj(B
    (setf (region-left draw-polyline-region) min-x
	  (region-bottom draw-polyline-region) min-y
	  (region-right draw-polyline-region) max-x
	  (region-bottom draw-polyline-region) max-y)

    ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
    (setf (world-region stream) draw-polyline-region)

    ;;; $B@^$l@~$N%W%j%_%F%#%V(B
    (draw-polyline-primitive (presentation-instance stream)
			     new-xy-list line-width operation
			      line-edge line-joint-type 
			      (color-no color) line-dash
			      xy-list)
			      

    ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
    (set-drawing-region drawing-region draw-polyline-region)
    stream))


;;; draw-polyline-primitive for presentation 
(defun draw-polyline-primitive (presentation
				tno xy-list line-width operation
				line-edge line-joint-type 
				cl-no line-dash
				old-xy-list)
  (push #'(lambda (pt-no dx dy)
	    (yy-protocol-23 pt-no (add-value-for-xy-list xy-list dx dy)
			    line-width operation line-edge
			    line-joint-type cl-no line-dash))
	(cdr (last (after-drawing-list presentation))))
  
  (push #`(lambda (stream)
	    (draw-polyline-xy stream old-xy-list))
	(cdr (last (drawing-sequence presentation))))
)


;;; $BB?3Q7A$NIA2h(B
;;; draw-polygon-xy stream xy-list
;;; ARG.
;;;         stream            =    $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;         xy-list           =    $B0LCV(B(x y)$B74(B
(defmethod p-draw-polygon-xy ((stream graphic-stream) &rest xy-list)
  (p-draw-polygon-xy-internal stream xy-list))

;;; $BB?3Q7A$NIA2h(B $B%]%8%7%g%sMQ(B
;;; draw-polygon stream position-list
;;; ARG.
;;;           stream           =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           position-list    =  $B0LCV%$%s%9%?%s%974(B
(defmethod p-draw-polygon ((stream graphic-stream) &rest position-list)
  (apply #'p-draw-polygon-xy-internal stream (reduce #'nconc
                               (map 'list #'(lambda (x) (list (position-x x)
                                                          (position-y x)))
                                position-list)))
  )

;;; draw-polygon-xy$BIA2h%W%j%_%#%F%V(B
(defun p-draw-polygon-xy-internal (stream &rest xy-list)
  (declare (inline + < > min max / car cdr))
 (with-temp-region-args ((draw-polygon-region) (work-region1 stream))
  (let* ((drawing-region (drawing-region stream))
         (max-x most-negative-fixnum) (max-y most-negative-fixnum)
         (min-x most-positive-fixnum) (min-y most-positive-fixnum)
         (x-start (world-x-start stream))
         (y-start (world-y-start stream))
         (new-xy-list (copy-seq xy-list))
         ;; graphic-statement
         (color (graphic-color stream))
         (operation (graphic-operation stream))
         (line-width (line-width stream))
         (filled-type (filled-type stream))
         (filled-rule (filled-rule stream))
         (filled-pattern (filled-pattern stream))
         (line-joint-type (line-joint-type stream))
         (line-dash (line-dashing stream)))

    ;;; $B:BI8%j%9%H$N:BI8CMJQ99(B
    (dotimes (ii (/ (length new-xy-list) 2))
       (multiple-value-bind (nx ny)
            (translate-transform-xy stream (car new-xy-list)
                                    (car (cdr new-xy-list)))

            (if (> nx max-x)
                (setf max-x nx))

            (if (> ny max-y)
                (setf max-y ny))

            (if (< nx min-x)
                (setf min-x nx))

            (if (< ny min-y)
                (setf min-y ny))

            (setf (car new-xy-list) (+ nx x-start)
                  (car (cdr new-xy-list)) (+ ny y-start)
                  new-xy-list (cddr new-xy-list))))

    ;;; $BIA2h%j!<%8%g%s$r@_Dj(B
    (setf (region-left draw-polygon-region) min-x
          (region-bottom draw-polygon-region) min-y
          (region-right draw-polygon-region) max-x
          (region-bottom draw-polygon-region) max-y)

    ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
    (setf (world-region stream) draw-polygon-region)

    ;;; $BIA2h%W%j%_%F%#%V(B
    (draw-polygon-primitive  (presentation-instance stream)
			     new-xy-list line-width operation
			     line-joint-type (color-no color) line-dash
			     filled-rule filled-pattern filled-type
			     xy-list)

    ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
    (set-drawing-region drawing-region draw-polygon-region)
    stream)))

;;; draw-polyline-primitive for presentation 
(defun draw-polygon-primitive (presentation
				xy-list line-width operation
				line-edge line-joint-type 
				cl-no line-dash
				old-xy-list)
  (push 
   (cond
    ((eql filled-type *fillednon*)
    #'(lambda (pt-no dx dy)
	    (yy-protocol-24 pt-no (add-value-for-xy-list xy-list dx dy)
			    line-width operation 
			    line-joint-type cl-no line-dash)))

    ((null filled-pattern)
     #'(lambda (pt-no dx dy)
            (yy-protocol-27 pt-no (add-value-for-xy-list xy-list dx dy)
			    operation line-joint-type
			    cl-no  filled-rule 0)))
    (t
     #'(lambda (pt-no dx dy)
            (yy-protocol-27 pt-no (add-value-for-xy-list xy-list dx dy)
			    operation line-joint-type
			    cl-no filled-rule filled-pattern))))

   (cdr (last (after-drawing-list presentation))))
  
  (push #`(lambda (stream)
	    (draw-polygon-xy stream old-xy-list))
	(cdr (last (drawing-sequence presentation))))
  )


;;; $B;03Q7A$NIA2h(B   ($BFsEyJU;03Q7A(B)
;;; p-draw-triangle-xy stream x y width height
;;; ARG.
;;;           stream            =   $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;           x y               =   $BDcJU$NCf?40LCV(B
;;;           width             =   $BDcJU$NI}(B
;;;           height            =   $B9b$5(B
(defmethod p-draw-triangle-xy ((stream graphic-stream) (x integer) (y integer)
                             (width integer) (height integer))
  (p-draw-triangle-xy-internal stream x y width height))

;;; draw-triangle-xy $BIA2h%W%j%_%F%#%t(B
(defun p-draw-triangle-xy-internal (stream x y width height)
  (declare (inline - / + round))
  (p-draw-polygon-xy-internal stream
                   (- x (round (/ width 2))) y (+ x (round (/ width 2))) y
                   x (+ y height))
  )

;;; $B;03Q7A$NIA2h(B $B%]%8%7%g%sMQ(B ($BFsEyJU;03Q7A(B)
;;; p-draw-triangle stream position width height
;;; ARG.
;;;              stream            = $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;              position          = $BDcJU$NCf?40LCV%$%s%9%?%s%9(B
;;;              width             = $B;03Q7A$NI}(B
;;;              height            = $B9b$5(B
(defmethod p-draw-triangle ((stream graphic-stream) (position position)
                          (width integer) (height integer))
  (p-draw-triangle-xy-internal stream (position-x position) (position-y position)
                   width height))

;;; $BBJ1_!"BJ1_8L!"BJ1_$N@p7?$NIA2h(B
;;; p-draw-ellipse-xy stream center-x center-y x-radius y-radius 
;;;                          &key (start-angle 0)
;;;                               (end-angle 2pi) (clockwize :clockwize)
;;; ARG.
;;;               stream     =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;     center-x center-y    =  $BBJ1_$NCf?4(B
;;;               x-radius   =  $B2#<4$K$*$1$kH>7B(B
;;;               y-radius   =  $B=D<4$K$*$1$kH>7B(B
;;;          start-angle     =  $B3+;O3QEY(B($B%i%G%#%"%s(B)
;;;          end-angle       =  $B=*N;3QEY(B($B%i%G%#%"%s(B)
;;;          clockwize       =  $BIA2hJ}8~(B
(defmethod p-draw-ellipse-xy ((stream graphic-stream) 
			    (center-x integer) (center-y integer)
			    (x-radius integer) (y-radius integer) 
			    &key (start-angle 0)
			    (end-angle (* 2 pi)) (clockwize :clockwize))
  (p-draw-ellipse-xy-internal stream center-x center-y 
			    x-radius y-radius 
			    :start-angle start-angle
			    :end-angle end-angle
			    :clockwize clockwize))

;;; $BBJ1_!"BJ1_8L!"BJ1_$N@p7?$NIA2h(B
;;; draw-ellipse   stream posistion x-radius y-radius 
;;;                         &key (start-angle 0)
;;;                               (end-angle 2pi) (clockwize :clockwize)
;;; ARG.
;;;               stream     =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;               position   =  $BBJ1_$NCf?4$N0LCV%$%s%9%?%s%9(B
;;;               x-radius   =  $B2#<4$K$*$1$kH>7B(B
;;;               y-radius   =  $B=D<4$K$*$1$kH>7B(B
;;;          start-angle     =  $B3+;O3QEY(B($B%i%G%#%"%s(B)
;;;          end-angle       =  $B=*N;3QEY(B($B%i%G%#%"%s(B)
;;;          clockwize       =  $BIA2hJ}8~(B
(defmethod p-draw-ellipse ((stream graphic-stream) (position position)
			 (x-radius integer) (y-radius integer) &key (start-angle 0)
			    (end-angle (* 2 pi)) (clockwize :clockwize))
  (p-draw-ellipse-xy-internal
                   stream (position-x position) (position-y position)
		   x-radius y-radius :start-angle start-angle
		   :end-angle end-angle :clockwize clockwize))


;;; draw-ellipse-xy$BIA2h%W%j%_%F%#%V(B
(defun p-draw-ellipse-xy-internal (stream center-x center-y x-radius y-radius
				 &key (start-angle 0)
				 (end-angle (* 2 pi))
				 (clockwize :clockwize))
  (declare (inline * + - max min))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-ellipse : Unkown clockwize ~s" clockwize)))
    (with-translate-transform-xy ((new-x new-y) stream center-x center-y)
      (with-temp-region-args ((draw-ellipse-region) (work-region1 stream)
                           :left (- new-x x-radius) :right (+ new-x x-radius)
                           :top (+ new-y y-radius) :bottom (- new-y y-radius))
      (let* ((drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (graphic-color stream))
	     (operation (graphic-operation stream))
	     (width (line-width stream))
	     (dash (line-dashing stream))
	     (filled-type (filled-type stream))
	     (filled-pattern (filled-pattern stream))
	     (arc-mode (arc-mode stream)))
        ;; 1/64 $B$r$?$s$$$H$9$k!#(B
        (setf start-angle (floor (* (/ start-angle pi) 180 64))
  	      end-angle (floor (* (/ end-angle pi) 180 64)))

        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-ellipse-region)

      ;;; $B0LCV$NJQ99(B
      (setf new-x (+ new-x (world-x-start stream))
	    new-y (+ new-y (world-y-start stream)))

      ;;; $BIA2h%W%j%_%F%#%V(B
      (draw-ellipse-primitive (presentation-instance stream)
			      new-x new-y x-radius y-radius
			      start-angle end-angle width operation
			      (color-no color) dash arc-mode 
			      filled-pattern filled-type
			      x y)
        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
      (set-drawing-region drawing-region draw-ellipse-region)
      stream))))


;;; draw-ellipse-primitive for world
(defun draw-ellipse-primitive (presentation new-x new-y x-radius y-radius
				      start-angle end-angle width operation
				      cl-no dash arc-mode filled-pattern 
				      filled-type old-x old-y)
  (push 
   ;;; call yy-protocol-41
   (cond
    ;; draw circle without filled
    ((eql filled-type *fillednon*)
     #'(lambda (pt-no dx dy) (yy-protocol-41 pt-no (+ new-x dx)
					     (+ new-y  dy)
					     (* 2 x-radius) (* 2 y-radius) 
					     start-angle end-angle
					     width operation cl-no dash)))

    ;; draw circle filled color
    ((null filled-pattern)
     #'(lambda (pt-no dx dy) (yy-protocol-42 pt-no (+ new-x dx)
					     (+ new-y dy)
					     (* 2 x-radius) (* 2 y-radius)
					     start-angle end-angle
					     operation cl-no 0 arc-mode)))

    ;; draw circle filled pattern
    (t
     #'(lambda (pt-no dx dy) (yy-protocol-42 pt-no (+ new-x dx)
				      (+ new-y dy)
				      (* 2 x-radius) (* 2 y-radius)
				      start-angle end-angle
				      operation cl-no filled-pattern arc-mode))))
   (cdr (last (after-drawing-list presentation))))

  (push 
   #'(lambda (new-stream) 
       (draw-ellipse-xy new-stream old-x old-y x-radius y-radius
			:start-angle start-angle :end-angle end-angle
			:clockwize clockwize)
       (cdr (last (drawing-sequence presentation)))))
   )


;;; $B1_!"@p7?!"1_8L$NIA2h(B
;;; p-draw-circle-xy stream center-x center-y radius &key (start-angle 0)
;;;		(end-angle 2pi) clockwize
;;; ARG.
;;;           stream             =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;       center-x center-y      =  $B1_$NCf?4(B
;;;          radius              =  $B1_$NH>7B(B
;;;          start-angle         =  $B3+;O3QEY(B
;;;          end-angle           =  $B=*N;3QEY(B
;;;          clockwize           =  $BIA2hJ}8~(B
(defmethod p-draw-circle-xy ((stream graphic-stream) (center-x integer)
			   (center-y integer) (radius integer)
			   &key (start-angle 0) (end-angle (* 2 pi))
			   (clockwize :clockwize))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-circle : Unkown clockwize ~s" clockwize)))

    ;;; draw-circle-intenal
    (p-draw-circle-xy-internal stream 
			       center-x center-y radius start-angle end-angle))


;;; $B1_!"@p7?!"1_8L$NIA2h(B
;;; draw-circle stream position radius &key (start-angle 0)
;;;		(end-angle 2pi) clockwize
;;; ARG.
;;;           stream             =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;          position            =  $B1_$NCf?40LCV$N%$%s%9%?%s%9(B
;;;          radius              =  $B1_$NH>7B(B
;;;          start-angle         =  $B3+;O3QEY(B
;;;          end-angle           =  $B=*N;3QEY(B
;;;          clockwize           =  $BIA2hJ}8~(B
(defmethod draw-circle ((stream graphic-stream) (position position)
			(radius integer) &key (start-angle 0)
			(end-angle (* 2 pi)) (clockwize :clockwize))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-circle : Unkown clockwize ~s" clockwize)))

  (draw-circle-xy-internal
                  stream (position-x position) (position-y position) radius
                  start-angle end-angle))

;;; $B1_!"@p7?!"1_8L$NIA2h(B
;;; draw-circle stream position radius &key (start-angle 0)
;;;		(end-angle 2pi) clockwize
;;; ARG.
;;;           stream             =  $B%0%i%U%#%/%9%9%H%j!<%`(B
;;;          position            =  $B1_$NCf?40LCV$N%$%s%9%?%s%9(B
;;;          radius              =  $B1_$NH>7B(B
;;;          start-angle         =  $B3+;O3QEY(B
;;;          end-angle           =  $B=*N;3QEY(B
;;;          clockwize           =  $BIA2hJ}8~(B
(defmethod p-draw-circle ((stream graphic-stream) (position position)
			(radius integer) &key (start-angle 0)
			(end-angle (* 2 pi)) (clockwize :clockwize))
    (case clockwize
      ((:clockwize))
      ((:counter-clockwize)
       (let ((tmp start-angle))
         (setf start-angle end-angle
	       end-angle tmp)))
      (otherwise (error "Draw-circle : Unkown clockwize ~s" clockwize)))

  (p-draw-circle-xy-internal
                  stream (position-x position) (position-y position) radius
                  start-angle end-angle))


;;; p-draw-circle-xy-internal stream center-x center-y radius start-angle
;;;				 end-angle
(defun p-draw-circle-xy-internal (stream center-x center-y radius start-angle
				end-angle)
  (declare (inline + * max min))
  (with-translate-transform-xy ((new-x new-y) stream center-x center-y)
    (with-temp-region-args ((draw-circle-region) (work-region1 stream))
       (setf draw-circle-region (get-draw-circle-region stream new-x new-y
					   radius start-angle end-angle
					   draw-circle-region))
       (let* ((drawing-region (drawing-region stream))
	     ;; graphic-statement
	     (color (graphic-color stream))
	     (operation (graphic-operation stream))
	     (width (line-width stream))
	     (dash (line-dashing stream))
	     (filled-type (filled-type stream))
	     (filled-pattern (filled-pattern stream))
	     (arc-mode (arc-mode stream)))
        ;; 1/64 $B$r$?$s$$$H$9$k!#(B
        (setf start-angle (floor (* (/ start-angle pi) 180 64))
  	      end-angle (floor (* (/ end-angle pi) 180 64)))

      
        ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	(setf (world-region stream) draw-circle-region)

      ;;; $B0LCV$NJQ99(B
	(setf new-x (+ new-x (world-x-start stream))
	      new-y (+ new-y (world-y-start stream)))

        ;;; call draw-circle-primitive
	(draw-circle-primitive (presentation-instance stream)
			       new-x new-y radius start-angle end-angle
			       width operation (color-no color) dash 
			       arc-mode filled-pattern filled-type
			       center-x center-y)
	
        ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
	(set-drawing-region drawing-region draw-circle-region)
	stream))))


;;; draw-circle-primitive for presentation 
(defun draw-circle-primitive (presentation
			      new-x new-y radius start-angle end-angle
			      width operation cl-no dash arc-mode 
			      filled-pattern filled-type old-x old-y)

  (push 
   ;;; call yy-protocol-21
   (cond
    ;; draw circle without filled
    ((eql filled-type *fillednon*)
     #'(lambda (pt-no dx dy) (yy-protocol-25 pt-no (+ new-x dx)
					     (+ new-y  dy)
					     radius start-angle end-angle
					     width operation cl-no dash)))

    ;; draw circle filled color
    ((null filled-pattern)
     #'(lambda (pt-no dx dy) (yy-protocol-30 pt-no (+ new-x dx)
					     (+ new-y dy)
					     radius start-angle end-angle
					     operation cl-no 0 arc-mode)))

    ;; draw circle filled pattern
    (t
     #'(lambda (pt-no dx dy) (yy-protocol-30 pt-no (+ new-x dx)
				      (+ new-y dy)
				      radius start-angle end-angle
				      operation cl-no filled-pattern arc-mode))))
  (cdr (last (after-drawing-list presentation))))


  (push 
	#'(lambda (new-stream) 
	    (draw-circle-xy new-stream old-x old-y radius
			    :start-angle start-angle :end-angle end-angle
			    :clockwize clockwize))
	(cdr (last (drawing-sequence presentation))))
   )


;;; $B%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rE>Aw$9$k!#(BXY$BMQ(B
;;; p-put-image-xy stream  image x y &optinal width height 
;;;                                &key (image-x 0) (image-y 0)
;;; ARG.
;;;       stream          = $B%0%i%U%#%/%9%H%j!<%`(B
;;;       x y             = $B%$%a!<%8$rIA2h$9$k0LCV(B
;;;       width           = $BIA2h$9$kI}(B
;;;       height          = $BIA2h$9$k9b$5(B
;;;       image-x         = $B%$%a!<%8Fb$N3+;O0LCV(B
;;;       image-y         = $B%$%a!<%8Fb$N3+;O0LCV(B
(defmethod p-put-image-xy ((stream graphic-stream) (image image)
			 (x integer) (y integer) 
			 &optional (width (image-width image))
			 (height (image-height image))
			 &key (image-x 0) (image-y 0))
  (p-put-image-xy-internal stream image x y width height image-x image-y))

;;; $B%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rE>Aw$9$k!#%]%8%7%g%sMQ(B
;;; p-put-image stream  image position width height &key (image-x 0) (image-y 0)
;;; ARG.
;;;       stream          = $B%0%i%U%#%/%9%H%j!<%`(B
;;;       position        = $B%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;       width           = $BIA2h$9$kI}(B
;;;       height          = $BIA2h$9$k9b$5(B
;;;       image-x         = $B%$%a!<%8Fb$N3+;O0LCV(B
;;;       image-y         = $B%$%a!<%8Fb$N3+;O0LCV(B
(defmethod p-put-image ((stream graphic-stream) (image image)
		      (position position) &optional
		      (width (image-width image))
		      (height (image-height image)) &key (image-x 0) (image-y 0))
  (put-image-xy-internal stream image (position-x position)
			 (position-y position) width height image-x image-y))

;;; $B%S%C%H%^%C%W$K%$%a!<%8$rE>Aw$9$k!#(BXY$BMQ(B
;;; p-put-image-xy bitmap  image x y &optinal width height 
;;;                                &key (image-x 0) (image-y 0)
;;; ARG.
;;;       bitmap          = $B%S%C%H%^%C%W(B
;;;       x y             = $B%$%a!<%8$rIA2h$9$k0LCV(B
;;;       width           = $BIA2h$9$kI}(B
;;;       height          = $BIA2h$9$k9b$5(B
;;;       image-x         = $B%$%a!<%8Fb$N3+;O0LCV(B
;;;       image-y         = $B%$%a!<%8Fb$N3+;O0LCV(B
(defmethod p-put-image-xy ((bitmap bitmap) (image image)
			 (x integer) (y integer) 
			 &optional (width (image-width image))
			 (height (image-height image))
			 &key (image-x 0) (image-y 0))
  (put-image-xy-bitmap bitmap image x y width height image-x image-y))

;;; $B%S%C%H%^%C%W$K%$%a!<%8$rE>Aw$9$k!#(B
;;; p-put-image bitmap image position width height &key (image-x 0) (image-y 0)
;;; ARG.
;;;       bitmap          = $B%S%C%H%^%C%W(B
;;;       position        = $B%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;       width           = $BIA2h$9$kI}(B
;;;       height          = $BIA2h$9$k9b$5(B
;;;       image-x         = $B%$%a!<%8Fb$N3+;O0LCV(B
;;;       image-y         = $B%$%a!<%8Fb$N3+;O0LCV(B
(defmethod p-put-image ((bitmap bitmap) (image image)
		      (position position) &optional
		      (width (image-width image))
		      (height (image-height image)) &key (image-x 0) (image-y 0))
  (put-image-xy-bitmap bitmap image (position-x position)
			 (position-y position) width height image-x image-y))


;;; put-image$BIA2h%W%j%_%F%#%V(B $B%9%H%j!<%`MQ(B
(defun p-put-image-xy-internal (stream image x y width height image-x image-y)
  (let* ((image-data (image-data image))
	 (old-width (image-width image))
	 (old-line-bytes (line-bytes image))
	 (old-height (image-height image))
	 (new-image (clip-image-xy image image-x image-y width height
				   :create nil)))
    
    (p-put-image-internal-xy stream new-image x y width height image-x image-y)
    
    (setf (slot-value image 'image-width) old-width
	  (line-bytes image) old-line-bytes
	  (slot-value image 'image-height) old-height
	  (slot-value image 'image-data) image-data)
    stream))

;;; $B%$%a!<%8$NI=<(%$%s%?!<%J%k(B $B%9%H%j!<%`MQ(B
(defun p-put-image-internal-xy (stream image x y width height ix iy)
  (declare (inline + min max =))
  (with-translate-transform-xy ((new-x new-y) stream x y)
    (with-translate-transform-xy ((x1 y1) stream (+ x width) (+ y height))
      (with-temp-region-args ((draw-image-region) (work-region1 stream)
                           :left (min new-x x1) :width width
                           :bottom (min new-y y1) :height height)
       (let* ((drawing-region (drawing-region stream))
	      (format1 (if (eq (image-format image) :yy)
			   2
			 1))
	   (yy (with-translate-coordinate-stream y stream))
	   (def-y (if (= yy y)
		       0
		      height))
	   (format (case (image-type image)
		     (:color
		      (logior #x8000 format1))
		     (:gray
		      (logior #x4000 format1))
		     (t
		      (logior #x2000 format1)))))

          ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	 (setf (world-region stream) draw-image-region)

         ;;; $B0LCV$NJQ99(B
	 (setf new-x (+ new-x (world-x-start stream))
	       new-y (- (+ new-y (world-y-start stream)) def-y))
      
          ;;; $BIA2h%W%j%_%F%#%V(B
	 (push #'(lambda (ptno dx dy)
		   (yy-protocol-61 ptno (+ new-x dx) (+ new-x dy) 
				   (image-width image) (image-height image)
				   format (image-data image)))
	       (cdr (last (after-drawing-list (presentation-instance stream)))))

          ;;; $BIA2h%7!<%1%s%9$r5-21(B
	 (push #'(lambda (new-stream)
		   (put-image-xy new-stream image x y
				 width height :image-x ix :image-y iy))
				 
	       (cdr (last  (drawing-sequence (presentation-instance stream)))))
	 
          ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
	 (set-drawing-region drawing-region draw-image-region)
	 stream)))))



;;; $B%0%i%U%#%/%9%H%j!<%`$K%0%i%U%#%/%9%9%H%j!<%`$N%$%a!<%8$rIA2h$9$k!#(B
;;; p-draw-image distination-stream source-stream position width height
;;;            &key image-position
;;; ARG.
;;;       distination-stream   = $B%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(B
;;;       source-stream        = $B%=!<%9%0%i%U%#%/%9%H%j!<%`(B
;;;       position             = $B%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;       width                = $BIA2h$9$kI}(B
;;;       height               = $BIA2h$9$k9b$5(B
;;;       image-position       = $B%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV$N(B
;;;                              $B%$%s%9%?%s%9(B
(defmethod p-draw-image ((d-stream graphic-stream) (s-stream graphic-stream)
		       (position position) (width integer) (height integer)
		       &key (image-position nil))
  (if image-position
    (p-draw-image-stream-xy-internal d-stream s-stream
				   (position-x position)
				   (position-y position)
				   width height
				   (position-x image-position)
				   (position-y image-position))
    (p-draw-image-stream-xy-internal d-stream s-stream
                                   (position-x position)
                                   (position-y position)
                                   width height 0 0)))


(defmethod p-draw-image :after ((d-stream graphic-stream) s-stream
				position width height &key (image-position nil))

  	   ;;; $BIA2h%7!<%1%s%9$r5-21(B
  (push #'(lambda (new-stream)
	    (draw-image new--stream s-stream position  width height
			:image-position image-position))
	(cdr (last  (drawing-sequence presentation)))))


;;; $B%0%i%U%#%/%9%H%j!<%`$K%0%i%U%#%/%9%9%H%j!<%`$N%$%a!<%8$rIA2h$9$k!#(BXY$BMQ(B
;;; p-draw-image-xy distination-stream source-stream dx dy width height
;;;            &key image-x image-y
;;; ARG.
;;;       distination-stream   = $B%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(B
;;;       source-stream        = $B%=!<%9%0%i%U%#%/%9%H%j!<%`(B
;;;       dx dy                = $B%$%a!<%8$rIA2h$9$k0LCV(B
;;;       width                = $BIA2h$9$kI}(B
;;;       height               = $BIA2h$9$k9b$5(B
;;;       image-x image-y      = $B%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV(B
(defmethod p-draw-image-xy ((d-stream graphic-stream) (s-stream graphic-stream)
			  (dx integer) (dy integer) (width integer) 
			  (height integer) &key (image-x 0) (image-y 0))
  (p-draw-image-stream-xy-internal d-stream s-stream
				 dx dy
				 width height
				 image-x image-y))


(defmethod p-draw-image-xt :after ((d-stream graphic-stream) s-stream
				dx dy width height &key (image-x 0) (image-y 0))

  	   ;;; $BIA2h%7!<%1%s%9$r5-21(B
  (push #'(lambda (new-stream)
	    (draw-image-xy new-stream s-stream dx dy  width height
			:image-x image-x :image-y image-y)
	(cdr (last  (drawing-sequence presentation))))))



;;; $B%0%i%U%#%/%9%H%j!<%`$K%S%C%H%^%C%W$N%$%a!<%8$rIA2h$9$k!#(B
;;; p-draw-image distination-stream source-bitmap position width height
;;;            &key image-position
;;; ARG.
;;;       distination-stream   = $B%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(B
;;;       source-bitmap        = $B%=!<%9%S%C%H%^%C%W(B
;;;       position             = $B%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;       width                = $BIA2h$9$kI}(B
;;;       height               = $BIA2h$9$k9b$5(B
;;;       image-position       = $B%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV$N(B
;;;                              $B%$%s%9%?%s%9(B
(defmethod p-draw-image ((d-stream graphic-stream) (s-bitmap bitmap)
		       (position position) (width integer) (height integer)
		       &key (image-position nil))
  (if image-position
      (multiple-value-bind (image-data real-width real-height)
          (get-image-data-bitmap s-bitmap (position-x image-position)
				 (position-y image-position)
				 width height)
	  (let* ((line-byte (if (= (* real-width real-height 4)
				   (length image-data))
				(* real-width 4)
			      (ceiling (/ real-width 4)))))
	    (p-image-to-graphic-stream d-stream (position-x position)
				   (position-y position) 
				   real-width real-height 
				   image-data
				   line-byte
				   (if (= line-byte (* real-width 4))
				       :color
				     :mono)
				   0 0)))
    (multiple-value-bind (image-data real-width real-height)
	(get-image-data-bitmap s-bitmap 0 0 width height)
	(let ((line-byte (if (= (* real-width real-height 4)
                                   (length image-data))
                                (* real-width 4)
                              (ceiling (/ real-width 4)))))
	  (p-image-to-graphic-stream d-stream (position-x position)
                                   (position-y position)
                                   real-width real-height
                                   image-data
				   line-byte
				   (if (= line-byte (* real-width 4))
                                       :color
                                     :mono)
				   0 0)))
    ))


;;; $B%0%i%U%#%/%9%H%j!<%`$K%S%C%H%^%C%W$N%$%a!<%8$rIA2h$9$k!#(BXY$BMQ(B
;;; p-draw-image-xy distination-stream source-bitmap dx dy width height
;;;            &key image-x image-y
;;; ARG.
;;;       distination-stream   = $B%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(B
;;;       source-bitmap        = $B%=!<%9%S%C%H%^%C%W(B
;;;       dx dy                = $B%$%a!<%8$rIA2h$9$k0LCV(B
;;;       width                = $BIA2h$9$kI}(B
;;;       height               = $BIA2h$9$k9b$5(B
;;;       image-x image-y      = $B%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV(B
(defmethod p-draw-image-xy ((d-stream graphic-stream) (s-bitmap bitmap)
			  (dx integer) (dy integer) (width integer) 
			  (height integer) &key (image-x 0) (image-y 0))
  (multiple-value-bind (image-data real-width real-height)
    (get-image-data-bitmap s-bitmap image-x image-y width height)
    (let ((line-byte (if (= (* real-width real-height 4)
                                   (length image-data))
                                (* real-width 4)
                              (ceiling (/ real-width 4)))))
      (p-image-to-graphic-stream d-stream dx dy real-width real-height image-data
			       line-byte
			       (if (= line-byte (* real-width 4))
                                       :color
                                     :mono)
			       0 0))
    ))

;;; $B%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rIA2h$9$k!#(B
;;; p-draw-image distination-stream source-image position width height
;;;            &key image-position
;;; ARG.
;;;       distination-stream   = $B%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(B
;;;       source-image         = $B%=!<%9%$%a!<%8(B
;;;       position             = $B%$%a!<%8$rIA2h$9$k0LCV$N%$%s%9%?%s%9(B
;;;       width                = $BIA2h$9$kI}(B
;;;       height               = $BIA2h$9$k9b$5(B
;;;       image-position       = $B%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV$N(B
;;;                              $B%$%s%9%?%s%9(B
(defmethod p-draw-image ((d-stream graphic-stream) (s-image image)
		       (position position) (width integer) (height integer)
		       &key (image-position nil))
  (let* ((x (if image-position
		(position-x image-position)
	      0))
	 (y (if image-position
		(position-y image-position)
	      0))
	 (real-width (if (< (+ width x) (image-width s-image))
			 width
		       (- (image-width s-image) x)))
	 (real-height (if (< (+ height y) (image-height s-image))
			  height
			(- (image-height s-image) y))))

    (p-image-to-graphic-stream d-stream (position-x position)
			     (position-y position) 
			     real-width real-height
			     (image-data s-image)
			     (line-bytes s-image)
			     (image-type s-image)
			     x y)
    d-stream))



;;; $B%0%i%U%#%/%9%H%j!<%`$K%$%a!<%8$rIA2h$9$k!#(BXY$BMQ(B
;;; p-draw-image-xy distination-stream source-image dx dy width height
;;;            &key image-x image-y
;;; ARG.
;;;       distination-stream   = $B%G%#%9%F%#%M!<%7%g%s%0%i%U%#%/%9%H%j!<%`(B
;;;       source-image         = $B%=!<%9%$%a!<%8(B
;;;       dx dy                = $B%$%a!<%8$rIA2h$9$k0LCV(B
;;;       width                = $BIA2h$9$kI}(B
;;;       height               = $BIA2h$9$k9b$5(B
;;;       image-x image-y      = $B%=!<%9%0%i%U%#%/%9%H%j!<%`Fb$G$N0LCV(B
(defmethod p-draw-image-xy ((d-stream graphic-stream) (s-image image)
			  (dx integer) (dy integer) (width integer) 
			  (height integer) &key (image-x 0) (image-y 0))
  (let ((real-width (if (< (+ width image-x) (image-width s-image))
                         width
                       (- (image-width s-image) image-x)))
         (real-height (if (< (+ height image-y) (image-height s-image))
                          height
                        (- (image-height s-image) image-y))))

    (p-image-to-graphic-stream d-stream dx dy
			     real-width real-height
                             (image-data s-image)
                             (line-bytes s-image)
                             (image-type s-image)
                             image-x image-y)

    d-stream))


;;; $B%9%H%j!<%`MQ$N%$%a!<%8$NIA2h(B
(defun p-draw-image-stream-xy-internal (d-stream s-stream dx dy width height
                                      sx sy)
  (multiple-value-bind (image-data real-width real-height)
		       (get-image-xy-stream s-stream sx sy width height)
    (let ((line-byte (if (= (length image-data) (* real-width real-height))
			 (* real-width 4)
		       (ceiling (/ real-width 4)))))
      (p-image-to-graphic-stream d-stream dx dy real-width real-height image-data
			       line-byte
			       (if (= line-byte (* real-width 4))
				   :color
				 :mono)
			       0 0))
    ))

;;; $B%$%a!<%8%G!<%?$r%9%H%j!<%`$KAw$k(B
(defun p-image-to-graphic-stream (stream dx dy width height image-data line-byte
				       type sx sy)
  (declare (inline + * / incf logior ash)
	   (special *black-color* *white-color* *transparent*))
  (with-translate-transform-xy ((x1 y1) stream dx dy)
    (let* ((drawing-region (drawing-region stream))
	   (count 0)
	   (x2 0)
	   (x3 0)
	   (x4 0)
	   (y2 0)
	   (y3 0)
	   (y4 0)
	   (yy (with-translate-coordinate-stream dy stream))
	   (real-y (if (= yy dy)
		       yy
		     (- yy height))))

      (multiple-value-setq (x2 y2)
	   (get-new-position-xy (+ dx width) 
				(with-translate-coordinate-stream dy stream)
				stream))
	 (multiple-value-setq (x3 y3)
	   (get-new-position-xy dx (+ (with-translate-coordinate-stream dy 
						    stream)
				      height)
				stream))
	 (multiple-value-setq (x4 y4)
	   (get-new-position-xy (+ dx width)
				(+ (with-translate-coordinate-stream dy
						     stream)
				   height)
				stream))
				      
	 (with-temp-region-args ((draw-image-region) (work-region1 stream)
			     :left (min x1 x2 x3 x4)
			     :top (max y1 y2 y3 y4)
			     :right (max x1 x2 x3 x4)
			     :bottom (min y1 y2 y3 y4))
				
            ;;; $B%o!<%k%I%j!<%8%g%s$NJQ99(B
	   (setf (world-region stream) draw-image-region)

	    ;;; drawing-region$B$K:#$N%j!<%8%g%s$rDI2C(B
	   (set-drawing-region drawing-region draw-image-region))
	 
	 (push #'(lambda (ptno dx dy)
		   (p-image-drawing stream image-data dx real-y width height
				    sx sy ptno ddx ddy))
	       (cdr (last (after-drawing-list (presentation-instance stream))))
	       )
	 )))

(defun p-image-drawing (stream image-data dx real-y width height 
			       sx sy ptno ddx ddy)
  (let ((x1 0) (y1 0) (count 0)
	(black-no (color-no *black-color*))
	(white-no (color-no *white-color*)))
    (case type
	  (:color 	 ;;; $B%+%i!<(B
	       ;;; $BIA2h(B
	   (setf y1 sy)
	   (do ((y real-y (incf y)))
	       ((= y (+ height real-y)))
	       (setf x1 sx)
	       (do ((x dx (incf x)))
		   ((= x (+ dx width)))

		   (setf count (+ (* line-byte y1) (* x1 4)))

		   (multiple-value-bind (xx yy) (get-new-position-xy x y stream)
					(incf xx (world-x-start stream))
					(incf yy (world-y-start stream))
		(yy-protocol-20
		   ptno (+ xx ddx) (+ yy ddy)
		   *GCOPY* (logior (ash (elt image-data count) 24)
				   (ash (elt image-data (+ count 1)) 16)
				   (ash (elt image-data (+ count 2)) 8)
				   (elt image-data (+ count 3))))
		   )
		(incf x1))
	      (incf y1)))
	   (t ;;; $B%b%N%/%m(B
	    (setf y1 sy)
	    (do ((y real-y(incf y)))
		((= y (+ height real-y)))
	      (setf x1 sx)
	      (do ((x dx (incf x)))
		  ((= x (+ width dx)))
		
		(multiple-value-bind (xx yy) (get-new-position-xy x y stream)
		  (incf xx (world-x-start stream))
		  (incf yy (world-y-start stream))
		  (yy-protocol-20
		   ptno (+ xx ddx) (+ yy ddy) *GSET*
		   (case (yy-image-bit image-data line-byte x1 y1)
		     (1 black-no)
		     (0 white-no)
		     (2 -1))))
		(incf x1))
	      (incf y1)))
	   )))

