;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; bitmap.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.3 91/01/23 by t.kosaka

;;; $B%S%C%H%^%C%W%/%i%9(B

(in-package :yy)

;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; $B%S%C%H%^%C%W%/%i%9(B ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;
(defclass bitmap ()
  ((bitmap-territory-no :initarg :bitmap-territory-no
			:accessor bitmap-territory-no
			:initform 0)
   (bitmap-width :initarg :bitmap-width
		 :initform 0 :reader bitmap-width)
   (bitmap-height :initarg :bitmap-height
		  :initform 0 :reader bitmap-height)))

;;; print-object
(defmethod print-object ((bitmap bitmap) stream)
  (format stream "\#<Bitmap size ~ax~a>" (bitmap-width bitmap)
	  (bitmap-height bitmap)))

;;; $B%S%C%H%^%C%W$+(B
;;; bitmapp object
;;; ARG.
;;;         object  = $B%S%C%H%^%C%W%$%s%9%?%s%9(B
;;; RET.
;;;      T or NIL
(defun bitmapp (object)
  (typep object 'bitmap))


;;; $B%S%C%H%^%C%W$N@8@.(B
;;; make-bitmap width height &key image
;;; ARG.
;;;         width   =  $BI}(B
;;;         height  =  $B9b$5(B
;;;         image   =  $B%$%a!<%8%$%s%9%?%s%9(B
(defun make-bitmap (width height &key (image nil))
  (declare (function image-data (t) vector)
	   (function put-image-xy-bitmap (t t integer integer 
					    integer integer integer 
					    integer) T))
  (let* ((instance (make-instance 'bitmap
				  :bitmap-width width
				  :bitmap-height height))
#+ :YY2.0
	 (tno (with-object-make-territory instance
		       :width width :height height
		       :visible nil
		       :transparent T))
#- :YY2.0
         (tno (with-object-make-territory instance
                       :width width :height height
                       :visible nil))
	 )
    (setf (bitmap-territory-no instance) tno)
    ;;; $B%$%a!<%8$rE>Aw(B
    (if image
	(put-image-xy-bitmap instance image 0 0 width height 0 0))
    instance))
      
;;; $B%S%C%H%^%C%W$N>C5n(B
;;; flush-bitmap bitmap
;;; ARG.
;;;       bitmap  = $B%S%C%H%^%C%W%$%s%9%?%s%9(B
(defmethod flush-bitmap ((bitmap bitmap))
  (let ((val (yy-protocol-5 (bitmap-territory-no bitmap))))
    (if (zerop val)
	(error "Sorry ,can not remove ~a" bitmap)
      (progn 
	(delete-lisp-object (bitmap-territory-no bitmap))
      nil)))
  )
  
;;; $B%S%C%H%^%C%W$N3HBg!?=L>.(B
;;; expange-bitmap bitmap x-ratio y-ratio &key create
;;; ARG.
;;;         bitmap      =  $BBP>]$J$k%S%C%H%^%C%W%$%s%9%?%s%9(B
;;;         x-ratio     =  $B?eJ?J}8~$N3HBg(B/$B=L>.G\N((B 
;;;         y-ratio     =  $B?bD>J}8~$N3HBg(B/$B=L>.G\N((B
;;;                        $B<B:]$K3HBg!?=L>.$5$l$kC10L$O(B1/100$B$G$"$k!#(B
;;;                        $B$=$l$h$j:Y$+$$$b$N$O4]$a$i$l$k(B
;;;         create      =  $B%S%C%H%^%C%W$r?7$?$K@8@.$9$k$+(B
;;;                         T -> $B@8@.(B
;;; RET.
;;;         $B%S%C%H%^%C%W%$%s%9%?%s%9(B
(defmethod expange-bitmap ((bitmap bitmap) x-ratio y-ratio 
			   &key (create T))
  (let* ((xx (round (* x-ratio 100)))
	 (yy (round (* y-ratio 100)))
	 (width (round (* x-ratio (bitmap-width bitmap))))
	 (height (round (* y-ratio (bitmap-height bitmap))))
	 (temp-tno 
#+:YY2.0
	  (make-territory
		    :width width :height height
		    :visible nil
		    :window-mode t
		    :transparent T)
          (make-territory
                    :width width :height height
                    :visible nil
                    :window-mode t)
	  ))
    ;;; $B3HBg!?=L>.(B
    (yy-protocol-38 (bitmap-territory-no bitmap)
		    temp-tno xx yy)

    (if create
	;;;$B@8@.(B
	(let ((ins
	       (make-instance 'bitmap :bitmap-territory-no temp-tno
			      :bitmap-width width :bitmap-height height)))
	  (set-territory-object temp-tno ins)
	  ins)
      (progn 
	(yy-protocol-5 (bitmap-territory-no bitmap))
	(delete-lisp-object (bitmap-territory-no bitmap))
	(set-territory-object temp-tno bitmap)
	(setf (slot-value bitmap 'bitmap-width) width
	      (slot-value bitmap 'bitmap-height) height
	      (slot-value bitmap 'bitmap-territory-no) temp-tno)
	bitmap))
    ))

;;; $B%S%C%H%^%C%W$N2sE>(B
;;; rotate-bitmap bitmap dgree &key create
;;; ARG.
;;;           bitmap   =  $B%S%C%H%^%C%W%$%s%9%?%s%9(B
;;;           dgree    =  $B2sE>3QEY(B ($B%G%#%0%j!<(B)
;;;           create   = $B%S%C%H%^%C%W$r?7$?$K@8@.$9$k$+(B
;;;                      T -> $B@8@.(B
;;; RET.
;;;           $B%S%C%H%^%C%W%$%s%9%?%s%9(B
(defmethod rotate-bitmap ((bitmap bitmap) dgree &key (create T))
  (let* ((xros (sqrt (+ (* (bitmap-width bitmap)
			   (bitmap-width bitmap))
			(* (bitmap-height bitmap)
			   (bitmap-height bitmap)))))
	 (height (floor (* xros (sin (* pi dgree 180.0)))))
	 (width (floor (* xros (cos (* pi dgree 180.0)))))

	 (temp-tno 
#+ :YY2.0
	  (make-territory
		    :width width :height height
		    :visible nil
		    :window-mode t
		    :transparent T)
	  (make-territory
                    :width width :height height
                    :visible nil
                    :window-mode t)))
    ;;; $B2sE>(B
    (yy-protocol-39 (bitmap-territory-no bitmap)
		    temp-tno (floor (* dgree 64.0)))
    (if create
	;;;$B@8@.(B
	(let ((ins
	       (make-instance 'bitmap :bitmap-territory-no temp-tno
			      :bitmap-width width :bit-map-height height)))
	  (set-territory-object temp-tno ins)
	  ins)
      (progn 
	(yy-protocol-5 (bitmap-territory-no bitmap))
	(delete-lisp-object (bitmap-territory-no bitmap))
	(set-territory-object temp-tno bitmap)
	(setf (slot-value bitmap 'bitmap-width) width
	      (slot-value bitmap 'bitmap-height) height
	      (slot-value bitmap 'bitmap-territory-no) temp-tno)
	bitmap))
    ))


  
;;; $B%$%a!<%8$N<h$j9~$_(B
;;; get-image bitmap position width height
;;; ARG.
;;;          bitmap           =  $B%S%C%H%^%C%W%$%s%9%?%s%9(B
;;;          position         =  $B%S%C%H%^%C%W$G$N0LCV%$%s%9%?%s%9(B
;;;          width            =  $B%$%a!<%8$NI}(B
;;;          height           =  $B%$%a!<%8$N9b$5(B
;;; RET.
;;;          image instance
(defmethod get-image ((bitmap bitmap) (pos position) (width integer)
                      (height integer))
  (declare (inline - + )
	   (function get-image-xy-internal (integer integer integer 
					    integer integer integer 
					    integer) T))
  (get-image-xy-internal (bitmap-territory-no bitmap)
                         (position-x pos) (position-y pos)
                         (bitmap-width bitmap)
                         (bitmap-height bitmap) width height)
    )

;;; $B%$%a!<%8$N<h$j9~$_(B XY
;;; get-image bitmap x y width height
;;; ARG.
;;;          bitmap           =  $B%S%C%H%^%C%W%$%s%9%?%s%9(B
;;;          xy               =  $B%S%C%H%^%C%W$G$N0LCV(B
;;;          width            =  $B%$%a!<%8$NI}(B
;;;          height           =  $B%$%a!<%8$N9b$5(B
;;; RET.
;;;          image instance
(defmethod get-image-xy ((bitmap bitmap) (x integer) (y integer) (width integer)
                      (height integer))
  (declare (inline - + )
	   (function get-image-xy-internal (integer integer integer
                                            integer integer integer
                                            integer) T))
  (get-image-xy-internal (bitmap-territory-no bitmap)
                         x y
                         (bitmap-width bitmap)
                         (bitmap-height bitmap) width height)
    )

;;; $B%$%a!<%8%G!<%?$r<h$j9~$`(B
(defun get-image-data-bitmap (bitmap x y width height)
  (declare (function get-image-data-xy-internal (integer integer integer
                                            integer integer integer
                                            integer) T))
  (get-image-data-xy-internal (bitmap-territory-no bitmap)
                         x y
                         (bitmap-width bitmap)
                         (bitmap-height bitmap) width height)
    )

