;;;
;;;	------ YY Grapher ------ 1990
;;;
;;;	by Takashi Kosaka (Aoyama Gakuin University / CSK)
;;;	written as a part of a project under the Jeida Common Lisp Committee 1990
;;;
;;;	The source code is completely in public domain.
;;;
;;;	The author(s), the organization the author belongs, Jeida,
;;;	Jeida CLC, Masayuki Ida (the chairman), and Aoyama Gakuin University
;;;	have no warranty about the contents, the results caused by the use,
;;;	and any other possible situations.
;;;
;;;	This codes is written as an experimentation.
;;;	This codes do not mean the best functinality the system supports.


;;; nodes-and-arcs.lisp

(defvar *all-the-nodes* nil)

(defvar *next-node-index* 0
  "Unique index assigned to each node.")

(defvar *next-arc-index* 0
  "Unique index assigned to each node.")

(defclass node ()
  ((arcs :initform nil :accessor node-arcs)
   (xpos :accessor node-xpos
	 :initarg :xpos)
   (ypos :accessor node-ypos
	 :initarg :ypos)
   (radius :initform :needs-calculation
	   :accessor node-radius)
   (label :initform nil
	  :accessor node-label)
   (shape :initform :circle
	  :accessor node-shape)
   (unique :initform (incf *next-node-index*)
	   :accessor node-unique))
  )

(defmethod initialize-instance :after ((node node) &rest initargs)
  (declare (ignore initargs))
  (push node *all-the-nodes*))

(defmethod print-object ((node node) stream)
  (let ((name (or (node-label node)
		  (format nil "Unnamed node ~D" (node-unique node)))))

      (write-string name stream)))

(defmacro map-over-nodes ((node-var) &body body)
  `(dolist (,node-var *all-the-nodes*)
     ,@body))

(defmethod (setf node-label) :after (new-value (node node))
  (setf (node-radius node) :needs-calculation))

(defmethod move-node ((node node) new-xpos new-ypos)
  (with-slots (xpos ypos) node
    (setf xpos new-xpos
	  ypos new-ypos)))

(defmethod delete-self ((node node))
  (dolist (arc (node-arcs node))
    (delete-self arc))
  (setf *all-the-nodes* (delete node *all-the-nodes*)))

(defmethod present-self ((node node) window)
  (calculate-radius node window)
  (with-slots (xpos ypos radius label) node
    (draw-circle-xy window xpos ypos radius)
    (draw-string-xy window (or label " ") (- xpos radius) ypos)
    ))

(defmethod calculate-radius ((node node) window)
  (with-slots (radius label) node
    (when (and (stringp label)
	       (zerop (length label)))
      (setf label nil))
    (when (eq radius :needs-calculation)
      (setf radius (ceiling (string-width window (or label " ")) 2)))
    radius))


(defclass arc ()
  ((mark :initform nil
	 :accessor arc-mark)
   (node1 :accessor arc-node1
	  :initarg :node1)
   (node2 :accessor arc-node2
	  :initarg :node2)
   (position1 :initform (make-position)
	      :accessor position1)
   (position2 :initform (make-position)
	      :accessor position2)
   (label :initform nil
          :accessor arc-label)
  ))

(defmethod add-arc ((node node) (arc arc))
  (incf *next-arc-index*)
  (with-slots (arcs) node
    (push arc arcs)))

(defmethod remove-arc ((node node) (arc arc))
  (with-slots (arcs) node
    (setf arcs (delete arc arcs))))

(defmethod initialize-instance :after ((arc arc) &rest initargs)
  (declare (ignore initargs))
  (with-slots (node1 node2) arc
    (add-arc node1 arc)
    (add-arc node2 arc)))

(defmethod print-object ((arc arc) stream)
  (with-slots (node1 node2) arc
      (format stream "~a <--> ~a" node1 node2)))

(defmacro map-over-arcs ((arc-var) &body body)
  (let ((mark-var (make-symbol "MARK"))
	(node-var (make-symbol "NODE")))
    `(let ((,mark-var (list nil)))
       (dolist (,node-var *all-the-nodes*)
	 (dolist (,arc-var (node-arcs ,node-var))
           (unless (eq (arc-mark ,arc-var) ,mark-var)
             ,@body
             (setf (arc-mark ,arc-var) ,mark-var)))))))


(defmethod delete-self ((arc arc))
  (with-slots (node1 node2) arc
    (remove-arc node1 arc)
    (remove-arc node2 arc)))

(defmethod present-self ((arc arc) window)
  (with-slots (node1 node2 position1 position2) arc
    (multiple-value-bind (x1 y1 x2 y2)
        (find-edges-of-nodes (node-radius node1)
			     (node-xpos node1) (node-ypos node1)
			     (node-radius node2)
			     (node-xpos node2) (node-ypos node2))
      (setf (position-x position1) (min x1 x2)
	    (position-x position2) (max x1 x2)
	    (position-y position1) (min y1 y2)
	    (position-y position2) (max y1 y2))
      (draw-line-xy window x1 y1 x2 y2))))

(defmethod present-self :after ((arc arc) window)
  (with-slots (label position1 position2) arc
    (let* ((width (string-width window (or label " ")))
	   (x (- (+ (position-x position1)
                   (ceiling (/ (- (position-x position2)
                                 (position-x position1)) 2)))
		 (ceiling (/ width 2))))
	  (y (+(position-y position1)
                   (ceiling (/ (- (position-y position2)
                                  (position-y position1)) 2))))
	  (height (font-kanji-height (stream-font window)))
	  (c-base (font-kanji-base-line (stream-font window))))

      (with-graphic-state ((color graphic-color) (filled filled-type)) window
			  (setf color *white-color*
				filled *FillSolid*)
	    (draw-region-xy window (- x 1) (- y c-base) (+ width 2) height))
      (draw-region-xy window (- x 1) (- y c-base) (+ width 2) height)
     (draw-string-xy window (or label " ") x y)
     )))

(defun find-edges-of-nodes (r1 xpos1 ypos1 r2 xpos2 ypos2)
  (let* ((dx (- xpos2 xpos1))
	 (dy (- ypos2 ypos1))
	 (length (isqrt (+ (* dx dx) (* dy dy)))))
    (values (+ xpos1 (ceiling (* dx r1) length))
	    (+ ypos1 (ceiling (* dy r1) length))
	    (- xpos2 (floor (* dx r2) length))
	    (- ypos2 (floor (* dy r2) length)))))


;;;Follwings are implementation dependent functions
;;; YY implementation
;;; Coded by T.kosaka

(defun string-width (window string)
  "returns the width (pixels) of string"
  (font-string-length (stream-font window) string)
  )

;;; Window stream 
(defvar *grapher-window* nil)
(defvar *command-window* nil)
(defvar *menu-window* nil)

;;; Command status
(defvar *command-status* nil)

;;; Menu List
(defvar *menu-list* '((add-arc-command "Add Arc" "Add Arc")
		      (create-node-command "Create" "Create Node")
		      (delete-arc-command "Delete Arc" "Delete " "Arc")
		      (delete-node-command "Delete Node" "Delete " "Node")
		      (move-node-command "Move" "Move")
		      (reset-command "Reset" "Reset")
		      (set-lable-command "Set Lable Node" "Set Lable " "Node")
		      (set-lable-arc-command "Set Lable Arc" "Set Lable " "Arc")))



;;; Redisplay all nodes and arcs
(defun redisplay-all ()
  (clear-window-stream *grapher-window*)
  (map-over-nodes (node) (present-self node *grapher-window*))
  (map-over-arcs (arc) (present-self arc *grapher-window*)))

;;; selection node from mouse position
(defun select-node (x y)
  (map-over-nodes (node)
     (if (> (node-radius node)
	    (sqrt (+ (expt (- x (node-xpos node)) 2)
		     (expt (- y (node-ypos node)) 2))))
	 (return-from select-node node))))

;;; Selection ARC from XY
(defun select-arc-xy (x y)
  (map-over-arcs (arc) 
     (with-slots (position1 position2) arc
       (when (and (< (position-x position1) x)
		  (> (position-x position2) x)
		  (< (position-y position1) y)
		  (> (position-y position2) y))
	   (let ((katamuki1 (ceiling (* (/ (- (position-y position2)
					      (position-y position1))
					   (- (position-x position2)
					      (position-x position1)))
					10)))
		 (katamuki2 (ceiling (* (/ (- y (position-y position1))
					   (- x (position-x position1)))
					10))))
	   (if (> 10 (- katamuki1 katamuki2))
	       (return-from  select-arc-xy arc))
	   )
	 ))))


;;; Selection ARC from two nodes
(defmethod select-arc ((node1 node) (node2 node))
  (map-over-arcs (arc)
   (if (or (and (eq node1 (arc-node1 arc)) (eq node2 (arc-node2 arc)))
	   (and (eq node1 (arc-node2 arc)) (eq node2 (arc-node1 arc))))
       (return-from select-arc arc))))
  
;;; Setting command status macro
(defmacro with-command-status ((status) &body body)
  (let ((old-status (gentemp))
	(ret-val (gentemp)))
  `(let ((,old-status ,*command-status*)
	 (,ret-val nil))
    
    (with-slots (input-string) ,*command-window*
		 (setf input-string nil))

     (setf  *command-status* ,status
	    ,ret-val (progn ,@body)
	    *command-status* ,old-status)
     ,ret-val)))

;;; Command functions
(defun create-node-command ()
  (let ((x nil) (y nil))
    (multiple-value-setq (x y)
      (catch 'position (get-xy-position x y)))
    (new-write-message (format nil "Create node ~a ~a" x y))
    (present-self (make-instance 'node :xpos x :ypos y) *grapher-window*)))

(defun get-one-node (message)
  (let ((node nil))
    (new-write-message (format nil "Click ~a node or Type " message))
    (with-command-status (:find-node)
      (setf node
	  (catch 'node
	    (loop
	     (let ((d-node (eval (read *command-window*))))
	     (if (eq (class-name (class-of d-node)) 'node)
		 (return d-node))))
	     )))
      node))

(defun get-one-arc ()
  (let ((arc nil))
    (new-write-message "Click Arc or Type ")
    (with-command-status (:find-arc)
      (setf arc
	  (catch 'arc
	    (loop
	     (let ((d-arc (eval (read *command-window*))))
	     (if (eq (class-name (class-of d-node)) 'arc)
		 (return d-arc))))
	    )))
    arc))


(defun get-xy-position (x y)
  (let ((xx nil)
	(yy nil))
    (with-command-status (:position-define)
      (when (or (null x) (not (integerp x)))
             ;;; Ask x position
	(select-window *command-window*)
	(new-write-message "Input Node X Position or Mouse click : ")

	(setf xx (read *command-window*))

      (when (or (null y) (not (integerp y)))
	(new-write-message "Input Node Y Position or Mouse click : ")
	(setf yy (read *command-window*))
	)
      ))
    (values xx yy)
    ))


(defun add-arc-command (&rest arg)
  (let ((node1 nil)
	(node2 nil))

    (unless (zerop *next-node-index*)
	    (setf node1 (get-one-node "No1 ")
		  node2 (get-one-node "No2 "))

	    (present-self (make-instance 'arc :node1 node1 :node2 node2) 
			  *grapher-window*)))
  )
  

(defun delete-arc-command (&rest arg)
  (let ((node1 nil)
        (node2 nil))
    (unless (zerop *next-arc-index*)
	    (setf  node1 (get-one-node "No1 ")
		   node2 (get-one-node "No2 "))
	    (delete-self (select-arc node1 node2))
	    (redisplay-all))))

(defun delete-node-command (&rest arg)
  (let ((node nil))
    (setf node (get-one-node " "))
    (delete-self  node)
    (redisplay-all)))

(defun move-node-command (&optional (x nil) (y nil))
  (let ((node nil))
    (setf node (get-one-node " "))
    (setf *select-node* node)
    (enable-event *grapher-window*)

    (multiple-value-setq (x y)
      (catch 'position 
	(get-xy-position x y)))

    (move-node node x y)
    (disnable-event *grapher-window* *mouse-move*)
    (redisplay-all)))


(defun reset-command (&rest arg)
  (clear-window-stream *grapher-window*)
  (clear-window-stream *command-window*)
  (setf *all-the-nodes* nil
	*next-node-index* 0
	*nex-arc-index* 0))

(defun set-lable-command (&optional (string nil) &rest state)
  (let ((node nil))
    (unless (zerop *next-node-index*)
	    (terpri *command-window*)
	    (setf node (get-one-node " "))
	    (new-write-message "Input lable: ")
	    (setf string (read-line *command-window*))
	    (setf (node-label node) string)
	    (redisplay-all)))
  )

(defun set-lable-arc-command ()
  (let ((arc nil)
	(string ""))
    (unless (zerop *next-arc-index*)
	    (setf arc (get-one-arc))
	    (new-write-message "Input lable: ")
	    (select-window *command-window*)
	    (setf string (read-line *command-window*))
	    (setf (arc-label arc) string)
	    (redisplay-all)))
  )

;;; Make Menu window
(defun make-menu-window ()
  (let* ((width (font-string-length *default-font* "Set Lable Node  "))
	 (window (make-window-stream :width width
		 :height (* (font-kanji-height *default-font*) 10)
		 :left 500 :bottom 500
		 :title-string "Menu Window"
		 :title-bar-visible nil
		 :vertical-scroll-visible nil
		 :horizontal-scroll-visible nil
		 :coordinate-area-visible nil))
	 (y 0)
	 (a-region nil))

    (dolist (item *menu-list*)
	    (setf a-region (make-active-region :left 0
		              :bottom y
			      :width width
			      :height (font-kanji-height *default-font*)
			      :parent window))

	    (setf (get (active-region-symbol a-region) 'command-name)
		  (car item)
		  (button1-method a-region) 'menu-command
		  (get 'menu-command 'single-process) t)

	    (incf y (font-kanji-height *default-font*))

	    (write-string (second item) window)
	    (terpri window)
	    (force-output window))
    (setf *menu-window* window)))

(defmethod menu-command ((a-region active-region) state)
  (throw 'command
	 (values (get (active-region-symbol a-region) 'command-name))))

;;; Make Grapher window and Command window
(defun make-grapher-window ()
  (let ((window (make-window-stream :left 10 :bottom 10
				     :width 600 :height 500
				     :title-string "Grapher Window"))
	(c-window (make-window-stream :left 10 :bottom 500
				      :width 490 :height 
				      (* (font-kanji-height *default-font*) 10)
				      :title-string "Message Window"
				      )))
    (set-window-method window
		       'right-button-down
		       :event-mask *mouse-right-1*)

    (set-window-method window
		       'move-node-mouse
		       :event-mask *mouse-move*)

    (setf (get 'right-button-down 'single-process) t
	  (get 'move-node-mouse 'single-process) t
	  *command-window* c-window
	  *grapher-window* window)

    (disnable-event window *mouse-move*)

  ))
   
;;; Button method for *grapher-window*
(defmethod right-button-down ((window window-stream) state)

    (case *command-status*
      (:position-define
          ;;; Set integer to window stream read buffer
       (throw 'position (values (mouse-state-x-position state)
			    (mouse-state-y-position state))))

      (:find-node
       (let ((node nil))
	 (if (eq (class-name (class-of (setf node 
			(select-node (mouse-state-x-position state) 
			  (mouse-state-y-position state))))) 'node)
	     (throw 'node (values node)))
	 ))
      (:find-arc
       (let ((arc nil))
	 (if (eq (class-name 
		  (class-of (setf arc 
			      (select-arc-xy (mouse-state-x-position state)
					     (mouse-state-y-position state)))))
		 'arc)
	 (throw 'arc (values arc)))))
		      
      (t
       )))

;;; move-node-method
(defmethod move-node-mouse ((window window-stream) state)
  (let ((select-node *select-node*))
    (with-slots (arcs) select-node
      (with-graphic-state ((op graphic-operation)) window
  	  (setf op *GXOR*)
	  (present-self select-node *grapher-window*)
	  (dolist (s-arc arcs)
	    (present-self s-arc *grapher-window*))
	  (setf 
	      (node-xpos select-node) (mouse-state-x-position state)
	      (node-ypos select-node) (mouse-state-y-position state))
	  
	  (present-self select-node *grapher-window*)
	  (dolist (s-arc arcs)
	    (present-self s-arc *grapher-window*)
	    )))
    ))

;;; initialize-grapher
(defun initialize-grapher ()
  ;;; Makeing grapher window
  (make-grapher-window)
  (make-menu-window)
  (setf *all-the-nodes* nil
	*next-node-index* 0
	*nex-arc-index* 0)
  (start-grapher))

;;; Start grapher 
(defun start-grapher ()
  (select-window *command-window*)
  (let ((exec-command nil))
    (clear-window-stream *command-window*)
    (loop 
     (write-message "Grapher Command: ")
     (setf exec-command
	   (catch 'command
	     (get-command-list)))

     (apply exec-command nil)
     (terpri *command-window*)
  )))

;;; get-command-list
(defun get-command-list ()
  (let ((item nil)
	(real-menu nil)
	(ret nil)
	(new-menu-list *menu-list*))

    (setf item (string-capitalize (string (read *command-window*))))

    (do ((menu (car new-menu-list) (setf new-menu-list (cdr new-menu-list)
					 menu (car new-menu-list))))
	((null new-menu-list))
	    (when (string= item (third menu) :end1 (length item)
			   :end2 (length item))
		  (setf real-menu menu)
		  (return)))

    (clear-string-region *command-window* item)

    (write-message (nth 2 real-menu))

    (when (stringp (fourth real-menu))
	  (setf item (string-capitalize (string (read *command-window*))))
	  (dolist (menu new-menu-list)
		  (when (string= item (fourth menu) :end1 (length item)
				  :end2 (length item))
			(setf real-menu menu)
                  (return))))

    (clear-string-region *command-window* item)
    (write-message (fourth real-menu))

    (values (car real-menu))))


;;; scrolling in same window    
(defmethod stream-force-output :before 
	   ((stream window-stream))
 (when (equal stream *command-window*)
  (when (> (+ (stream-cursor-y-position stream) 
	      (font-kanji-height *default-font*))
	   (region-height (frame-region stream)))
	(clear-window-stream stream))
  ))

;;; Write Message to command window
(defun write-message (message)
    (write-string  message *command-window*)
    (force-output *command-window*))

(defun new-write-message (message)
    (terpri *command-window*)
    (write-string  message *command-window*)
    (force-output *command-window*))    

;;; claear region in window stream
(defmethod clear-string-region ((stream graphic-stream) item)
  (let ((x (stream-cursor-x-position stream))
	(y (- (stream-cursor-y-position stream)
	      (font-kanji-base-line (stream-font stream))))
	(w  (font-string-length (stream-font stream) item))
	(h  (font-kanji-height (stream-font stream))))

  (with-graphic-state ((color graphic-color) (filled filled-type)) stream
     (setf color *white-color*
	   filled *FillSolid*)
     (draw-region-xy stream x y w h))))
