;;; -*- Mode: LISP; Syntax: COMMON-LISP; Base: 10.; Package: XIT -*-
;;;_____________________________________________________________________________
;;;
;;;                       System: XIT
;;;                       Module: Containers
;;;                       Version: 1.0
;;;
;;; Copyright (c): Forschungsgruppe DRUID, Juergen Herczeg
;;;                Universitaet Stuttgart
;;;
;;; File: /usr/local/lisp/xit/kernel/containers.lisp
;;; File Creation Date: 04/10/92 15:11:41
;;; Last Modification Time: 07/15/93 15:06:44
;;; Last Modification By: Juergen Herczeg
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;
;;; 01/13/1993 (Hubertus) allow :client-window initarg to hold a real contact 
;;;                       too
;;;
;;; 04/07/1993 (Juergen)  New event action (call :client ...), which may be
;;;                       used for container-windows.
;;;_____________________________________________________________________________

(in-package :xit)

;;;___________________________________________________________________________
;;;
;;;                      Layouter for Containers
;;;___________________________________________________________________________


(defclass container-layouter (layouter)
  ())

(defmethod layout ((self container-layouter) window)
  (with-slots (parent adjust-size? width height border-width) window
    (with-slots ((parent-adjust-size? adjust-size?)
		 (parent-width width)
		 (parent-height height)) parent
	(if parent-adjust-size?
	    (values (x-margin parent)
		    (y-margin parent)
		    width
		    height
		    border-width)
	  (values (x-margin parent)
		  (y-margin parent)
		  (- parent-width (x-margins parent)
		     border-width border-width)
		  (- parent-height (x-margins parent)
		     border-width border-width)
		  border-width)))))

;;;___________________________________________________________________________
;;;
;;;                             Containers
;;;___________________________________________________________________________

(defcontact container-window (intel)
  ((name :initform :container-window)
   (layouter :initform 'container-layouter))
  (:resources
   (border-width :initform 0)
   (inside-border :initform 0))
  (:documentation "Intels which are composed of usually one part, the client 
                   window"))

(defmethod initialize-instance :after ((self container-window)
				       &rest initargs
				       &key client-window)
  (assert (not (null client-window)) nil
    "A client-window specification is missing.")
  (unless (eq client-window :none)
    (add-client self client-window)))

(defmethod client-window ((self container-window) &key (errorp t))
  (or (first (parts self))
      (and errorp
	   (error "A client-window is missing."))))

(define-call-action :client (&rest function&args)
  `(,(first function&args) (client-window *self*)
    ,@(rest function&args)))

;;; this layouts container-windows after size changes by the user
;;;
(defmethod resize-window-with-mouse :after ((self container-window))
  (change-layout self))

;; Problem: If both container and client are non-self-adjusting
;; a change-layout to the container should trigger a change-layout
;; to the client!
;; However, things are somewhat different with subclass margined-window

#||
(defmethod adjusted-window-size ((self container-window))
  (with-slots (adjust-size?) self
    (let ((client (client-window self)))
      (if (and adjust-size? client)
	  (with-slots (border-width) client
	    (multiple-value-bind (width height)
		(adjusted-window-size client)
	      (values (+ (x-margins self) width border-width border-width)
		      (+ (y-margins self) height border-width border-width))))
	(call-next-method)))))
||#

(defmethod identification ((self container-window))
  (value (client-window self :errorp t)))

(defmethod (setf identification) (value (self container-window))
  (setf (value (client-window self :errorp t)) value))

(defmethod add-client ((self container-window) (client-specs symbol)
		       &rest override-initargs)
  (apply #'add-part self :class client-specs override-initargs))

(defmethod add-client ((self container-window) (client-specs cons)
		       &rest override-initargs)
  (apply #'add-part self
	 :class (car client-specs)
	 (append override-initargs (cdr client-specs))))

(defmethod add-client ((self container-window) (child contact)
		       &rest override-initargs)
  (declare (ignore override-initargs))
  ;; ensure width and height are initialized
  (with-slots (x y width height) self
    (or (plusp x)
	(setf x (contact-x child)))
    (or (plusp y)
	(setf y (contact-y child)))
    (or (plusp width)
	(setf width (contact-total-width child)))
    (or (plusp height)
	(setf height (contact-total-height child))))
  (reparent-contact child self))
  
(defmacro children-to-be-reparented (window)
  `(getf (window-plist ,window) :children-to-be-reparented))

(defun reparent-contact (contact composite)
  (cond ((and (realized-p composite)
              (realized-p contact))
	 (setf (contact-parent contact) composite))
        ((realized-p contact)
	 (push contact (children-to-be-reparented composite)))
        ((realized-p composite)
	 (error "Cannot add an unrealized child to a container-window already~
                 realized!"))
        (t (with-slots (parent) contact
             (when parent
	       (delete-child parent contact)
	       (setf parent composite)
	       (add-child composite contact))))))

(defmethod realize :after ((self container-window))
  (dolist (child (children-to-be-reparented self))
    (setf (contact-parent child) self))
  (setf (children-to-be-reparented self) nil))


;;;___________________________________________________________________________
;;;
;;;                        Special Containers
;;;___________________________________________________________________________

(defcontact popup-container (popup-window container-window)
  ()
  (:resources
   (border-width :initform 0)
   (inside-border :initform 0)))

#||
;; Example:
(popup
(make-window 'popup-container
	     :destroy-after? t
	     :client-window '(text-dispel
			      :inside-border 10
			      :border-width 1
			      :background "white"
			      :text "popup text"
			      :font (:size 18)))
)
||#

(defcontact popup-part-container (popup-part-connection container-window)
  ())

(defcontact shadow-borders-popup-part-container
    (shadow-borders-mixin popup-part-connection container-window)
  ())

#||
;; Example:
(make-window 'popup-part-container
	     :popup-part :default
	     :reactivity '((:menu))
	     :client-window '(text-dispel
			      :inside-border 10
			      :border-width 1
			      :background "white"
			      :text "text with popup"
			      :font (:size 18)))
||#

(defcontact window-icon-container (window-icon-mixin container-window)
  ())

#||
;; Example:
(make-window 'window-icon-container
	     :window-icon :default
	     :reactivity '((:double-left-button "Shrink"
						(call :self shrink)))
	     :client-window '(text-dispel
			      :inside-border 10
			      :border-width 1
			      :background "white"
			      :text "text with icon"
			      :font (:size 18)))
||#

(defcontact shadow-borders-container (shadow-borders-mixin container-window)
  ())

#||
;; Example:
(make-window 'shadow-borders-container
	     :client-window '(text-dispel
			      :inside-border 10
			      :border-width 1
			      :background "white"
			      ;:adjust-size? nil
			      :text "text with shadow"
			      :font (:size 18)))
||#

(defcontact shadow-borders-popup-container (shadow-borders-mixin
					    popup-container)
  ())

(defcontact window-icon-popup-part-container (window-icon-mixin
					      popup-part-container)
  ())
