
(in-package "PT")

;;;*********************************************************
;;;
;;; NODE class, methods, and support functions:
;;;

(defdbclass node (hyper-object owned-obj)
  (;; (description :initform "A node" )
   (type    :initform nil :type symbol :reader type)
   (dataset :initarg :dataset :initform "" :type string :accessor dataset
            :documentation "pointer to location of node's data (pathname)")
   (linkset :initarg :linkset :initform "" :type string :accessor linkset
	    :documentation "pointer to file containing link info")
   (link-markers :initform nil :type list :accessor link-markers
           :documentation "list of link-markers defined within the node")
   (bookmarks :initform nil :type list :accessor bookmarks
           :documentation "list of bookmarks defined within the node")
   ;;list of hyperdocs containing this node:
   (hyperdocs :initarg :hyperdocs :initform nil :type list :accessor hyperdocs)
   (modified :initform nil :type symbol :accessor modified?
              :documentation "if t, node has been modified since last save")
   (visited :initform nil :type symbol :accessor visited? 
             :documentation "if t, node has been visited during this session")
   ;; I'd rather store this on the hyper-widget, but that's causing
   ;; circularity problems in the file dependencies:
   (opener :initform nil :accessor opener :type link
           :documentation 
              "Link followed, if any, to arrive at this node, if open"))
  (:documentation "abstract class defining general node attrs")
  )


(defmethod new-instance ((n node) 
			 &key name
                         (hyperdocs (list #!*current-hyperdoc*))
			 (dataset "")
			 (linkset "")
                         &allow-other-keys)
  (if (equal dataset "")
      (warn "No data file specified for node ~a" name))
  (if (equal linkset "")
      (setf (linkset n) (concatenate 'string dataset ".links")))
  (if (not name)
      (setf (name n) (intern (gensym "node-") (find-package 'pt))))
  (call-next-method)
  (dolist (h hyperdocs) (if h (add-obj n h)))
  (store n))  ;; add to global set of nodes

(defmethod viewer ((n node))
  (let ((panel (find-if #'(lambda (p) (eq n #!node@p)) *hip-panels*)))
    (when panel #!hw@panel)))

(defmethod (setf viewer) (val (n node))
  (declare (ignore val))
  nil)

(defmethod markers ((n node))
  (append (link-markers n) (bookmarks n)))

(defmethod visible-markers ((n node))
  (remove-if #'null (markers n) :key #'region))

(defmethod links-from ((n node))
  "returns the set of links emanating from n"
  (links-from (link-markers n)))

(defmethod links-into ((n node))
  (links-into (link-markers n)))

(defmethod visit ((n node))
  (setf (visited? n) t)
  (stamp n)  ;; note current time
  (update-style (graph-box n))
  ;; put it at the front of the history list:
  (if (member n *history*)
      (setf *history* (cons n (remove n *history*)))
    (push n *history*)))
      
(defmethod unvisit ((n node))
  (setf (visited? n) nil)
  (update-style (graph-box n)))

(defmethod modify ((n node))
  (setf (modified? n) t))
(defmethod unmodify ((n node))
  (setf (modified? n) nil))

;; This needs to be modified so it is specialized for each node type
;; to deliver a widget to display itself
(defmethod get-display-widget ((n node) env)
  (case (type n)
	(table #!table-wid@env)
	(text  #!text-wid@env)
	(wip #!coll-wid@env)  
	(image #!image-wid@env)
	(video #!video-wid@env)
	))


(defmethod open-node ((n node) &key (offset nil) (opener nil))
  (if (not (has-access *user* n 'read))
      (announce-error (format nil "User ~a denied access to node ~a" *user* (name n)))
    (with-feedback
     (format nil "Opening ~a node ~a" (type n) (name n))
     ;; note what link, if any, caused us to arrive at this node:
     (setf (opener n) opener)
     (when *widgets-loaded*
	(let ((panel
	       ;; if this node is already in a panel somewhere, use it:
	       (if (node-in-use n)
		   (reopen-panel (node-panel n))
		 ;; else grab a panel from somebody else:
		 (progn
		   (record-open-node n)
		   (use-new-panel :node n)))))
	  ;; I think moving the scrolling here will solve some problems...
	  (when offset (scroll-to #!hw@panel offset))))
     (visit n))))

(defmethod close-node (n)
  "Updates interface to reflect that n is no longer being viewed"
  (when (node-in-use n)
        (if (and (modified? n)
                 (ask-user-to-confirm 
                  (format nil "Save node ~a before closing?" (name n))))
            (save n))
        (setf (opener n) nil)
	(record-close-node n)
        (close-panel (node-panel n))
        (update-style (graph-box n))   
        ))

(defun user-close-node (n)
  (setq *explicit-close-node* t)
  (close-node n)
  (setq *explicit-close-node* nil))

(defmethod save ((n node) &optional hw)
  (declare (ignore hw))
  "default method: just save link info, not contents"
  (write-node n)
  (unmodify n))
  

(defmethod preview ((n node))
  "displays descriptor for other endpoint of link"
  (if *widgets-loaded*
      (call (find-po-named '("new-hip" "previewer" . "panel")) :node n)
    (format t "~%Node ~a: ~s" (name n) (description n)))
  n)

(defmethod node-summary ((n node))
  "returns a list of strings representing the name, header, keywords,
   and dataset of n"
  (list (string (name n)) (description n) (stringify-list (keywords n)) (dataset n)))

(defun get-all-node-info (&optional (node-table *nodes*) &aux (info  nil))
  ;; this typecase can go away once we nail down the right rep for *nodes*...
  (typecase node-table
      (hash-table
       (maphash #'(lambda (key node) 
			  (declare (ignore key))
			  (setq info (append info (list (node-summary node)))))
		node-table)
       info)
      (list
       (mapcar #'node-summary node-table))))


(defmethod store ((n node))
  (setf (gethash (name n) *nodes*) n))

(defmethod unstore ((n node))
  (db-unstore-object n)
  (remhash (name n) *nodes*))

(defmethod edit ((n node))
  (let ((result (call (find-po-named '("new-hip" "edit-node" . "dialog"))
                                     :node n)))
    (when result
          (update n result)
	  (modify n))))


(defmethod copy ((n node))
  (announce-error "Oops - haven't implemented copying nodes yet...")
  n)

;;;************************************************************
;;;
;;;  TEXT-NODE
;;;  Nodes to display text (duh!)

(defdbclass text-node (node)
  ((description :initform "A text node")
   (type :initform 'text :reader type))
  (:documentation "Class of nodes for displaying text files"))

(defun make-text-node (&rest args)
  (apply #'make-instance 'text-node args))

(defmethod save ((tn text-node) &optional hw)
  (declare (ignore hw))
  ;;  This should probably be modified to use or at least resemble the
  ;;  save handler for text-widgets.  Need to think about updating the
  ;;  'modified attribute when links are added/deleted, since that also
  ;;  signals a need to save the node.
  (with-feedback (format nil "Saving file ~a" (dataset tn)) (sleep 3))
  (save-file (widget (viewer tn)) (dataset tn))
  (call-next-method)  ;; ok, so it should be a :before method...
  )

;;;************************************************************
;;;
;;;  TABLE-NODE
;;;  

(defdbclass table-node (node)
  ((description :initform "A table node")
   (type :initform 'table :reader type))
  (:documentation "Class of nodes for displaying tabular data"))

(defun make-table-node (&rest args)
  (apply #'make-instance 'table-node args))

(defmethod save ((tn table-node) &optional hw)
  (declare (ignore hw))
   ;; add code to save table data..
  (call-next-method))


;;;************************************************************
;;;
;;;  IMAGE-NODE
;;;  

(defdbclass image-node (node)
  ((description :initform "An image node")
   (type :initform 'image :reader type)
   (bitmap-p :initarg :bitmap-p :initform nil :reader bitmap-p))
  (:documentation "Class of nodes for displaying image data"))

(defun make-image-node (&rest args)
  (apply #'make-instance 'image-node args))

(defmethod new-instance ((n image-node) 
                     &key dataset &allow-other-keys)
  (setf (slot-value n 'bitmap-p) 
        (or
         (search "bitmap" dataset :test #'char=)
         (search "xbm" dataset :test #'char=)))
  (call-next-method))


;;;************************************************************
;;;
;;;  VIDEO-NODE
;;;  


;; (in-package 'pt :use '(video))

(defdbclass video-node (node)
  ((description :initform "An image node")
   (type :initform 'video :reader type)
   (videodisk :initform (video:make-indexed-video-disk)
              :initarg :videodisk 
              :type video-disk :accessor videodisk))
  (:documentation "Class of nodes for displaying image data")
  )


(defun make-video-node (&rest args)
  (apply #'make-instance 'video-node args))


(defmethod new-instance ((n video-node) 
                     &key dataset name video-disk &allow-other-keys)
  (call-next-method)
  (if (and (null dataset) name video-disk)
      (setf (dataset n) 
            (format nil "~a/~a.index"
                    (disk-title video-disk) name)))
  )


;;;************************************************************
;;;
;;;  GRAPHIC-NODE
;;;  

(defdbclass graphic-node (node)
  ((description :initform "A graphic node")
   (type :initform 'graphic :reader type))
  (:documentation "Class of nodes for displaying graphic data"))

(defun make-graphic-node (&rest args)
  (apply #'make-instance 'graphic-node args))


;;;************************************************************
;;;
;;;  PROC-NODE
;;;   we may want further subclasses for different types of code...BSB 4/19

(defdbclass proc-node (node)
  ((description :initform "A procedural node"))
   (:documentation  
    "Class of nodes containing a body of code to execute upon opening"))

(defun make-proc-node (&rest args)
  (apply #'make-instance 'proc-node args))


;;;************************************************************
;;;
;;; NODE-SET class, methods,and support functions
;;;

(defdbclass node-set (node)
  (
   (nodes :initarg :nodes :initform nil :type list :accessor nodes)
   ;; did we also want links stored here?
  )
)

(defmethod browse ((ns node-set))
  "invokes graphical browser on set of nodes"
  (announce-error "Browsing node sets is not yet implemented"))


;;;*******************************************************
;;;
;;; NODE-RECORD class, etc
;;;  Not quite sure how we'll use this yet...

(defdbclass node-record (node)
  ((type :initform 'record :reader type)
   (fields :initarg :fields :initform nil :type list :accessor fields))
)

(defun make-node-record (&rest args)
  (apply #'make-instance 'node-record args))


(defmethod field-value ((self node-record) name)
  (some #'(lambda (pair) (if (eql name (car pair)) (cdr pair) nil))
        (fields self)))

;;;*******************************************************
;;;
;;;  WIP-NODE subclass of node-record
;;;

;; Dataset should be "(run_id log_id)" pair

(defdbclass wip-node (node-record)
  (
   (type :initform 'wip :reader type)
   (run_id :initform 0 :accessor run-id)
   (log_id :initform 0 :accessor log-id)
   (description :initform "A WIP log entry" :initarg :description 
		:accessor description)))

(defun make-wip-node (&rest args)
  (apply #'make-instance 'wip-node args))

(defmethod new-instance ((self wip-node) &key (dataset nil) &allow-other-keys &aux args)
  (call-next-method)
  (when dataset
        (setq args (read-from-string dataset))
        (setf 
         (run-id self) (first args)
         (log-id self) (second args)
         (fields self) (fetch-wip-fields self)
         (description self) (field-value self 'proc)
	 (linkset self) nil)))


(defmethod fetch-wip-data ((self wip-node))
  '("log_type" "time" "step_path" "tag" "proc" "log"))


(defmethod fetch-wip-fields ((self wip-node))
  (let* ((fields (mapcar #'cons *wip-fields* (fetch-wip-data self)))
         (log-data (cdr (assoc 'log fields))))
    (unless (string= log-data "log")
            (setf (cdr (assoc 'log fields)) 
                  (string-to-pair-list (cdr (assoc 'log fields)))))
    fields))


(defmethod save ((n wip-node) &optional hw)
  (declare (ignore hw))
  (feedback "WIP nodes are read-only.  Just saving link info...~%")
  (write-node n)
  (unmodify n))
