;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: chungl $
;;; $Source: /pic2/picasso/src/toolkit/resource/RCS/display.cl,v $
;;; $Revision: 1.4 $
;;; $Date: 1992/04/11 04:03:20 $
;;;

(in-package "PT")

(defun default-display-name ()
  (if (and *default-display-name* 
	   (> (length *default-display-name*) 0))
      *default-display-name*
      (let ((h (get-environment "DISPLAY"))
	    (s nil))
	   (when h 
		 (setq h (string-trim ":0.0" h)) 

		 (when (string= h "unix")
		       (setq h "localhost"))

		 (setq *default-display-name* h) 
		 (return-from default-display-name h))
	   (setq s (#+allegro excl::run-shell-command
		    #+lucid run-program
	                "hostname" :output :stream :wait nil))
	   (warn "IGNORE following ioctl error")
	   (setq *default-display-name* (string-downcase (read s)))
	   (close s)
	   *default-display-name*)))

;;;
;;; display class
;;;

(defclass display (pmc)
  ((name
    :initarg :name
    :initform nil
    :type string
    :accessor name)
   (res
    :initform nil
    :type vector
    :reader res)
   (screen-table
    :initform nil
    :type vector
    :reader screen-table)
   (window-table
    :initform nil
    :type hash-table
    :reader window-table)
   (font-table
    :initform (make-hash-table :test #'equal :size 100 :rehash-size 20)
    :type hash-table
    :reader font-table)
   (cursor-table
    :initform (make-hash-table :test #'equal :size 50 :rehash-size 20)
    :type hash-table
    :reader cursor-table)
   (primary-screen
    :initform nil
    :type atom
    :reader primary-screen)))

(defun make-display (&rest keys)
  (apply #'make-instance 'display :allow-other-keys t keys))

;;;
;;; 	Find screen given name/number
;;;

(defun get-screen (&optional (num 0) display)
  (if display 
      (unless (display-p display)
	      (error "find-screen: illegal display \`~s\`." display))
      (setq display *current-display*))
  (if (integerp num)
      (aref (screen-table display) num)
      (warn "find-screen: illegal screen-number \`~s\`." num)))

(defun get-display (&optional name)
  (if (null name)
      (current-display)
      (progn
       (gethash name *display-table*))))

;;;
;;; 	Accessors for "current" display and screen
;;;
;;;	See macros in macros.cl

(defun setf-current-display (value)
  "Set the current display instance to some active display instance"
  ;; test if value is null
  (unless value
	  (setq *current-display* nil
		*current-screen* nil
		*current-root* nil)
	  (return-from setf-current-display))
  (unless (and value (display-p value))
	  (error "display.setf-current-display: invalid display ~s" value))
  ;; update display variables
  (setq *current-display* value)
  (setq *current-screen* (primary-screen value))
  (setq *current-root* (root-window *current-screen*))
  (setf (gethash (name value) *display-table*) value)
  (pushnew value *active-displays*))

(defun setf-current-screen (screen &optional display)
  "Set the current screen of the specified display to specified screen"
  (if display
      (unless (display-p display)
	      (error "setf-current-screen: illegal display \'~s\'." display))
      (setq display *current-display*))
  (if (screen-p screen)
      (progn 
       (setf (slot-value display 'primary-screen) screen)
       (if (eq display *current-display*)
	   (setq *current-screen* screen
		 *current-root* (root-window screen))))
      (error "setf-current-screen: illegal screen \'~s\'." screen)))

;;;
;;; display initialization method
;;;

(defmethod new-instance ((self display) &key (name nil))
  ;; test if name specified
  (if name
      ;; test name argument
      (loop (cond ((not (stringp name))
		   ;; ask for a new name
		   (cerror "enter new name" "invalid display name")
		   ;; read value
		   (setq name (read)))
		  (t (return)))) 
      ;; get default display name 
      (setf (slot-value self 'name) 
	    (setq name (default-display-name))))

  ;; initialize slots 'screen-table and 'primary-screen
  ;; we only support one screen.
  (let* ((n 1) ;should be (xlib:display-nscreens (res display))
	 (table (make-array (list n)))
	 (screen (make-screen :number n :display self))
	 )
	(setf (aref table 0) screen)
	(setf (slot-value self 'screen-table) table)
	(setf (slot-value self 'primary-screen) screen)
	)
  self)

;; this is a list of what do-attach does, and do-detach must undo
;; - call xlib:open-display
;; - set value of (res self)
;; - set (current-display) to self
;; - set up defaults
;; - make and attach a bunch of images

(defmethod do-attach ((self display) &aux name res)

  ;; allocate resource and open display
  (setf name (slot-value self 'name))
  (unless name
	  (setf (slot-value self 'name) 
		(setq name (default-display-name))))
  (unless (setq res (xlib:open-display name))
	  (error "display.new-instance: can't open display ~s" name))
  ;; ======================================================
  ;; Hack to fix CLX bug
  ; (setf (xlib:display-image-lsb-first-p res) nil)
  #+DEC3100
  (if (and (xlib:display-p res) 
	   (member (xlib::display-host res)
		   '("bugs-bunny" "unix" "erewhon" "redrider" 
				  "shangri-la" "eden" "utopia"
				  "elysium" "stinson" "cloud9"
				  "arcadia" "gaia") :test #'string=))
      (setf (xlib:display-image-lsb-first-p res) nil))
  
  ;; ======================================================
  ;; set display resource slot 
  (setf (slot-value self 'res) res)

  ;; set current display to instance
  (setf (current-display) self)

  ;; attach screens
  ;; we only support one screen
  (attach (primary-screen self))

  ;; set current display to instance again, this time to
  ;; set the value of *current-screen* correctly. 
  ;; (this is weird, but works.)
  (setf (current-display) self)

  ;; Load in an create default resources database
  (load-defaults)

  ;; attach stuff already in hash tables
  (maphash #'(lambda (k v) (attach v))
	   (font-table self))
  (maphash #'(lambda (k v) (attach v))
	   (cursor-table self))

  ;; make and attach default display images
  (init-display-images)

  ;; make default font and cursor
  (init-display-fonts self)
  (init-display-cursors self)

  ;; make shared gcs
  (init-display-gcs self)

  ;; Initialize garbage collection cursor
  ;;  (gc_cursor_init)
  ;;  (gc_set_window (xlib:window-id (res rw)))

  ;; Register self in hashtable
  (setf (gethash res *global-display-hashtab*) self)

  ;; return self
  self)

;;; 
;;; display resource initialization functions.
;;;

(defun init-display-fonts (&optional (self (current-display)) &aux font)
  (attach (make-font :display self))
  (if (setq font (get-default "button" "font"))
      (attach (make-font :name font :display self))))

(defun init-display-cursors (&optional (self (current-display)))
  ;; cursors come attached.
  (make-cursor :file "arrow.cursor"
	       :name "arrow"
	       :mask-file "arrow_mask.cursor"
	       :display self)
  (make-cursor :file "left_ptr.cursor"
	       :name "left-ptr"
	       :mask-file "left_ptr_mask.cursor"
	       :display self)
  (make-cursor :file "push.cursor"
	       :name "push"
	       :foreground "black"
	       :background "white"
	       :mask-file "push_mask.cursor"
	       :display self)
  (make-cursor :file "clock.cursor"
	       :name "clock"
	       :mask-file "clock_mask.cursor"
	       :display self))

(defun init-display-images (&optional (disp (current-display)))
  ;; make default display images (bitmaps)
  (make-image :file "left_gray_arrow.bitmap" :name "left-gray-arrow")
  (make-image :file "left_gray_arrow_mask.bitmap" :name "left-gray-arrow-mask")
  (make-image :file "right_gray_arrow.bitmap" :name "right-gray-arrow")
  (make-image :file "right_gray_arrow_mask.bitmap" 
	      :name "right-gray-arrow-mask")
  (make-image :file "weave.bitmap" :name "weave")
  (make-image :name "picasso" :file "picasso.icon")
  
  ;; only create grays if black & white display 
  (when (and (black-and-white-display-p disp) 
	     (find-library-file "gray75.bitmap"))
	(make-image :file "gray75.bitmap" :name "gray75")
	(make-image :file "gray50.bitmap" :name "gray50")
	(make-image :file "gray25.bitmap" :name "gray25")))

(defun init-display-gcs (&optional self)
  (declare (ignore self))
  (register-gc "default" nil))
  
;;;
;;; display functions
;;;

;;
;;	Flushes any buffered output in the display
;;
(defun flush-window-output (&optional (display (current-display) displayp))
  ;; test display
  (if (and displayp (not (display-p display)))
      (error "flush-window-output: invalid display ~s" display))

  (xlib:display-force-output (res display)))

;;
;;	Flushes any buffered output to the display,
;;	flushes any buffered errors to the designated handlers,
;;	and makes sure all events have reached designated windows.
;;	Will not return until all completed.
;;
(defun flush-display (&optional (display (current-display) displayp))
  ;; test display
  (if (and displayp (not (display-p display)))
      (error "flush-display: invalid display ~s" display))

  (xlib:display-finish-output (res display)))

(defun grab-display (&optional (display (current-display) displayp))
  ;; test display
  (if (and displayp (not (display-p display)))
      (error "grab-display: invalid display ~s" display))

  (xlib:grab-server (res display)))

(defun ungrab-display (&optional (display (current-display) displayp))
  ;; test display
  (if (and displayp (not (display-p display)))
      (error "ungrab-display: invalid display ~s" display))

  (xlib:ungrab-server (res display)))

;;;
;;; display operation methods
;;;

(defmethod do-detach ((self display) &aux table)
  ;; remove garbage-collect cursor
  ;;  (gc_cursor_finish)

  ;; detach font and cursor hash tables
  ;; force detach by rigging slot 'ref-count
  (maphash #'(lambda (k v) (setf (slot-value v 'ref-count) 1) (detach v))
	   (font-table self))
  (maphash #'(lambda (k v) (setf (slot-value v 'ref-count) 1) (detach v))
	   (cursor-table self))

  ;; detach screens
  (dotimes (i (length (setq table (screen-table self))))
	   (detach (aref table i)))
  (setf (slot-value self 'screen-table) nil)
  (detach (primary-screen self))

  ;; remove display from active-list
  (remhash (name self) *display-table*)
  (setq *active-displays* (delete self *active-displays*))
  
  ;; close the X display
  (remhash (res self) *global-window-table-hashtab*)
  (remhash (res self) *global-display-hashtab*)
  (xlib:close-display (res self))

  ;; clear window hash table
  (setf (slot-value self 'window-table) nil)

  ;; set the resource slot to nil
  (setf (slot-value self 'res) nil)

  (setf *default-display-name*  nil)
  (setf (slot-value self 'name) nil)

  ;; test if detaching the current display
  (when (eq self (current-display))
	;; reset current display to nil
	(setf (current-display) nil))

  )

;;;
;;;	functions that query display
;;;

(defun black-and-white-display-p 
  (&optional (disp (current-display)))
  ;; test display
  (cond ((display-p disp) 
	 (= 1 (xlib:screen-root-depth 
	       (xlib:display-default-screen (res disp)))))
	((screen-p disp)
	 (= 1 (xlib:screen-root-depth (res disp))))
	((colormap-p disp)
	 (= 1 (xlib:drawable-depth (res (root (screen disp))))))
	(t (error "black-and-white-display-p: invalid display ~s" disp))))

(defun color-display-p 
  (&optional (disp (current-display)))
  ;; test display
  (cond ((display-p disp) 
	 (< 1 (xlib:screen-root-depth 
	       (xlib:display-default-screen (res disp)))))
	((screen-p disp)
	 (< 1 (xlib:screen-root-depth (res disp))))
	((colormap-p disp)
	 (< 1 (xlib:drawable-depth (res (root (screen disp))))))
	(t (error "color-display-p: invalid display ~s" disp))))
