/*
 SuperCollider real time audio synthesis system
 Copyright (c) 2002 James McCartney. All rights reserved.
 http://www.audiosynth.com
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 */

//Wavelets UGen code by Nick Collins summer 2011


#include "WT_UGens.h"



struct WT_Unit : Unit
{
    
};


struct WT_Freeze : Unit
{
	int numbins_;
	float * coefficientstore_;
};

struct WT_Onset : Unit
{
	int numbins_;
    int onsetcount_; //for imposing gap
	float * coefficientstore_;
    float outval_; //for storing held value for detection function 
};

struct WT_Transient : Unit
{
	int numscales_;
    float ** scales_; 
	int * scalesize_;
    
    float * sumtoscale_; 
    int * scalestatus_; 
    int * scalepos_; 
    float * mask_; 
    float ** maskscales_;
    
    
};


struct WT_ModulusSum : Unit
{
    float outval_; //for storing held value
};


//struct PV_OutOfPlace : Unit
//{
//	int m_numbins;
//	float *m_tempbuf;
//};
//
//struct PV_MagSmear : PV_OutOfPlace
//{
//};
//
//struct PV_MagShift : PV_OutOfPlace
//{
//};
//
//struct PV_BinShift : PV_OutOfPlace
//{
//};
//
//struct PV_Diffuser : Unit
//{
//	int m_numbins;
//	float m_prevtrig, *m_shift;
//	bool m_triggered;
//};
//
//struct PV_RandWipe : Unit
//{
//	int *m_ordering, m_numbins;
//	float m_prevtrig;
//	bool m_triggered;
//};
//
//struct PV_RandComb : Unit
//{
//	int *m_ordering, m_numbins;
//	float m_prevtrig;
//	bool m_triggered;
//};
//
//struct PV_BinScramble : Unit
//{
//	int *m_from, *m_to, m_numbins;
//	float m_prevtrig;
//	float *m_tempbuf;
//	bool m_triggered;
//};
//
//struct PV_Conj : PV_Unit {};

//////////////////////////////////////////////////////////////////////////////////////////////////

extern "C"
{
    
    //void PV_MagClip_Ctor(PV_Unit *unit);
	//void PV_MagClip_next(PV_Unit *unit, int inNumSamples);
    void WT_MagAbove_Ctor(WT_Unit *unit);
    void WT_MagAbove_next(WT_Unit *unit, int inNumSamples);
    
    void WT_MagBelow_Ctor(WT_Unit *unit);
    void WT_MagBelow_next(WT_Unit *unit, int inNumSamples);
    
    void WT_MagClip_Ctor(WT_Unit *unit);
    void WT_MagClip_next(WT_Unit *unit, int inNumSamples);
    
    void WT_SoftThreshold_Ctor(WT_Unit *unit);
    void WT_SoftThreshold_next(WT_Unit *unit, int inNumSamples);
    
    void WT_FilterScale_Ctor(WT_Unit *unit);
    void WT_FilterScale_next(WT_Unit *unit, int inNumSamples);
    
    void WT_Mul_Ctor(WT_Unit *unit);
    void WT_Mul_next(WT_Unit *unit, int inNumSamples);
    
    void WT_TimeWipe_Ctor(WT_Unit *unit);
    void WT_TimeWipe_next(WT_Unit *unit, int inNumSamples);
    
    void WT_Freeze_Ctor(WT_Freeze *unit);
	void WT_Freeze_Dtor(WT_Freeze *unit);
	void WT_Freeze_next(WT_Freeze *unit, int inNumSamples);
    
    void WT_Onset_Ctor(WT_Onset *unit);
	void WT_Onset_Dtor(WT_Onset *unit);
	void WT_Onset_next(WT_Onset *unit, int inNumSamples);
    
    void WT_Transient_Ctor(WT_Transient *unit);
	void WT_Transient_Dtor(WT_Transient *unit);
	void WT_Transient_next(WT_Transient *unit, int inNumSamples);
    
    
    void WT_ModulusSum_Ctor(WT_ModulusSum *unit);
    void WT_ModulusSum_next(WT_ModulusSum *unit, int inNumSamples);
    
    
    //	void PV_MagClip_Ctor(PV_Unit *unit);
    //	void PV_MagClip_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_MagAbove_Ctor(PV_Unit *unit);
    //	void PV_MagAbove_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_MagBelow_Ctor(PV_Unit *unit);
    //	void PV_MagBelow_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_Min_Ctor(PV_Unit *unit);
    //	void PV_Min_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_Max_Ctor(PV_Unit *unit);
    //	void PV_Max_next(PV_Unit *unit, int inNumSamples);
    //
    
    //	void PV_Div_Ctor(PV_Unit *unit);
    //	void PV_Div_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_Add_Ctor(PV_Unit *unit);
    //	void PV_Add_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_RectComb_Ctor(PV_Unit *unit);
    //	void PV_RectComb_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_RectComb2_Ctor(PV_Unit *unit);
    //	void PV_RectComb2_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_MagSquared_Ctor(PV_Unit *unit);
    //	void PV_MagSquared_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_MagMul_Ctor(PV_Unit *unit);
    //	void PV_MagMul_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_MagDiv_Ctor(PV_Unit *unit);
    //	void PV_MagDiv_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_Copy_Ctor(PV_Unit *unit);
    //	void PV_Copy_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_MagSmear_Ctor(PV_MagSmear *unit);
    //	void PV_MagSmear_Dtor(PV_MagSmear *unit);
    //	void PV_MagSmear_next(PV_MagSmear *unit, int inNumSamples);
    //
    //	void PV_BinShift_Ctor(PV_BinShift *unit);
    //	void PV_BinShift_Dtor(PV_BinShift *unit);
    //	void PV_BinShift_next(PV_BinShift *unit, int inNumSamples);
    //
    //	void PV_MagShift_Ctor(PV_MagShift *unit);
    //	void PV_MagShift_Dtor(PV_MagShift *unit);
    //	void PV_MagShift_next(PV_MagShift *unit, int inNumSamples);
    //
    //	void PV_MagNoise_Ctor(PV_Unit *unit);
    //	void PV_MagNoise_next(PV_Unit *unit, int inNumSamples);
    //
    
    //	void PV_BinWipe_Ctor(PV_Unit *unit);
    //	void PV_BinWipe_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_LocalMax_Ctor(PV_Unit *unit);
    //	void PV_LocalMax_next(PV_Unit *unit, int inNumSamples);
    //
    //	void PV_RandComb_Ctor(PV_RandComb *unit);
    //	void PV_RandComb_Dtor(PV_RandComb *unit);
    //	void PV_RandComb_next(PV_RandComb *unit, int inNumSamples);
    //
    //	void PV_RandWipe_Ctor(PV_RandWipe *unit);
    //	void PV_RandWipe_Dtor(PV_RandWipe *unit);
    //	void PV_RandWipe_next(PV_RandWipe *unit, int inNumSamples);
    //
    //	void PV_Diffuser_Ctor(PV_Diffuser *unit);
    //	void PV_Diffuser_Dtor(PV_Diffuser *unit);
    //	void PV_Diffuser_next(PV_Diffuser *unit, int inNumSamples);
    //
    //	void PV_MagFreeze_Ctor(PV_MagFreeze *unit);
    //	void PV_MagFreeze_Dtor(PV_MagFreeze *unit);
    //	void PV_MagFreeze_next(PV_MagFreeze *unit, int inNumSamples);
    //
    //	void PV_BinScramble_Ctor(PV_BinScramble *unit);
    //	void PV_BinScramble_Dtor(PV_BinScramble *unit);
    //	void PV_BinScramble_next(PV_BinScramble *unit, int inNumSamples);
    
}

//////////////////////////////////////////////////////////////////////////////////////////////////


void WT_MagAbove_next(WT_Unit *unit, int inNumSamples)
{
	WT_GET_BUF
    
    float * target = buf->data;
    
	float thresh = ZIN0(1);
    
    //defaults to 1, so avoid first coefficient, where would kill smoothest level, lowest pass data
    int startbin = (int) ZIN0(2);
    
    if(startbin<0) startbin = 0; 
    //if (startbin>=numbins) startbin = numbins; //not needed, will do nothing here
    
    //printf("check bufnum %d \n",ibufnum); 
    
    
    for (int i=startbin; i<numbins; ++i) {
        
        float valnow = fabs(target[i]); 
        
        
        if (valnow < thresh) target[i] = 0.f;
        
        //        if(i<100)
        //        printf("check val %d is %5.12f \n",i,valnow);
        //        
    }
    
}

void WT_MagAbove_Ctor(WT_Unit *unit) {
    
	ZOUT0(0) = ZIN0(0);
    
	SETCALC(WT_MagAbove_next);
}



void WT_MagBelow_next(WT_Unit *unit, int inNumSamples)
{
	WT_GET_BUF
    
    float * target = buf->data;
    
	float thresh = ZIN0(1);
    
    int startbin = (int) ZIN0(2);
    
    if(startbin<0) startbin = 0; 
    
    for (int i=startbin; i<numbins; ++i) {
        
        float valnow = fabs(target[i]); 
        
        
        if (valnow >= thresh) target[i] = 0.f;
        
        //        if(i<100)
        //        printf("check val %d is %5.12f \n",i,valnow);
        //        
    }
    
}

void WT_MagBelow_Ctor(WT_Unit *unit) {
    
	ZOUT0(0) = ZIN0(0);
    
	SETCALC(WT_MagBelow_next);
}

void WT_MagClip_next(WT_Unit *unit, int inNumSamples)
{
	WT_GET_BUF
    
    float * target = buf->data;
    
	float thresh = ZIN0(1);
    float negthresh = -1.0f*thresh; 
    
    int startbin = (int) ZIN0(2);
    
    if(startbin<0) startbin = 0; 
    
    //avoid first coefficient...
    for (int i=startbin; i<numbins; ++i) {
        
        float valnow = target[i]; 
        
        if (valnow > thresh) target[i] = thresh;
        else if (valnow < negthresh) target[i] = negthresh;
        
        //        if(i<100)
        //        printf("check val %d is %5.12f \n",i,valnow);
        //        
    }
    
}

void WT_MagClip_Ctor(WT_Unit *unit) {
    
	ZOUT0(0) = ZIN0(0);
    
	SETCALC(WT_MagClip_next);
}


void WT_SoftThreshold_next(WT_Unit *unit, int inNumSamples)
{
	WT_GET_BUF
    
    float * target = buf->data;
    
	float thresh = ZIN0(1);
    float negthresh = -1.0f*thresh; 
    
    int startbin = (int) ZIN0(2);
    
    if(startbin<0) startbin = 0; 
    
    for (int i=startbin; i<numbins; ++i) {
        
        float valnow = target[i]; 
        
        if (valnow > thresh) target[i] = valnow-thresh;
        else if (valnow < negthresh) target[i] = valnow+thresh;
        else target[i] = 0.0f;
        
        //        if(i<100)
        //        printf("check val %d is %5.12f \n",i,valnow);
        //        
    }
    
}

void WT_SoftThreshold_Ctor(WT_Unit *unit) {
    
	ZOUT0(0) = ZIN0(0);
    
	SETCALC(WT_SoftThreshold_next);
}


void WT_FilterScale_next(WT_Unit *unit, int inNumSamples)
{
	WT_GET_BUF
    
    int numscales = log2(numbins);
    
    float * target = buf->data;
    
	float wipe = ZIN0(1);
    
    int i,j; 
    
	if ((wipe > 0) && (wipe<=1.0)) {
        
        int scalesuppress = sc_min((int)(wipe*numscales+0.5),numscales);
        
        int baseindex=1; 
        int mult= 1; 
        
        for (i=0; i<scalesuppress; ++i) {
            
            float * now = target+ baseindex; 
            
            for(j=0; j<mult; ++j) {
                now[j] = 0.0f; 
            }
            
            baseindex += mult;  
            
            mult *=2; 
        }
        
	} else if ((wipe < 0) && (wipe>=(-1.0))) {
		
        wipe = wipe*(-1.0); 
        
        int scalesuppress = numscales- sc_min((int)(wipe*numscales+0.5),numscales);
        
        int baseindex=1; 
        int mult= 1; 
        
        for (i=0; i<numscales; ++i) {
            
            if(i>=scalesuppress) {
                float * now = target+ baseindex; 
                
                for(j=0; j<mult; ++j) {
                    now[j] = 0.0f; 
                }
            }
            
            baseindex += mult;  
            
            mult *=2; 
        }
        
        
	}
}

void WT_FilterScale_Ctor(WT_Unit *unit)
{
	SETCALC(WT_FilterScale_next);
	ZOUT0(0) = ZIN0(0);
}


void WT_Mul_next(WT_Unit *unit, int inNumSamples)
{
	WT_GET_BUF2
    
    float * data1 = buf1->data;
    float * data2 = buf2->data; 
    
    //leave coefficient 0 alone, only use abs of data2 to avoid messing up signs too much
	for (int i=1; i<numbins; ++i) {
		data1[i] *= fabs(data2[i]);
	}
}

void WT_Mul_Ctor(WT_Unit *unit)
{
	SETCALC(WT_Mul_next);
	ZOUT0(0) = ZIN0(0);
}



void WT_TimeWipe_Ctor(WT_Unit *unit) {
    
    SETCALC(WT_TimeWipe_next);
	ZOUT0(0) = ZIN0(0);
    
}

//wipe a proportion of bins nearer time 0 for each scale
void WT_TimeWipe_next(WT_Unit *unit, int inNumSamples) {
    
    WT_GET_BUF
    
    int numscales = log2(numbins);
    
    float * target = buf->data;
    
	float wipe = ZIN0(1);
    
    int i,j; 
    
	if ((wipe > 0) && (wipe<=1.0)) {
        
        int baseindex=1; 
        int mult= 1; 
        
        for (i=0; i<numscales; ++i) {
            
            float * now = target+ baseindex; 
            
            int timesuppress = sc_min((int)(wipe*mult+0.5),mult);
            
            for(j=0; j<timesuppress; ++j) {
                now[j] = 0.0f; 
            }
            
            baseindex += mult;  
            
            mult *=2; 
        }
        
	} 
    
}


void WT_Freeze_next(WT_Freeze *unit, int inNumSamples)
{
    int i, j; 
    
	WT_GET_BUF
    
	float freeze = ZIN0(1);
    
    float * target = buf->data;
    
    int numscales = log2(numbins);
    
    float scalewipe = ZIN0(2);
    int freezescale = scalewipe*numscales; 
    
    if(freezescale<1) freezescale = 1; 
    if(freezescale>numscales) freezescale = numscales; 
    
	if (!unit->coefficientstore_) {
		unit->coefficientstore_ = (float*)RTAlloc(unit->mWorld, numbins * sizeof(float));
		unit->numbins_ = numbins;
        // The first fft frame must use the else branch below
        // so that unit->m_mags gets populated with actual mag data
        // before reading; otherwise it might be used uninitialized.
		freeze = 0.f;
	} else if (numbins != unit->numbins_) return;
    
    float * source = unit->coefficientstore_;
    
    int baseindex=1; 
    int mult= 1; 
    
	if (freeze > 0.f) {
        
        for (i=0; i<freezescale; ++i) {
            
            float * now = target+ baseindex; 
            float * sourcenow = source+ baseindex; 
            
            for(j=0; j<mult; ++j) {
                now[j] = sourcenow[j]; 
            }
            
            baseindex += mult;  
            
            mult *=2; 
        }
        
	} else {
        
		for (int i=0; i<numbins; ++i) {
			source[i] = target[i];
		}
        
	}
}


void WT_Freeze_Ctor(WT_Freeze* unit)
{
	SETCALC(WT_Freeze_next);
	ZOUT0(0) = ZIN0(0);
	unit->coefficientstore_ = 0;
}

void WT_Freeze_Dtor(WT_Freeze* unit)
{
	RTFree(unit->mWorld, unit->coefficientstore_);
}



void WT_Onset_Ctor(WT_Onset* unit)
{
	SETCALC(WT_Onset_next);
    unit->onsetcount_ = 0; 
	ZOUT0(0) = 0.0f; //ZIN0(0);
	unit->coefficientstore_ = 0;
    unit->outval_ = 0.0f; 
}

void WT_Onset_Dtor(WT_Onset* unit)
{
	RTFree(unit->mWorld, unit->coefficientstore_);
}


void WT_Onset_next(WT_Onset *unit, int inNumSamples)
{
    int i, j; 
    
	float fbufnum = ZIN0(0); 
    
    int mingap = ZIN0(2); 
    
    if(mingap<0) mingap=0; 
    //safety
    if(unit->onsetcount_>mingap) 
        unit->onsetcount_ = mingap; 
    
    //if(mingap>1000) mingap=1000; 
    
    float rawodf = ZIN0(3); 
    
    
    //onsets detected on new window boundaries only
	if (fbufnum < 0.f) { ZOUT0(0) = unit->outval_; return; } 
    
	uint32 ibufnum = (uint32)fbufnum; 
	World *world = unit->mWorld; 
	SndBuf *buf; 
	if (ibufnum >= world->mNumSndBufs) { 
		int localBufNum = ibufnum - world->mNumSndBufs; 
		Graph *parent = unit->mParent; 
		if(localBufNum <= parent->localBufNum) { 
			buf = parent->mLocalSndBufs + localBufNum; 
		} else { 
			buf = world->mSndBufs; 
		} 
	} else { 
		buf = world->mSndBufs + ibufnum; 
	} 
	LOCK_SNDBUF(buf); 
	int numbins = buf->samples; 
    
	float threshold = ZIN0(1);
    
    float * target = buf->data;
    
   
	if (!unit->coefficientstore_) {
        
        //printf("allocate \n");
        
		unit->coefficientstore_ = (float*)RTAlloc(unit->mWorld, numbins * sizeof(float));
		unit->numbins_ = numbins;
        
        float * source = unit->coefficientstore_;
        
        for (int i=0; i<numbins; ++i) 
            source[i]= target[i]; 
        
        ZOUT0(0) = 0.0f; 
        
        //printf("post allocate \n");
        
        
	} else { 
        
        
        float * source = unit->coefficientstore_;
        
        
        if (numbins != unit->numbins_) {
            
            ZOUT0(0) = 0.0f; 
            return; 
            
        }
        
        float summation = 0.0f; 
         
        for (int i=0; i<numbins; ++i) {
            
            float valnow = target[i]; 
            float valprev = source[i]; 
            
            summation += fabs(valnow-valprev); 
            
            
            //printf("i %d sum %f now %f prev %f \n", i, summation, valnow, valprev);
            
            
            source[i]= valnow; 
            
        }
        
        summation /= (float)numbins; //divide by numbins to avoid big detection function
        
  
        
        float onsetoutput = 0.0f; 
        
        if(unit->onsetcount_>0) 
            --unit->onsetcount_; 
        
        if(unit->onsetcount_==0) {
            if(summation>threshold) {
                
                onsetoutput = 1.0f; 
                
                unit->onsetcount_=mingap; 
            }
        }
        
        if(rawodf>0.5) {
            
            unit->outval_ = summation; 
            ZOUT0(0) = summation; 
        }
        else
            ZOUT0(0) = onsetoutput; 
        
        
    }
    
    
    
}




void WT_Transient_Ctor(WT_Transient *unit) {
    
    unit->scales_ = 0; 
    
	ZOUT0(0) = ZIN0(0);
    
    SETCALC(WT_Transient_next);
    
}

void WT_Transient_Dtor(WT_Transient *unit) {
    
    if (unit->scales_!=0) {
     	RTFree(unit->mWorld, unit->scales_);   
        RTFree(unit->mWorld, unit->scalesize_);
        RTFree(unit->mWorld, unit->sumtoscale_);
        RTFree(unit->mWorld, unit->scalestatus_);
        RTFree(unit->mWorld, unit->scalepos_);
        RTFree(unit->mWorld, unit->mask_);
        RTFree(unit->mWorld, unit->maskscales_);
          
    }
    
}

void WT_Transient_next(WT_Transient *unit, int inNumSamples) {
  
    int i; 
    int numscales = unit->numscales_; 
    float ** scales = unit->scales_; 
    int * scalesize = unit->scalesize_; 
    float * mask = unit->mask_; 
    float ** maskscales = unit->maskscales_; 
    
    WT_GET_BUF
    

    float * source = buf->data;

	if (scales==0) {
        
        //printf("allocation begin \n"); 
        
        
        numscales = log2(numbins);
        unit->numscales_ = numscales;
        
		unit->scales_ = (float**)RTAlloc(unit->mWorld, numscales * sizeof(float*));
        unit->scalesize_ = (int*)RTAlloc(unit->mWorld, numscales * sizeof(int));
        
        unit->sumtoscale_ = (float*)RTAlloc(unit->mWorld, numscales * sizeof(float));
        unit->scalestatus_ = (int*)RTAlloc(unit->mWorld, numscales * sizeof(int));
        unit->scalepos_ = (int*)RTAlloc(unit->mWorld, numscales * sizeof(int));
        unit->mask_ = (float*)RTAlloc(unit->mWorld, numbins * sizeof(float));
        unit->maskscales_ = (float**)RTAlloc(unit->mWorld, numscales * sizeof(float*));
        
        scales = unit->scales_; 
        scalesize = unit->scalesize_; 
        
        maskscales = unit->maskscales_; 
        mask = unit->mask_;
        
        int baseindex=1; 
        int mult= 1; 
        
        for (i=0; i<numscales; ++i) {
            
            scales[i] = source + baseindex; 
            scalesize[i] = mult; 
            maskscales[i] = mask + baseindex; 
            
            baseindex += mult;  
            
            mult *=2; 
        }
        
        
        //printf("allocation end \n");
        
	} else if (source != (scales[0]-1)) return;
    

    float branchthreshold = ZIN0(1);
    float prunethreshold = ZIN0(2);
    
    float * sumtoscale = unit->sumtoscale_;
    int * scalestatus = unit->scalestatus_;
    int * scalepos = unit->scalepos_;
   
    //preparation
    
    
    //zero mask
    
    for (i=0; i<numbins; ++i)
        mask[i] = 0.f; 
    
    for (i=0; i<numscales; ++i) {
        scalestatus[i] = 0; 
        scalepos[i] = 0; 
    }
        
    
    //printf("tree walk start t1 %f t2 %f \n",branchthreshold,prunethreshold);
    
    int scalenow=1; //no point dealing at scale 0, go straight in at 1
    int topscale = numscales-1; 
    
    //scalepos[0]= 0; 
    //scalepos[1]= 0; 
    //sumtoscale[0] never used
    sumtoscale[1] = source[1]; //sum below this scale point
    
    float sumnow = 0.0f; 
    float valnow; 
    
    //one sweep through tree, writing confirmed nodes into mask
    //running status via stack info in scalestatus, scalepos, sumtoscale
    
    bool stillgoing = true; 
    
    //int count = 0; 
    
    while(scalepos[1]<2) {
        
        //++count; 
        
        stillgoing = true;
        
        //int sizecheck = scalesize[scalenow]; 
        
        int posnow = scalepos[scalenow]; 
        
        int status = scalestatus[scalenow]; 
        
        float valnow = fabs(scales[scalenow][posnow]); //modulus
        
        //printf("test %d %d %d %f\n", scalenow, posnow, status, valnow); 
        
       
        if(scalenow==topscale) {
 
            sumnow += valnow; 
            
            if(sumnow>=branchthreshold) {
                
                status = 2; 
                
                //and all lower nodes on branch? no, will propagate itself
                
            }
            
            
            
        } else if (status==0) {
            
            sumnow += valnow; 
            
            scalestatus[scalenow] = 1; 
            ++scalenow;   
            sumtoscale[scalenow] = sumnow; 
            stillgoing = false;
        }
        
        if(stillgoing) {
            
        if(status==2) {
            
            if(valnow>=prunethreshold) {
                
                //copy branch from here to mask 

                mask[1] = 0.0f; 
                
                int baseindex2=1;
                int mult2 = 1; 
                
                //have to take account of accumulating binary choices
                for (i=1; i<=scalenow; ++i) {
                    
                    baseindex2 = baseindex2*2 + scalepos[i];
                    
                    mask[baseindex2] = 1.f;  
                      
                }
                
                
                
            } else
            {
                
                //propagate status 2 lower
                scalestatus[scalenow-1] = 2; 
                
                
            }
                
            scalestatus[scalenow] = 0; 
        }
            
        //has served purpose since moving one forwards now    
        if(status==1)  
           scalestatus[scalenow] = 0;  
            
        ++posnow; 
        scalepos[scalenow] = posnow;
        
        if((posnow>=2) && (scalenow>1)) {
            
            //zero this level and all levels ABOVE
            
            for (i=scalenow; i<numscales; ++i) {
                scalestatus[i] = 0; 
                scalepos[i] = 0; 
            }
            
            //down one level
            
            --scalenow; 
    
            sumnow = sumtoscale[scalenow]; 
        } else {
           
            
           scalestatus[scalenow] = 0;  
            
        }
            
        }

    }
    
    //printf("tree walked count %d \n",count);
    
    
    if(mask[1]>0.5) 
        mask[0] = 1.0f; 

    //zero out any coefficients not in mask 
    
    int mode = ZIN0(3);
    
    if(mode==0) {
    for (i=0; i<numbins; ++i)
        source[i] *= mask[i];  
    }
    else {
        for (i=0; i<numbins; ++i)
            source[i] = mask[i]; 
    }
        
    //alternative 2 sweep strategy (more top down from top scale)
    //create full tree
    //prune tree
    
    
}



void WT_ModulusSum_Ctor(WT_ModulusSum *unit) {

	ZOUT0(0) = 0.0f;
    
    unit->outval_ = 0.0f; 
    
    SETCALC(WT_ModulusSum_next);
    
}

void WT_ModulusSum_next(WT_ModulusSum *unit, int inNumSamples) {
    
    int i, j; 
    
	float fbufnum = ZIN0(0); 
    
    //onsets detected on new window boundaries only
	if (fbufnum < 0.f) { ZOUT0(0) = unit->outval_; return; } 
    
	uint32 ibufnum = (uint32)fbufnum; 
	World *world = unit->mWorld; 
	SndBuf *buf; 
	if (ibufnum >= world->mNumSndBufs) { 
		int localBufNum = ibufnum - world->mNumSndBufs; 
		Graph *parent = unit->mParent; 
		if(localBufNum <= parent->localBufNum) { 
			buf = parent->mLocalSndBufs + localBufNum; 
		} else { 
			buf = world->mSndBufs; 
		} 
	} else { 
		buf = world->mSndBufs + ibufnum; 
	} 
	LOCK_SNDBUF(buf); 
	int numbins = buf->samples; 

    float * source = buf->data;
    
    float summation = 0.0f; 
        
    for (int i=0; i<numbins; ++i)
        summation += fabs(source[i]); 
            
    unit->outval_ = summation; 
    
    ZOUT0(0) = summation;
    
    
}






//void PV_LocalMax_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF
//
//	SCPolarBuf *p = ToPolarApx(buf);
//
//	float thresh = ZIN0(1);
//	float dc, nyq, mag;
//
//	// DC is only compared with the one above it
//	dc = std::abs(p->dc);
//	mag = p->bin[0].mag;
//	if(dc < thresh || dc < mag) p->dc = 0.f;
//	// 0th bin compared against DC and 1th
//	if(mag < thresh || mag < dc || mag < p->bin[1].mag) p->bin[0].mag = 0.f;
//	// All the middling bins
//	for (int i=1; i<numbins-1; ++i) {
//		float mag = p->bin[i].mag;
//		if (mag < thresh || mag < p->bin[i-1].mag || mag < p->bin[i+1].mag) {
//			p->bin[i].mag = 0.f;
//		}
//	}
//	// Penultimate is compared against the one below and the nyq
//	nyq = std::abs(p->nyq);
//	mag = p->bin[numbins-1].mag;
//	if(mag < thresh || mag < nyq || mag < p->bin[numbins-2].mag) p->bin[numbins-1].mag = 0.f;
//	// Nyquist compared against penultimate
//	if(nyq < thresh || nyq < mag) p->nyq = 0.f;
//}
//
//void PV_LocalMax_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_LocalMax_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_MagSmear_next(PV_MagSmear *unit, int inNumSamples)
//{
//	PV_GET_BUF
//	MAKE_TEMP_BUF
//
//	SCPolarBuf *p = ToPolarApx(buf);
//	SCPolarBuf *q = (SCPolarBuf*)unit->m_tempbuf;
//
//	int width = (int)ZIN0(1);
//	width = sc_clip(width, 0, numbins-1);
//	float scale = 1.f / (2*width+1);
//
//	q->dc = p->dc;
//	q->nyq = p->nyq;
//	for (int j=0; j<numbins; j++) {
//		float sum = 0.0;
//		for (int pos = j-width; pos <= j+width; pos++) {
//			if (pos >= 0 && pos < numbins) {
//				sum += p->bin[pos].mag;
//			}
//		}
//		q->bin[j].Set( sum * scale, p->bin[j].phase );
//	}
//	for (int i=0; i<numbins; i++) {
//		p->bin[i] = q->bin[i];
//	}
//}
//
//void PV_MagSmear_Ctor(PV_MagSmear *unit)
//{
//	SETCALC(PV_MagSmear_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_tempbuf = 0;
//}
//
//void PV_MagSmear_Dtor(PV_MagSmear *unit)
//{
//	RTFree(unit->mWorld, unit->m_tempbuf);
//}
//
//void PV_BinShift_next(PV_BinShift *unit, int inNumSamples)
//{
//	PV_GET_BUF
//	MAKE_TEMP_BUF
//
//	// get shift and stretch params
//	float stretch = ZIN0(1);
//	float shift = ZIN0(2);
//
//	SCComplexBuf *p = ToComplexApx(buf);
//	SCComplexBuf *q = (SCComplexBuf*)unit->m_tempbuf;
//
//	// initialize output buf to zeroes
//	for (int i=0; i<numbins; ++i) {
//		q->bin[i] = 0.f;
//	}
//
//	float fpos;
//	int i;
//	q->dc = p->dc;
//	q->nyq = p->nyq;
//	for (i=0, fpos = shift; i < numbins; ++i, fpos += stretch) {
//		int32 pos = (int32)(fpos + 0.5);
//		if (pos >= 0 && pos < numbins) {
//			q->bin[pos] += p->bin[i];
//		}
//	}
//	memcpy(p->bin, q->bin, numbins * sizeof(SCComplex));
//
//}
//
//void PV_BinShift_Ctor(PV_BinShift *unit)
//{
//	SETCALC(PV_BinShift_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_tempbuf = 0;
//}
//
//void PV_BinShift_Dtor(PV_BinShift *unit)
//{
//	RTFree(unit->mWorld, unit->m_tempbuf);
//}
//
//void PV_MagShift_next(PV_MagShift *unit, int inNumSamples)
//{
//	PV_GET_BUF
//	MAKE_TEMP_BUF
//
//	// get shift and stretch params
//	float stretch = ZIN0(1);
//	float shift = ZIN0(2);
//
//	SCPolarBuf *p = ToPolarApx(buf);
//	SCPolarBuf *q = (SCPolarBuf*)unit->m_tempbuf;
//
//	// initialize output buf to zeroes
//	for (int i=0; i<numbins; ++i) {
//		q->bin[i].mag = 0.f;
//		q->bin[i].phase = p->bin[i].phase;
//	}
//
//	float fpos;
//	int i;
//	q->dc = p->dc;
//	q->nyq = p->nyq;
//	for (i=0, fpos = shift; i < numbins; ++i, fpos += stretch) {
//		int32 pos = (int32)(fpos + 0.5);
//		if (pos >= 0 && pos < numbins) {
//			q->bin[pos].mag += p->bin[i].mag;
//		}
//	}
//	memcpy(p->bin, q->bin, numbins * sizeof(SCComplex));
//}
//
//void PV_MagShift_Ctor(PV_MagShift *unit)
//{
//	SETCALC(PV_MagShift_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_tempbuf = 0;
//}
//
//void PV_MagShift_Dtor(PV_MagShift *unit)
//{
//	RTFree(unit->mWorld, unit->m_tempbuf);
//}
//
//void PV_MagNoise_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF
//
//	RGET
//	if (buf->coord == coord_Complex) {
//		SCComplexBuf *p = (SCComplexBuf*)buf->data;
//		for (int i=0; i<numbins; ++i) {
//			float r = frand2(s1, s2, s3);
//			p->bin[i].real *= r;
//			p->bin[i].imag *= r;
//		}
//		p->dc  *= frand2(s1, s2, s3);
//		p->nyq *= frand2(s1, s2, s3);
//	} else {
//		SCPolarBuf *p = (SCPolarBuf*)buf->data;
//		for (int i=0; i<numbins; ++i) {
//			float r = frand2(s1, s2, s3);
//			p->bin[i].mag *= r;
//		}
//		p->dc  *= frand2(s1, s2, s3);
//		p->nyq *= frand2(s1, s2, s3);
//	}
//	RPUT
//}
//
//void PV_MagNoise_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_MagNoise_next);
//	ZOUT0(0) = ZIN0(0);
//}
//

//void PV_MagSquared_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_MagSquared_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_BrickWall_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF
//
//	SCComplexBuf *p = (SCComplexBuf*)buf->data;
//
//	int wipe = (int)(ZIN0(1) * numbins);
//	if (wipe > 0) {
//		wipe = sc_min(wipe, numbins);
//		p->dc = 0.f;
//		for (int i=0; i < wipe; ++i) {
//			p->bin[i] = 0.f;
//		}
//		if(wipe==numbins) p->nyq = 0.f;
//	} else if (wipe < 0) {
//		wipe = sc_max(wipe, -numbins);
//		if(wipe==-numbins) p->dc = 0.f;
//		for (int i=numbins+wipe; i < numbins; ++i) {
//			p->bin[i] = 0.f;
//		}
//		p->nyq = 0.f;
//	}
//}
//
//void PV_BrickWall_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_BrickWall_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_BinWipe_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCComplexBuf *p = (SCComplexBuf*)buf1->data;
//	SCComplexBuf *q = (SCComplexBuf*)buf2->data;
//
//	int wipe = (int)(ZIN0(2) * numbins);
//	if (wipe > 0) {
//		wipe = sc_min(wipe, numbins);
//		p->dc = q->dc;
//		for (int i=0; i < wipe; ++i) {
//			p->bin[i] = q->bin[i];
//		}
//		if(wipe==numbins) p->nyq = q->nyq;
//	} else if (wipe < 0) {
//		wipe = sc_max(wipe, -numbins);
//		if(wipe==-numbins) p->dc = q->dc;
//		for (int i=numbins+wipe; i < numbins; ++i) {
//			p->bin[i] = q->bin[i];
//		}
//		p->nyq = q->nyq;
//	}
//}
//
//void PV_BinWipe_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_BinWipe_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_MagMul_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCPolarBuf *p = ToPolarApx(buf1);
//	SCPolarBuf *q = ToPolarApx(buf2);
//
//	p->dc *= q->dc;
//	p->nyq *= q->nyq;
//	for (int i=0; i<numbins; ++i) {
//		p->bin[i].mag *= q->bin[i].mag;
//	}
//}
//
//void PV_MagMul_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_MagMul_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_MagDiv_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCPolarBuf *p = ToPolarApx(buf1);
//	SCPolarBuf *q = ToPolarApx(buf2);
//
//	float zeroed = ZIN0(2);
//
//	p->dc /= sc_max(q->dc, zeroed);
//	p->nyq /= sc_max(q->nyq, zeroed);
//	for (int i=0; i<numbins; ++i) {
//		p->bin[i].mag /= sc_max(q->bin[i].mag, zeroed);
//	}
//}
//
//void PV_MagDiv_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_MagDiv_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_Copy_next(PV_Unit *unit, int inNumSamples)
//{
//	float fbufnum1 = ZIN0(0);
//	float fbufnum2 = ZIN0(1);
//	if (fbufnum1 < 0.f || fbufnum2 < 0.f) { ZOUT0(0) = -1.f; return; }
//	ZOUT0(0) = fbufnum2;
//	uint32 ibufnum1 = (int)fbufnum1;
//	uint32 ibufnum2 = (int)fbufnum2;
//	World *world = unit->mWorld;
//	SndBuf *buf1;
//	SndBuf *buf2;
//	if (ibufnum1 >= world->mNumSndBufs) {
//		int localBufNum = ibufnum1 - world->mNumSndBufs;
//		Graph *parent = unit->mParent;
//		if(localBufNum <= parent->localBufNum) {
//			buf1 = parent->mLocalSndBufs + localBufNum;
//		} else {
//			buf1 = world->mSndBufs;
//		}
//	} else {
//		buf1 = world->mSndBufs + ibufnum1;
//	}
//	if (ibufnum2 >= world->mNumSndBufs) {
//		int localBufNum = ibufnum2 - world->mNumSndBufs;
//		Graph *parent = unit->mParent;
//		if(localBufNum <= parent->localBufNum) {
//			buf2 = parent->mLocalSndBufs + localBufNum;
//		} else {
//			buf2 = world->mSndBufs;
//		}
//	} else {
//		buf2 = world->mSndBufs + ibufnum2;
//	}
//
//	if (buf1->samples != buf2->samples) return;
//	int numbins = buf1->samples - 2 >> 1;
//
//	// copy to buf2
//	LOCK_SNDBUF2_SHARED_EXCLUSIVE(buf1, buf2);
//	buf2->coord = buf1->coord;
//	memcpy(buf2->data, buf1->data, buf1->samples * sizeof(float));
//}
//
//void PV_Copy_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_Copy_next);
//	ZOUT0(0) = ZIN0(1);
//}

//
//void PV_Mul_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCComplexBuf *p = ToComplexApx(buf1);
//	SCComplexBuf *q = ToComplexApx(buf2);
//
//	float preal, realmul, imagmul;
//
//	p->dc *= q->dc;
//	p->nyq *= q->nyq;
//	for (int i=0; i<numbins; ++i) {
//		preal = p->bin[i].real;
//		// Complex multiply using only 3 multiplications rather than 4. http://mathworld.wolfram.com/ComplexMultiplication.html
//		realmul = (preal * q->bin[i].real);
//		imagmul = (p->bin[i].imag * q->bin[i].imag);
//		p->bin[i].real = realmul - imagmul;
//		p->bin[i].imag = (preal + p->bin[i].imag) * (q->bin[i].real + q->bin[i].imag) - realmul - imagmul;
//	}
//}
//
//void PV_Mul_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_Mul_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_Div_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCComplexBuf *p = ToComplexApx(buf1);
//	SCComplexBuf *q = ToComplexApx(buf2);
//
//	p->dc  /= q->dc;
//	p->nyq /= q->nyq;
//	for (int i=0; i<numbins; ++i) {
//		// See http://gcc.gnu.org/ml/fortran/2004-05/msg00029.html
//		// Note that hypot has danger of overflow (see URL above),
//		// however FFT values typically stay within a small range,
//		// so I'm considering this OK for now.
//		float hypot = q->bin[i].real * q->bin[i].real + q->bin[i].imag * q->bin[i].imag;
//		float preal = p->bin[i].real;
//		p->bin[i].real = (preal          * q->bin[i].real + p->bin[i].imag * q->bin[i].imag) / hypot;
//		p->bin[i].imag = (p->bin[i].imag * q->bin[i].real - preal          * q->bin[i].imag) / hypot;
//	}
//}
//
//void PV_Div_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_Div_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_Add_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCComplexBuf *p = ToComplexApx(buf1);
//	SCComplexBuf *q = ToComplexApx(buf2);
//
//	p->dc += q->dc;
//	p->nyq += q->nyq;
//	for (int i=0; i<numbins; ++i) {
//		p->bin[i].real += q->bin[i].real;
//		p->bin[i].imag += q->bin[i].imag;
//	}
//}
//
//void PV_Add_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_Add_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_Max_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCPolarBuf *p = ToPolarApx(buf1);
//	SCPolarBuf *q = ToPolarApx(buf2);
//
//	if(std::abs(q->dc ) > std::abs(p->dc )) p->dc  = q->dc ;
//	if(std::abs(q->nyq) > std::abs(p->nyq)) p->nyq = q->nyq;
//	for (int i=0; i<numbins; ++i) {
//		if (q->bin[i].mag > p->bin[i].mag) {
//			p->bin[i] = q->bin[i];
//		}
//	}
//}
//
//void PV_Max_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_Max_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_Min_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	SCPolarBuf *p = ToPolarApx(buf1);
//	SCPolarBuf *q = ToPolarApx(buf2);
//
//	if(std::abs(q->dc ) < std::abs(p->dc )) p->dc  = q->dc ;
//	if(std::abs(q->nyq) < std::abs(p->nyq)) p->nyq = q->nyq;
//	for (int i=0; i<numbins; ++i) {
//		if (q->bin[i].mag < p->bin[i].mag) {
//			p->bin[i] = q->bin[i];
//		}
//	}
//}
//
//void PV_Min_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_Min_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_RectComb_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF
//
//	float numTeeth = ZIN0(1);
//	float phase = ZIN0(2);
//	float width = ZIN0(3);
//	float freq = numTeeth / (numbins + 1);
//
//	SCComplexBuf *p = (SCComplexBuf*)buf->data;
//
//	if (phase > width) p->dc = 0.f;
//		phase += freq;
//		if (phase >= 1.f) phase -= 1.f;
//		else if (phase < 0.f) phase += 1.f;
//
//	for (int i=0; i < numbins; ++i) {
//		if (phase > width) p->bin[i] = 0.f;
//		phase += freq;
//		if (phase >= 1.f) phase -= 1.f;
//		else if (phase < 0.f) phase += 1.f;
//	}
//
//	if (phase > width) p->nyq = 0.f;
//}
//
//void PV_RectComb_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_RectComb_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//void PV_RectComb2_next(PV_Unit *unit, int inNumSamples)
//{
//	PV_GET_BUF2
//
//	float numTeeth = ZIN0(2);
//	float phase = ZIN0(3);
//	float width = ZIN0(4);
//	float freq = numTeeth / (numbins + 1);
//
//	SCComplexBuf *p = (SCComplexBuf*)buf1->data;
//	SCComplexBuf *q = (SCComplexBuf*)buf2->data;
//
//	if (phase > width) p->dc = q->dc;
//		phase += freq;
//		if (phase >= 1.f) phase -= 1.f;
//		else if (phase < 0.f) phase += 1.f;
//
//	for (int i=0; i < numbins; ++i) {
//		if (phase > width) p->bin[i] = q->bin[i];
//		phase += freq;
//		if (phase >= 1.f) phase -= 1.f;
//		else if (phase < 0.f) phase += 1.f;
//	}
//
//	if (phase > width) p->nyq = q->nyq;
//}
//
//void PV_RectComb2_Ctor(PV_Unit *unit)
//{
//	SETCALC(PV_RectComb2_next);
//	ZOUT0(0) = ZIN0(0);
//}
//
//
//static void PV_RandComb_choose(PV_RandComb* unit)
//{
//	int numbins = unit->m_numbins;
//	for (int i=0; i<numbins; ++i) {
//		unit->m_ordering[i] = i;
//	}
//	RGET
//	for (int i=0; i<numbins; ++i) {
//		int32 j = (int32)(frand(s1,s2,s3) * (numbins - i));
//		int32 temp = unit->m_ordering[i];
//		unit->m_ordering[i] = unit->m_ordering[j];
//		unit->m_ordering[j] = temp;
//	}
//	RPUT
//}
//
//void PV_RandComb_next(PV_RandComb *unit, int inNumSamples)
//{
//	float trig = ZIN0(2);
//	if (trig > 0.f && unit->m_prevtrig <= 0.f) unit->m_triggered = true;
//	unit->m_prevtrig = trig;
//
//	PV_GET_BUF
//
//	if (!unit->m_ordering) {
//		unit->m_ordering = (int*)RTAlloc(unit->mWorld, numbins * sizeof(int));
//		unit->m_numbins = numbins;
//		PV_RandComb_choose(unit);
//	} else {
//		if (numbins != unit->m_numbins) return;
//		if (unit->m_triggered) {
//			unit->m_triggered = false;
//			PV_RandComb_choose(unit);
//		}
//	}
//
//	int n = (int)(ZIN0(1) * numbins);
//	n = sc_clip(n, 0, numbins);
//
//	SCComplexBuf *p = (SCComplexBuf*)buf->data;
//
//	int *ordering = unit->m_ordering;
//	for (int i=0; i<n; ++i) {
//		p->bin[ordering[i]] = 0.f;
//	}
//	if(n==numbins){
//		// including dc and nyq in the above shuffle would add too much complexity. but at full "wipe" we should get silence.
//		p->dc  = 0.f;
//		p->nyq = 0.f;
//	}
//}
//
//
//void PV_RandComb_Ctor(PV_RandComb* unit)
//{
//	SETCALC(PV_RandComb_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_ordering = 0;
//	unit->m_prevtrig = 0.f;
//	unit->m_triggered = false;
//}
//
//void PV_RandComb_Dtor(PV_RandComb* unit)
//{
//	RTFree(unit->mWorld, unit->m_ordering);
//}
//
////////////////////////
//
//static void PV_RandWipe_choose(PV_RandWipe* unit)
//{
//	int numbins = unit->m_numbins;
//	for (int i=0; i<numbins; ++i) {
//		unit->m_ordering[i] = i;
//	}
//	RGET
//	for (int i=0; i<numbins; ++i) {
//		int32 j = (int32)(frand(s1,s2,s3) * (numbins - i));
//		int32 temp = unit->m_ordering[i];
//		unit->m_ordering[i] = unit->m_ordering[j];
//		unit->m_ordering[j] = temp;
//	}
//	RPUT
//}
//
//void PV_RandWipe_next(PV_RandWipe *unit, int inNumSamples)
//{
//	float trig = ZIN0(3);
//	if (trig > 0.f && unit->m_prevtrig <= 0.f) unit->m_triggered = true;
//	unit->m_prevtrig = trig;
//
//	PV_GET_BUF2
//
//	if (!unit->m_ordering) {
//		unit->m_ordering = (int*)RTAlloc(unit->mWorld, numbins * sizeof(int));
//		unit->m_numbins = numbins;
//		PV_RandWipe_choose(unit);
//	} else {
//		if (numbins != unit->m_numbins) return;
//		if (unit->m_triggered) {
//			unit->m_triggered = false;
//			PV_RandWipe_choose(unit);
//		}
//	}
//
//	int n = (int)(ZIN0(2) * numbins);
//	n = sc_clip(n, 0, numbins);
//
//	SCComplexBuf *p = (SCComplexBuf*)buf1->data;
//	SCComplexBuf *q = (SCComplexBuf*)buf2->data;
//
//	int *ordering = unit->m_ordering;
//	for (int i=0; i<n; ++i) {
//		p->bin[ordering[i]] = q->bin[ordering[i]];
//	}
//}
//
//
//void PV_RandWipe_Ctor(PV_RandWipe* unit)
//{
//	SETCALC(PV_RandWipe_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_ordering = 0;
//	unit->m_prevtrig = 0.f;
//	unit->m_triggered = false;
//}
//
//void PV_RandWipe_Dtor(PV_RandWipe* unit)
//{
//	RTFree(unit->mWorld, unit->m_ordering);
//}
//
////////////////////////
//
//static void PV_Diffuser_choose(PV_Diffuser* unit)
//{
//	RGET
//	for (int i=0; i<unit->m_numbins; ++i) {
//		unit->m_shift[i] = frand(s1,s2,s3) * twopi;
//	}
//	RPUT
//}
//
//void PV_Diffuser_next(PV_Diffuser *unit, int inNumSamples)
//{
//	float trig = ZIN0(1);
//	if (trig > 0.f && unit->m_prevtrig <= 0.f) unit->m_triggered = true;
//	unit->m_prevtrig = trig;
//
//	PV_GET_BUF
//
//	if (!unit->m_shift) {
//		unit->m_shift = (float*)RTAlloc(unit->mWorld, numbins * sizeof(float));
//		unit->m_numbins = numbins;
//		PV_Diffuser_choose(unit);
//	} else {
//		if (numbins != unit->m_numbins) return;
//		if (unit->m_triggered) {
//			unit->m_triggered = false;
//			PV_Diffuser_choose(unit);
//		}
//	}
//
//	int n = (int)(ZIN0(1) * numbins);
//	n = sc_clip(n, 0, numbins);
//
//	SCPolarBuf *p = ToPolarApx(buf);
//
//	float *shift = unit->m_shift;
//	for (int i=0; i<n; ++i) {
//		p->bin[i].phase += shift[i];
//	}
//}
//
//
//void PV_Diffuser_Ctor(PV_Diffuser* unit)
//{
//	SETCALC(PV_Diffuser_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_shift = 0;
//	unit->m_prevtrig = 0.f;
//	unit->m_triggered = false;
//}
//
//void PV_Diffuser_Dtor(PV_Diffuser* unit)
//{
//	RTFree(unit->mWorld, unit->m_shift);
//}
//
//
////////////////////////
//
//static void PV_BinScramble_choose(PV_BinScramble* unit)
//{
//	int numbins = unit->m_numbins;
//	int *to = unit->m_to;
//	int *from = unit->m_from;
//
//	for (int i=0; i<numbins; ++i) {
//		to[i] = i;
//	}
//	RGET
//	for (int i=0; i<numbins; ++i) {
//		int32 j = (int32)(frand(s1,s2,s3) * (numbins - i));
//		int32 temp = to[i];
//		to[i] = to[j];
//		to[j] = temp;
//	}
//
//	int32 width = (int32)(ZIN0(2) * numbins);
//	for (int i=0; i<numbins; ++i) {
//		int32 k = to[i];
//		int32 minr = sc_max(0, k-width);
//		int32 maxr = sc_min(numbins-1, k+width);
//		int32 j = (int32)(frand(s1,s2,s3) * (maxr - minr) + minr);
//		from[i] = j;
//	}
//	RPUT
//}
//
//void PV_BinScramble_next(PV_BinScramble *unit, int inNumSamples)
//{
//	float trig = ZIN0(3);
//	if (trig > 0.f && unit->m_prevtrig <= 0.f) unit->m_triggered = true;
//	unit->m_prevtrig = trig;
//
//	PV_GET_BUF
//
//	if (!unit->m_to) {
//		unit->m_to = (int*)RTAlloc(unit->mWorld, numbins * 2 * sizeof(int));
//		unit->m_from = unit->m_to + numbins;
//		unit->m_numbins = numbins;
//		unit->m_tempbuf = (float*)RTAlloc(unit->mWorld, buf->samples * sizeof(float));
//		PV_BinScramble_choose(unit);
//	} else {
//		if (numbins != unit->m_numbins) return;
//		if (unit->m_triggered) {
//			unit->m_triggered = false;
//			PV_BinScramble_choose(unit);
//		}
//	}
//
//	SCComplexBuf *p = (SCComplexBuf*)buf->data;
//	SCComplexBuf *q = (SCComplexBuf*)unit->m_tempbuf;
//
//	float wipe = ZIN0(1);
//	int32 scrambleBins = (int32)(numbins * sc_clip(wipe, 0.f, 1.f));
//
//	int *to = unit->m_to;
//	int *from = unit->m_from;
//	for (int j=0; j<scrambleBins; j++) {
//		q->bin[to[j]] = p->bin[from[j]];
//	}
//	for (int j=scrambleBins; j<numbins; j++) {
//		int32 a = to[j];
//		q->bin[a] = p->bin[a];
//	}
//	q->dc = p->dc;
//	q->nyq = p->nyq;
//	memcpy(p->bin, q->bin, numbins * sizeof(SCComplex));
//}
//
//
//void PV_BinScramble_Ctor(PV_BinScramble* unit)
//{
//	SETCALC(PV_BinScramble_next);
//	ZOUT0(0) = ZIN0(0);
//	unit->m_to = 0;
//	unit->m_prevtrig = 0.f;
//	unit->m_triggered = false;
//	unit->m_tempbuf = 0;
//}
//
//void PV_BinScramble_Dtor(PV_BinScramble* unit)
//{
//	RTFree(unit->mWorld, unit->m_to);
//	RTFree(unit->mWorld, unit->m_tempbuf);
//}
//
//



////////////////////////////////////////////////////////////////////////////////////////////////////////

#define DefineWTUnit(name) \
(*ft->fDefineUnit)(#name, sizeof(WT_Unit), (UnitCtorFunc)&name##_Ctor, 0, 0);


//void initWT(InterfaceTable *inTable);


InterfaceTable *ft;


void initWT(InterfaceTable *inTable)
{
    
    ft = inTable; 
    
    DefineWTUnit(WT_MagAbove);
    DefineWTUnit(WT_MagBelow);
    DefineWTUnit(WT_MagClip);
    DefineWTUnit(WT_SoftThreshold);
    
	DefineWTUnit(WT_FilterScale);
	DefineWTUnit(WT_Mul);
    DefineWTUnit(WT_TimeWipe);
    DefineDtorUnit(WT_Freeze);
    
    DefineDtorUnit(WT_Onset);
    DefineDtorUnit(WT_Transient);
    
    DefineWTUnit(WT_ModulusSum);
    
    //	DefinePVUnit(PV_MagDiv);
    //	DefinePVUnit(PV_MagSquared);
    //	DefinePVUnit(PV_MagNoise);
    //	DefinePVUnit(PV_Copy);
    //	DefinePVUnit(PV_CopyPhase);
    //	DefinePVUnit(PV_Min);
    //	DefinePVUnit(PV_Max);
    //	DefinePVUnit(PV_Mul);
    //	DefinePVUnit(PV_Div);
    //	DefinePVUnit(PV_Add);
    //	DefinePVUnit(PV_RectComb);
    //	DefinePVUnit(PV_RectComb2);
    //	DefinePVUnit(PV_BrickWall);
    //	DefinePVUnit(PV_BinWipe);
    //	DefinePVUnit(PV_LocalMax);
    //	DefinePVUnit(PV_Conj);
    //
    //	DefineDtorUnit(PV_BinScramble);
    //	DefineDtorUnit(PV_MagSmear);
    //	DefineDtorUnit(PV_MagShift);
    //	DefineDtorUnit(PV_BinShift);
    //	DefineDtorUnit(PV_RandWipe);
    //	DefineDtorUnit(PV_Diffuser);
    //	DefineDtorUnit(PV_RandComb);
    //	DefineDtorUnit(PV_MagFreeze);
}
