/*
	SuperCollider real time audio synthesis system
 Copyright (c) 2002 James McCartney. All rights reserved.
	http://www.audiosynth.com
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
//This file is part of FiveIMSNickCollinsPhD. Copyright (C) 2006  Nicholas M.Collins distributed under the terms of the GNU General Public License full notice in file FiveIMSNickCollinsPhD.help


//EventCapture on the fly with loudness detector and average slope peakpicker 1.0 Nick Collins 3 October 2004

#include "SC_PlugIn.h"
#include <vecLib/vecLib.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

//helpful constants
#define PI 3.1415926535898f
#define TWOPI 6.28318530717952646f 

//FFT data 
#define N 1024  //FFT size
#define NOVER2 512  //FFT size
#define NOVER4 256  //FFT size
#define OVERLAP 768  //512
#define OVERLAPINDEX 256 //512
#define HOPSIZE 256 // 512  
#define FS 44100 //assumes fixed sampling rate
#define FRAMESR 172.2656    //86.1328
#define FRAMEPERIOD 0.00581 //0.01161 //seconds

//for hanning window, constructed in plugin initialisation
float hanning[N]; 

/*
 //equal loudness contour data ISO226-2002
 double data[29][11]={ { 76.6, 83.1, 89.3, 94.6, 99.5, 104.3, 108.9, 113.5, 118, 122.5, 127.1 }, { 67.3, 75.7, 82.9, 88.8, 94.1, 99.1, 104.1, 108.9, 113.7, 118.5, 123.3 }, { 58.4, 68.6, 76.6, 83, 88.7, 94.1, 99.3, 104.5, 109.6, 114.7, 119.9 }, { 50.2, 61.7, 70.4, 77.2, 83.3, 89, 94.6, 100.1, 105.6, 111, 116.2 }, { 43.2, 55.6, 64.8, 72, 78.5, 84.6, 90.5, 96.3, 102.1, 107.8, 113.5 }, { 36.7, 49.5, 59.2, 66.9, 73.7, 80.2, 86.4, 92.6, 98.7, 104.8, 110.9 }, { 30.7, 43.5, 53.6, 61.7, 68.9, 75.8, 82.5, 89, 95.5, 102, 108.4 }, { 25.7, 38.3, 48.7, 57.1, 64.7, 72, 79, 85.9, 92.8, 99.6, 106.4 }, { 21.3, 33.5, 44.1, 52.8, 60.8, 68.4, 75.8, 83.1, 90.3, 97.5, 104.7 }, { 17, 28.5, 39.2, 48.2, 56.6, 64.6, 72.4, 80.1, 87.8, 95.5, 103.1 }, { 13.6, 24.4, 35.1, 44.4, 53.1, 61.5, 69.7, 77.8, 85.8, 93.9, 100.9 }, { 10.7, 20.9, 31.5, 41, 50, 58.7, 67.2, 75.7, 84.1, 92.5, 100.9 }, { 8, 17.6, 28.2, 37.8, 47.1, 56.1, 65, 73.8, 82.6, 91.4, 100.2 }, { 5.7, 14.8, 25.3, 35.1, 44.6, 53.9, 63.1, 72.3, 81.4, 90.5, 99.6 }, { 3.9, 12.6, 23, 32.9, 42.6, 52.2, 61.6, 71.1, 80.5, 89.9, 99.3 }, { 2.5, 10.8, 21.1, 31.1, 40.9, 50.7, 60.5, 70.2, 79.9, 89.6, 99.3 }, { 1.9, 9.9, 20, 30.1, 40, 49.9, 59.8, 69.7, 79.6, 89.5, 99.4 }, { 2.2, 10, 20, 30, 40, 50, 60, 70, 80, 90, 100 }, { 3.2, 11.1, 21.2, 31.4, 41.5, 51.6, 61.7, 71.9, 82, 92.1, 102.3 }, { 1.8, 10.4, 21.1, 31.6, 41.9, 52.2, 62.4, 72.6, 82.8, 92.9, 102.8 }, { -1.2, 7.5, 18.3, 28.8, 39.1, 49.4, 59.6, 69.8, 80, 90.2, 100.4 }, { -4.3, 4.7, 15.6, 26.2, 36.6, 46.9, 57.2, 67.4, 77.6, 87.7, 97.6 }, { -6.3, 3.2, 14.4, 25.2, 35.6, 46, 56.3, 66.5, 76.7, 86.8, 96.7 }, { -6, 3.8, 15.3, 26.1, 36.7, 47, 57.3, 67.5, 77.7, 87.9, 98.1 }, { -2.5, 7.1, 18.4, 29.2, 39.7, 50.1, 60.4, 70.6, 80.8, 91.1, 101.6 }, { 5.4, 14, 24.6, 35, 45.3, 55.4, 65.6, 75.7, 85.8, 96, 106.4 }, { 13.4, 21.4, 31.5, 41.4, 51.2, 61.1, 70.9, 80.6, 90.4, 100.5, 111.1 }, { 15, 23.9, 34.3, 44.2, 53.7, 63.2, 72.6, 81.9, 91.2, 100.6, 110.2 }, { 15.6, 23.9, 33.6, 42.7, 51.5, 60.2, 68.7, 77.3, 85.8, 94, 101.7 } };
 
 //double freqs[29]={ 20, 25, 31.5, 40, 50, 63, 80, 100, 125, 160, 200, 250, 315, 400, 500, 630, 800, 1000, 1250, 1600, 2000, 2500, 3150, 4000, 5000, 6300, 8000, 10000, 12500 };
 
 double logfreqs[29]={ 0, 0.034661720963303, 0.07056116746849, 0.10766913951835, 0.14233086048165, 0.17823030698684, 0.2153382790367, 0.25, 0.2846617209633, 0.32300741855504, 0.35766913951835, 0.39233086048165, 0.42823030698684, 0.4653382790367, 0.5, 0.53589944650519, 0.57300741855504, 0.60766913951835, 0.64233086048165, 0.68067655807339, 0.7153382790367, 0.75, 0.78589944650519, 0.82300741855504, 0.85766913951835, 0.89356858602353, 0.93067655807339, 0.9653382790367, 1 };
 
 //for log calc and scaling
 double l20= 1.301029995664; //log10(20)
 double logmult= 0.35766913951835; //((12500.log10)-(20.log10)).reciprocal
 
 inline double logcalc(double freq);
 inline double logcalc(double freq)
 {return (log10(freq)-l20)*logmult;}
 */

int eqlbandbins[43]= {1,2,3,4,5,6,7,8,9,11,13,15,17,19,22,25,28,32,36,41,46,52,58,65,73,82,92,103,116,129,144,161,180,201,225,251,280,312,348,388,433,483,513}; 
int eqlbandsizes[42]= {1,1,1,1,1,1,1,1,2,2,2,2,2,3,3,3,4,4,5,5,6,6,7,8,9,10,11,13,13,15,17,19,21,24,26,29,32,36,40,45,50,30}; 
float contours[42][11]= {{ 47.88, 59.68, 68.55, 75.48, 81.71, 87.54, 93.24, 98.84,104.44,109.94,115.31},{ 29.04, 41.78, 51.98, 60.18, 67.51, 74.54, 81.34, 87.97, 94.61,101.21,107.74},{ 20.72, 32.83, 43.44, 52.18, 60.24, 67.89, 75.34, 82.70, 89.97, 97.23,104.49},{ 15.87, 27.14, 37.84, 46.94, 55.44, 63.57, 71.51, 79.34, 87.14, 94.97,102.37},{ 12.64, 23.24, 33.91, 43.27, 52.07, 60.57, 68.87, 77.10, 85.24, 93.44,100.90},{ 10.31, 20.43, 31.03, 40.54, 49.59, 58.33, 66.89, 75.43, 83.89, 92.34,100.80},{  8.51, 18.23, 28.83, 38.41, 47.65, 56.59, 65.42, 74.16, 82.89, 91.61,100.33},{  7.14, 16.55, 27.11, 36.79, 46.16, 55.27, 64.29, 73.24, 82.15, 91.06, 99.97},{  5.52, 14.58, 25.07, 34.88, 44.40, 53.73, 62.95, 72.18, 81.31, 90.44, 99.57},{  3.98, 12.69, 23.10, 32.99, 42.69, 52.27, 61.66, 71.15, 80.54, 89.93, 99.31},{  2.99, 11.43, 21.76, 31.73, 41.49, 51.22, 60.88, 70.51, 80.11, 89.70, 99.30},{  2.35, 10.58, 20.83, 30.86, 40.68, 50.51, 60.33, 70.08, 79.83, 89.58, 99.32},{  2.05, 10.12, 20.27, 30.35, 40.22, 50.10, 59.97, 69.82, 79.67, 89.52, 99.38},{  2.00,  9.93, 20.00, 30.07, 40.00, 49.93, 59.87, 69.80, 79.73, 89.67, 99.60},{  2.19, 10.00, 20.00, 30.00, 40.00, 50.00, 59.99, 69.99, 79.98, 89.98, 99.97},{  2.71, 10.56, 20.61, 30.71, 40.76, 50.81, 60.86, 70.96, 81.01, 91.06,101.17},{  3.11, 11.05, 21.19, 31.41, 41.53, 51.64, 61.75, 71.95, 82.05, 92.15,102.33},{  2.39, 10.69, 21.14, 31.52, 41.73, 51.95, 62.11, 72.31, 82.46, 92.56,102.59},{  1.50, 10.11, 20.82, 31.32, 41.62, 51.92, 62.12, 72.32, 82.52, 92.63,102.56},{ -0.17,  8.50, 19.27, 29.77, 40.07, 50.37, 60.57, 70.77, 80.97, 91.13,101.23},{ -1.80,  6.96, 17.77, 28.29, 38.61, 48.91, 59.13, 69.33, 79.53, 89.71, 99.86},{ -3.42,  5.49, 16.36, 26.94, 37.31, 47.61, 57.88, 68.08, 78.28, 88.41, 98.39},{ -4.73,  4.38, 15.34, 25.99, 36.39, 46.71, 57.01, 67.21, 77.41, 87.51, 97.41},{ -5.73,  3.63, 14.74, 25.48, 35.88, 46.26, 56.56, 66.76, 76.96, 87.06, 96.96},{ -6.24,  3.33, 14.59, 25.39, 35.84, 46.22, 56.52, 66.72, 76.92, 87.04, 97.00},{ -6.09,  3.62, 15.03, 25.83, 36.37, 46.70, 57.00, 67.20, 77.40, 87.57, 97.68},{ -5.32,  4.44, 15.90, 26.70, 37.28, 47.60, 57.90, 68.10, 78.30, 88.52, 98.78},{ -3.49,  6.17, 17.52, 28.32, 38.85, 49.22, 59.52, 69.72, 79.92, 90.20,100.61},{ -0.81,  8.58, 19.73, 30.44, 40.90, 51.24, 61.52, 71.69, 81.87, 92.15,102.63},{  2.91, 11.82, 22.64, 33.17, 43.53, 53.73, 63.96, 74.09, 84.22, 94.45,104.89},{  6.68, 15.19, 25.71, 36.03, 46.25, 56.31, 66.45, 76.49, 86.54, 96.72,107.15},{ 10.43, 18.65, 28.94, 39.02, 49.01, 58.98, 68.93, 78.78, 88.69, 98.83,109.36},{ 13.56, 21.65, 31.78, 41.68, 51.45, 61.31, 71.07, 80.73, 90.48,100.51,111.01},{ 14.36, 22.91, 33.19, 43.09, 52.71, 62.37, 71.92, 81.38, 90.88,100.56,110.56},{ 15.06, 23.90, 34.23, 44.05, 53.48, 62.90, 72.21, 81.43, 90.65, 99.93,109.34},{ 15.36, 23.90, 33.89, 43.31, 52.40, 61.42, 70.29, 79.18, 88.00, 96.69,105.17},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70}};
double phons[11]={2,10,20,30,40,50,60,70,80,90,100};

//float bweights[40]= { 0, 0, 0, 0, 0, 0.15432646298622, 0.1986156888837, 0.17553530220127, 0.12767367625204, 0.011026279818415, 0.13399548316955, 0.042041540062882, 0.010204505881597, 0.036055636290171, 0, 0.095077049841321, 0.0051353555556125, 0, 0, 0, 0, 0, 0.01031301905723, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

#define NUMBARKBANDS 40

//in FFT frames
#define MAXEVENTDUR 50
#define MINEVENTDUR 6 
//7 frames is about 40 mS
#define DFFRAMESSTORED 6
#define SMOOTHFRAMESSTORED 2 
//previous 2 for peak decision

extern "C"
{
	void load(InterfaceTable *inTable);
}

InterfaceTable *ft;



struct OnsetDetection : Unit {
	
	//FFT data
	int m_bufWritePos;
	float * m_prepareFFTBuf;
	float * m_FFTBuf;
	
	//vDSP
	unsigned long m_vlog2n;
	COMPLEX_SPLIT m_vA;
	FFTSetup m_vsetup;
	
	//time positions
	long m_frame;
	long m_lastdetect;
	
	//loudness measure
	float m_loudbands[NUMBARKBANDS]; //stores previous loudness bands
	float m_df[DFFRAMESSTORED]; //detection function buffer (last 5)
	int m_dfcounter; 
	
	float m_loudness;   //store loudness, sent back with triggers for determining accents (probably not that useful since not measured at peak)
	
	float m_smooth[SMOOTHFRAMESSTORED];
	int m_smoothcounter;
	
	//recording state
	int m_onsetdetected;
	
	//triggers- greater than 0, send trigger message with that ID
	int m_triggerid;
	
};



extern "C"
{
	//required interface functions
	void OnsetDetection_next(OnsetDetection *unit, int wrongNumSamples);
	void OnsetDetection_Ctor(OnsetDetection *unit);
	void OnsetDetection_Dtor(OnsetDetection *unit);
}

//other functions
void preparefft(OnsetDetection *unit, float* in, int n);
void dofft(OnsetDetection *unit);
void calculateloudness(OnsetDetection *unit);
void peakpickdf(OnsetDetection *unit);


void OnsetDetection_Ctor(OnsetDetection* unit) {
	int j;
	
	////////FFT data///////////
	
	unit->m_prepareFFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	unit->m_FFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	unit->m_bufWritePos = 0;	
	
	////////vDSP///////////////
	
	unit->m_vA.realp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float)); 
	unit->m_vA.imagp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float));
	unit->m_vlog2n = 10; //N is hard coded as 1024, so 10^2=1024 //log2max(N);
	unit->m_vsetup = create_fftsetup(unit->m_vlog2n, 0);
	
	////////time positions//////////
	unit->m_frame=0;
	unit->m_lastdetect=-100;
	
	
	/////////loudness measure////////
	unit->m_dfcounter=DFFRAMESSTORED-1;
	//zero loudness store 
	for(j=0;j<DFFRAMESSTORED;++j) {
		unit->m_df[j]=0.0;
	}
	
		//smoothing buffer
	for(j=0;j<SMOOTHFRAMESSTORED;++j) {
		unit->m_smooth[j]=0.0;
	}
	
	unit->m_smoothcounter=SMOOTHFRAMESSTORED-1;
	
	//zero previous specific loudness in Bark bands
	for(j=0;j<NUMBARKBANDS;++j) {
		unit->m_loudbands[j]=0.0;
	}
	
	unit->m_onsetdetected=0;

	//triggers
	unit->m_triggerid=(int)ZIN0(2);

	//printf("triggerid %d \n", unit->m_triggerid);

//calc_BufRate
//	if (OUTRATE(0) == calc_FullRate) {
//		SETCALC(LFPulse_next_a);
//	} else {
//		SETCALC(LFPulse_next_k);
//	}

	unit->mCalcFunc = (UnitCalcFunc)&OnsetDetection_next;
	
}



void OnsetDetection_Dtor(OnsetDetection *unit)
{
	
	RTFree(unit->mWorld, unit->m_prepareFFTBuf);
	RTFree(unit->mWorld, unit->m_FFTBuf);
	
	if (unit->m_vA.realp) RTFree(unit->mWorld, unit->m_vA.realp);
	if (unit->m_vA.imagp) RTFree(unit->mWorld, unit->m_vA.imagp);
	if (unit->m_vsetup) destroy_fftsetup(unit->m_vsetup);
	
}


void OnsetDetection_next(OnsetDetection *unit, int wrongNumSamples)
{
	//would normally be float,will be cast to int for Tristan's optimisation
	float *in = IN(0);
	
	int numSamples = unit->mWorld->mFullRate.mBufLength;
	
	float *output = ZOUT(0);
	
	preparefft(unit, in, numSamples);
	
	//always output zero- no audio output as such	
	float outval= 0.0;
	
	if(unit->m_onsetdetected) {
	
	//printf("onset detected %d \n",(unit->m_onsetdetected));
	
	if(unit->m_triggerid) SendTrigger(&unit->mParent->mNode, unit->m_triggerid, unit->m_loudness);
	
	unit->m_onsetdetected=0;
	outval=1.0;
	}
	
	//control rate output- no, trouble
	//ZOUT0(0)=outval;
	
	for (int i=0; i<numSamples; ++i) {
		*++output = outval;
	}
	
}


//Tristan recommends copying ints rather than floats- I say negligible compared to other algorithm costs for the moment
// TO TREAT, check, update, probably replace entirely with pre allocated buffer based scheme? 
void preparefft(OnsetDetection *unit, float* in, int n) {
	
	int i, index = 0, cpt = n, maxindex;

	int bufpos= unit->m_bufWritePos;
	
	float * preparefftbuf=unit->m_prepareFFTBuf;
	float * fftbuf= unit->m_FFTBuf;
	
	// Copy input samples into prepare buffer	
	while ((bufpos < N) && (cpt > 0)) {
		preparefftbuf[bufpos] = in[index];
		bufpos++;
		index++;
		cpt--;
	}
	
	// When Buffer is full...
	if (bufpos >= N) {
		
		// Make a copy of prepared buffer into FFT buffer for computation
		for (i=0; i<N; i++) 
			fftbuf[i] = preparefftbuf[i];
		
		// Save overlapping samples back into buffer- no danger since no indices overwritten
		for (i=0; i<OVERLAP; i++) 
			preparefftbuf[i] = preparefftbuf[OVERLAPINDEX+i];
		
		unit->m_frame= unit->m_frame+1;
		dofft(unit);
		
		maxindex = n - index + OVERLAPINDEX;
		
		//blockSize less than N-OVERLAPINDEX so no problem
		// Copy the rest of incoming samples into prepareFFTBuffer
		for (i=OVERLAPINDEX; i<maxindex; i++) {
			preparefftbuf[i] = in[index];
			index++;
		}
		
		bufpos = maxindex;
		
	}
	
	
	unit->m_bufWritePos= bufpos;	
}



//calculation function once FFT data ready
void dofft(OnsetDetection *unit) {
	
	int i;
	
	//printf("dofft \n");
	
	float * fftbuf= unit->m_FFTBuf;
	
	for (i=0; i<N; ++i)
		fftbuf[i] *= hanning[i];
				
    // Look at the real signal as an interleaved complex vector by casting it.
    // Then call the transformation function ctoz to get a split complex vector,
    // which for a real signal, divides into an even-odd configuration.
    ctoz ((COMPLEX *) fftbuf, 2, &unit->m_vA, 1, NOVER2);
	
    // Carry out a Forward FFT transform
    fft_zrip(unit->m_vsetup, &unit->m_vA, 1, unit->m_vlog2n, FFT_FORWARD);
	
    // The output signal is now in a split real form.  Use the function
    // ztoc to get a split real vector.
    ztoc ( &unit->m_vA, 1, (COMPLEX *) fftbuf, 2, NOVER2);
	
	// Squared Absolute so get power
	for (i=0; i<N; i+=2)
		//i>>1 is i/2 
		fftbuf[i>>1] = (fftbuf[i] * fftbuf[i]) + (fftbuf[i+1] * fftbuf[i+1]);
	
	
	//calculate loudness detection function
	calculateloudness(unit);

	//use detection function 
	peakpickdf(unit);
		
	
}	



void calculateloudness(OnsetDetection *unit) {
	
	int j,k;
	
	float * fftbuf= unit->m_FFTBuf;
	
	float dfsum=0.0,lsum=0.0;
	
	for (k=0; k<NUMBARKBANDS; ++k){
		
		int bandstart=eqlbandbins[k];
		//int bandend=eqlbandbins[k+1];
		int bandsize= eqlbandsizes[k];
		
		float bsum=0.0;
		
		for (int h=0; h<bandsize;++h) {
			bsum= bsum+fftbuf[h+bandstart];
		}
		
		//store recips of bandsizes?
		bsum= bsum/bandsize;
		
		//into dB, avoid log of 0
		float db= 10*log10((bsum*10000000)+0.001); 
		
		//printf("bsum %f db %f \n",bsum,db);
		
		//convert via contour
		if(db<contours[k][0]) db=0; 
        else if (db>contours[k][10]) db=phons[10]; 
        else {
            
            float prop=0.0; 
			
            for (j=1; j<11; ++j) {
                if(db<contours[k][j]) {
                    prop= (db-contours[k][j-1])/(contours[k][j]-contours[k][j-1]);
                    break;
				}
				
				if(j==10) 
					prop=1.0;
            }
			
            db= (1.0-prop)*phons[j-1]+ prop*phons[j];
			//printf("prop %f db %f j %d\n",prop,db,j);
			
		}
		
		lsum=lsum+db;
		
		float lastloud=unit->m_loudbands[k];
		float diff= sc_max(db-lastloud,0.0);
		
		
		dfsum=dfsum+diff; //(bweights[k]*diff);
		
		unit->m_loudbands[k]=db;
	}
	
	//printf("loudnesscounter %d %f \n",unit->loudnesscounter, lsum);
	
	//printf("loudness %f \n",lsum*0.025);
	
	//avoid negative totals
	if(lsum<0.0) lsum=0.0;
	unit->m_loudness=lsum*0.025;
	
	//increment first so this frame is unit->m_loudnesscounterdfcounter
	unit->m_dfcounter=(unit->m_dfcounter+1)%DFFRAMESSTORED;
	
	unit->m_df[unit->m_dfcounter]=dfsum*0.025; //divide by num of bands to get a dB answer
	
	//printf("loudness %f %f \n",unit->loudness[unit->loudnesscounter], lsum);
	
	
}


float smoother[DFFRAMESSTORED]= { 0.37837837837838, 0.27027027027027, 0.16216216216216, 0.10810810810811, 0.054054054054054, 0.027027027027027 };

void peakpickdf(OnsetDetection *unit) {
	
	int i;
	
	//smooth over last 6
	
	float smoothed=0.0;
	
	int dfnow= unit->m_dfcounter+DFFRAMESSTORED;
	int pos;
	
	for (i=0; i<DFFRAMESSTORED; ++i) {
		pos= (dfnow-i)%DFFRAMESSTORED;
		smoothed+= unit->m_df[pos]*smoother[i];
	}
	
	//printf("smoothed %f\n",smoothed );
	
	smoothed= smoothed*0.2;		//up to 5dB as major changes, range normalised a bit
	
	unit->m_smoothcounter= (unit->m_smoothcounter+1)%SMOOTHFRAMESSTORED;
	
	unit->m_smooth[unit->m_smoothcounter]=smoothed;
	
	//if enough time since last detection
	if((unit->m_frame-unit->m_lastdetect)>=MINEVENTDUR) {
		
		//SIMPLE THRESHOLDING PEAKPICKER
		float threshold= ZIN0(1);
		
		int smoothnow=unit->m_smoothcounter+SMOOTHFRAMESSTORED;
		//test for SPREAD ago
		
		int left, right, testpos;	
		
		right=unit->m_smoothcounter;
		testpos= (smoothnow-1)%SMOOTHFRAMESSTORED;
		left= (smoothnow-2)%SMOOTHFRAMESSTORED;
		
		//float testsmooth=unit->m_smooth[testpos];
		//float leftav=testsmooth-unit->m_smooth[left];
		//float rightav=unit->m_smooth[right]-testsmooth;
		//float slopefunc= leftav-rightav;
		
		float testsmooth=unit->m_smooth[testpos];
		float leftdiff=testsmooth-unit->m_smooth[left];
		float rightdiff=testsmooth-unit->m_smooth[right];
		
		if((testsmooth>=threshold) && (leftdiff>=0) && (rightdiff>=0)) { 
		//if(leftav>0.0 && rightav<0.0 && slopefunc>threshold) {
			unit->m_lastdetect=unit->m_frame;
			
			unit->m_onsetdetected=1;
		}
	}
}



void prepareHanningWindow(){
	float ang;
	
	ang=(1.0/N)*TWOPI;
	
	for(int i=0;i<N;++i)
		hanning[i]=0.5 - 0.5*cos(ang*i);
	
}


void load(InterfaceTable *inTable)
{
	
	ft= inTable;
	
	prepareHanningWindow();
	
	DefineDtorUnit(OnsetDetection);
}



