/*
	SuperCollider real time audio synthesis system
 Copyright (c) 2002 James McCartney. All rights reserved.
	http://www.audiosynth.com
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
//This file is part of FiveIMSNickCollinsPhD. Copyright (C) 2006  Nicholas M.Collins distributed under the terms of the GNU General Public License full notice in file FiveIMSNickCollinsPhD.help


//Nick Collins 20 Feb 2006

//(Simple) Key tracker uses weights of FFT bins. 


#include "SC_PlugIn.h"
#include <vecLib/vecLib.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

//helpful constants
#define PI 3.1415926535898f
#define TWOPI 6.28318530717952646f 

//FFT data 
#define N 8192  //FFT size
#define NOVER2 4096  //FFT size
#define NOVER4 2048  //FFT size
#define OVERLAP 4096
//768  //512
#define OVERLAPINDEX 4096 
//256 //512
#define HOPSIZE 4096 
//256 // 512  
#define FS 44100 //assumes fixed sampling rate
#define FRAMESR  10.7666015625
//21.533203125   
//172.2656    //86.1328
#define FRAMEPERIOD 0.092879818594104
//0.00581 //0.01161 //seconds

//for hanning window, constructed in plugin initialisation
float hanning[N]; 

//weighting parameters

//for 4096 FFT
//double g_weights[120]= { 0.85013247759125, 0.14986752240875, 0.70026495524988, 0.29973504475012, 0.5994700893655, 0.4005299106345, 0.8010598215385, 0.1989401784615, 0.60211964361596, 0.39788035638404, 0.8723362413437, 0.1276637586563, 0.74467248261602, 0.25532751738398, 0.51065503491071, 0.48934496508929, 0.97868992989306, 0.021310070106935, 0.95737985921514, 0.042620140784862, 0.63776518588415, 0.36223481411585, 0.72446962830733, 0.27553037169267, 0.55106074323409, 0.44893925676591, 0.89787851383433, 0.10212148616567, 0.79575702827361, 0.20424297172639, 0.55129109602105, 0.44870890397895, 0.89741780787778, 0.10258219212222, 0.79483561559532, 0.20516438440468, 0.58967123087014, 0.41032876912986, 0.82065753890066, 0.17934246109934, 0.6921261550973, 0.3078738449027, 0.61574768972052, 0.38425231027948, 0.76850462072872, 0.23149537927128, 0.53700924179701, 0.46299075820299, 0.92598151572696, 0.07401848427304, 0.78187260822141, 0.21812739177859, 0.56374521653274, 0.43625478346726, 0.87250956675464, 0.12749043324536, 0.74501913314953, 0.25498086685047, 0.50996173442036, 0.49003826557964, 0.81749256879698, 0.18250743120302, 0.63498513768923, 0.36501486231077, 0.73002972443098, 0.26997027556902, 0.53994055151915, 0.46005944848085, 0.92011889619943, 0.079881103800574, 0.79576750809026, 0.20423249190974, 0.59153501628148, 0.40846498371852, 0.81692996723517, 0.18307003276483, 0.63385993406655, 0.36614006593345, 0.73228013267448, 0.26771986732552, 0.71328751368974, 0.28671248631026, 0.57342497251356, 0.42657502748644, 0.85315005518676, 0.14684994481324, 0.70630011080129, 0.29369988919871, 0.58739977754175, 0.41260022245825, 0.5664399092404, 0.4335600907596, 0.86712018140589, 0.13287981859411, 0.73424036258518, 0.26575963741482, 0.53151927528288, 0.46848072471712, 0.93696144852777, 0.063038551472232, 0.64860280252777, 0.35139719747223, 0.70279439506452, 0.29720560493548, 0.59441120963089, 0.40558879036911, 0.81117758121837, 0.18882241878163, 0.62235516339723, 0.37764483660277, 0.93589571371608, 0.064104286283918, 0.87179142730498, 0.12820857269502, 0.74358285435559, 0.25641714564441, 0.51283429179753, 0.48716570820247, 0.97433141538738, 0.025668584612617};
//int g_bins[120]= { 12, 13, 24, 25, 49, 48, 97, 98, 194, 195, 13, 12, 26, 25, 51, 52, 103, 102, 206, 205, 14, 13, 27, 28, 55, 54, 109, 110, 218, 219, 14, 15, 29, 28, 58, 57, 116, 115, 231, 232, 15, 16, 31, 30, 61, 62, 122, 123, 245, 244, 16, 17, 32, 33, 65, 64, 130, 129, 259, 260, 17, 18, 34, 35, 69, 68, 137, 138, 275, 274, 18, 19, 36, 37, 73, 72, 146, 145, 291, 292, 19, 20, 39, 38, 77, 78, 154, 155, 309, 308, 20, 21, 41, 40, 82, 81, 163, 164, 327, 326, 22, 21, 43, 44, 87, 86, 173, 174, 346, 347, 23, 22, 46, 45, 92, 91, 183, 184, 367, 366 };

//for 8192 FFT
//octaves 4 to 8
double g_weights[120]= { 0.7002649551825, 0.2997350448175, 0.59947008950024, 0.40052991049976, 0.80105982126901, 0.19894017873099, 0.602119643077, 0.397880356923, 0.79576071276807, 0.20423928723193, 0.74467248268741, 0.25532751731259, 0.51065503476795, 0.48934496523205, 0.97868993017859, 0.021310069821411, 0.95737985978613, 0.04262014021387, 0.91475971843028, 0.085240281569725, 0.7244696282317, 0.2755303717683, 0.55106074338534, 0.44893925661466, 0.89787851353182, 0.10212148646818, 0.79575702766866, 0.20424297233134, 0.59151405654723, 0.40848594345277, 0.8974178079579, 0.1025821920421, 0.79483561575557, 0.20516438424443, 0.58967123119064, 0.41032876880936, 0.82065753825972, 0.17934246174028, 0.64131507780132, 0.35868492219868, 0.61574768980541, 0.38425231019459, 0.76850462055896, 0.23149537944104, 0.53700924145744, 0.46299075854256, 0.92598151640598, 0.074018483594017, 0.85196303145392, 0.14803696854608, 0.56374521644281, 0.43625478355719, 0.87250956693451, 0.12749043306549, 0.74501913350929, 0.25498086649071, 0.50996173370095, 0.49003826629905, 0.98007653115928, 0.019923468840716, 0.63498513759395, 0.36501486240605, 0.73002972462153, 0.26997027537847, 0.53994055113805, 0.46005944886195, 0.9201188969617, 0.079881103038304, 0.84023779239885, 0.15976220760115, 0.59153501618053, 0.40846498381947, 0.81692996743705, 0.18307003256295, 0.63385993447034, 0.36614006552966, 0.7322801318669, 0.2677198681331, 0.53543973465105, 0.46456026534895, 0.57342497262051, 0.42657502737949, 0.85315005497287, 0.14684994502713, 0.70630011037352, 0.29369988962648, 0.58739977839741, 0.41260022160259, 0.8252004449165, 0.1747995550835, 0.8671201815192, 0.1328798184808, 0.73424036281179, 0.26575963718821, 0.53151927482963, 0.46848072517037, 0.93696144943425, 0.06303855056575, 0.87392289705554, 0.12607710294446, 0.70279439494447, 0.29720560505553, 0.59441120987097, 0.40558879012903, 0.81117758073822, 0.18882241926178, 0.62235516243675, 0.37764483756325, 0.75528967320554, 0.24471032679446, 0.87179142743216, 0.12820857256784, 0.74358285460995, 0.25641714539005, 0.51283429128881, 0.48716570871119, 0.97433141640494, 0.025668583595063, 0.94866283077477, 0.051337169225235 };
int g_bins[120]= { 24, 25, 49, 48, 97, 98, 194, 195, 389, 388, 26, 25, 51, 52, 103, 102, 206, 205, 412, 411, 27, 28, 55, 54, 109, 110, 218, 219, 436, 437, 29, 28, 58, 57, 116, 115, 231, 232, 462, 463, 31, 30, 61, 62, 122, 123, 245, 244, 490, 489, 32, 33, 65, 64, 130, 129, 259, 260, 519, 518, 34, 35, 69, 68, 137, 138, 275, 274, 550, 549, 36, 37, 73, 72, 146, 145, 291, 292, 583, 582, 39, 38, 77, 78, 154, 155, 309, 308, 617, 618, 41, 40, 82, 81, 163, 164, 327, 326, 654, 653, 43, 44, 87, 86, 173, 174, 346, 347, 693, 692, 46, 45, 92, 91, 183, 184, 367, 366, 734, 733 };

//octaves 4 to 9
//double g_weights[144]= { 0.7002649551825, 0.2997350448175, 0.59947008950024, 0.40052991049976, 0.80105982126901, 0.19894017873099, 0.602119643077, 0.397880356923, 0.79576071276807, 0.20423928723193, 0.59152142337996, 0.40847857662004, 0.74467248268741, 0.25532751731259, 0.51065503476795, 0.48934496523205, 0.97868993017859, 0.021310069821411, 0.95737985978613, 0.04262014021387, 0.91475971843028, 0.085240281569725, 0.82951943457601, 0.17048056542399, 0.7244696282317, 0.2755303717683, 0.55106074338534, 0.44893925661466, 0.89787851353182, 0.10212148646818, 0.79575702766866, 0.20424297233134, 0.59151405654723, 0.40848594345277, 0.81697188448572, 0.18302811551428, 0.8974178079579, 0.1025821920421, 0.79483561575557, 0.20516438424443, 0.58967123119064, 0.41032876880936, 0.82065753825972, 0.17934246174028, 0.64131507780132, 0.35868492219868, 0.7173698418336, 0.2826301581664, 0.61574768980541, 0.38425231019459, 0.76850462055896, 0.23149537944104, 0.53700924145744, 0.46299075854256, 0.92598151640598, 0.074018483594017, 0.85196303145392, 0.14803696854608, 0.70392606019141, 0.29607393980859, 0.56374521644281, 0.43625478355719, 0.87250956693451, 0.12749043306549, 0.74501913350929, 0.25498086649071, 0.50996173370095, 0.49003826629905, 0.98007653115928, 0.019923468840716, 0.96015305944093, 0.039846940559073, 0.63498513759395, 0.36501486240605, 0.73002972462153, 0.26997027537847, 0.53994055113805, 0.46005944886195, 0.9201188969617, 0.079881103038304, 0.84023779239885, 0.15976220760115, 0.68047558174885, 0.31952441825115, 0.59153501618053, 0.40846498381947, 0.81692996743705, 0.18307003256295, 0.63385993447034, 0.36614006552966, 0.7322801318669, 0.2677198681331, 0.53543973465105, 0.46456026534895, 0.92912053392797, 0.070879466072029, 0.57342497262051, 0.42657502737949, 0.85315005497287, 0.14684994502713, 0.70630011037352, 0.29369988962648, 0.58739977839741, 0.41260022160259, 0.8252004449165, 0.1747995550835, 0.65040089325544, 0.34959910674456, 0.8671201815192, 0.1328798184808, 0.73424036281179, 0.26575963718821, 0.53151927482963, 0.46848072517037, 0.93696144943425, 0.06303855056575, 0.87392289705554, 0.12607710294446, 0.74784579048514, 0.25215420951486, 0.70279439494447, 0.29720560505553, 0.59441120987097, 0.40558879012903, 0.81117758073822, 0.18882241926178, 0.62235516243675, 0.37764483756325, 0.75528967320554, 0.24471032679446, 0.51057934257028, 0.48942065742972, 0.87179142743216, 0.12820857256784, 0.74358285460995, 0.25641714539005, 0.51283429128881, 0.48716570871119, 0.97433141640494, 0.025668583595063, 0.94866283077477, 0.051337169225235, 0.89732565748, 0.10267434252 };
//int g_bins[144]= { 24, 25, 49, 48, 97, 98, 194, 195, 389, 388, 778, 777, 26, 25, 51, 52, 103, 102, 206, 205, 412, 411, 824, 823, 27, 28, 55, 54, 109, 110, 218, 219, 436, 437, 873, 872, 29, 28, 58, 57, 116, 115, 231, 232, 462, 463, 925, 924, 31, 30, 61, 62, 122, 123, 245, 244, 490, 489, 980, 979, 32, 33, 65, 64, 130, 129, 259, 260, 519, 518, 1038, 1037, 34, 35, 69, 68, 137, 138, 275, 274, 550, 549, 1100, 1099, 36, 37, 73, 72, 146, 145, 291, 292, 583, 582, 1165, 1166, 39, 38, 77, 78, 154, 155, 309, 308, 617, 618, 1234, 1235, 41, 40, 82, 81, 163, 164, 327, 326, 654, 653, 1308, 1307, 43, 44, 87, 86, 173, 174, 346, 347, 693, 692, 1386, 1385, 46, 45, 92, 91, 183, 184, 367, 366, 734, 733, 1468, 1467 };




//Krumhansl Kessler profiles (normalised)
double g_kkminor[12] = { 0.14221523253202, 0.060211188496967, 0.079083352055718, 0.12087171422152, 0.05841383958661, 0.079308020669512, 0.057065827903842, 0.10671759155246, 0.089418108290272, 0.060435857110762, 0.075039317007414, 0.071219950572905 };
double g_kkmajor[12] = { 0.15195022732711, 0.053362048336923, 0.083273510409189, 0.055754965302704, 0.10480976310122, 0.097870303900455, 0.060301507537688, 0.12419239052405, 0.057190715482173, 0.087580760947595, 0.054797798516391, 0.068916008614501 };

int g_minor[7] = {0,2,3,5,7,8,11};
int g_major[7] = {0,2,4,5,7,9,11};


extern "C"
{
	void load(InterfaceTable *inTable);
}

InterfaceTable *ft;



struct Kitch : Unit {
	
	//FFT data
	int m_bufWritePos;
	float * m_prepareFFTBuf;
	float * m_FFTBuf;
	
	//vDSP
	unsigned long m_vlog2n;
	COMPLEX_SPLIT m_vA;
	FFTSetup m_vsetup;
	
	uint32 m_frame;
	
	float m_chroma[12]; 
	float m_key[24]; 
	
	float m_histogram[24];   //key histogram
							 //float m_keyleak; //fade parameter for histogram
	
	int m_triggerid;	
	
	int m_currentKey;	
};



extern "C"
{
	//required interface functions
	void Kitch_next(Kitch *unit, int wrongNumSamples);
	void Kitch_Ctor(Kitch *unit);
	void Kitch_Dtor(Kitch *unit);
}

//other functions
void preparefft(Kitch *unit, float* in, int n);
void dofft(Kitch *unit);

//void calculatekey(Kitch *unit);

void Kitch_Ctor(Kitch* unit) {
	int i,j;
	
	World *world = unit->mWorld;
    
	////////EVAL
	
	//	uint32 bufnum2 = (uint32)ZIN0(6);
	//	if (bufnum2 >= world->mNumSndBufs) bufnum2 = 0;
	//	unit->m_bufNum2=bufnum2;
	//	
	//	SndBuf *buf2 = world->mSndBufs + bufnum2; 
	//	unit->m_bufSize2 = buf2->samples;
	//	
	//	unit->m_maxstored= (unit->m_bufSize2-1); //1 float per event, plus counter
	//	unit->m_eval= buf2->data; 	
	//	unit->m_debugstored=1;
	//	
	//	printf("deugging %d\n\n",bufnum2);
	//	
	
	/////////
	
	////////FFT data///////////
	
	unit->m_prepareFFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	unit->m_FFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	
	//	unit->m_fftnow= (float*)RTAlloc(unit->mWorld, DFCALCBINS * sizeof(float));
	//	unit->m_fftthen= (float*)RTAlloc(unit->mWorld, DFCALCBINS * sizeof(float));
	//	unit->m_fftthen2= (float*)RTAlloc(unit->mWorld, DFCALCBINS * sizeof(float));
	//	
	unit->m_bufWritePos = 0;	
	
	////////vDSP///////////////
	
	unit->m_vA.realp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float)); 
	unit->m_vA.imagp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float));
	unit->m_vlog2n = 13; //8192  N is hard coded as 1024, so 10^2=1024 //log2max(N);
	unit->m_vsetup = create_fftsetup(unit->m_vlog2n, 0);

	//zero chroma 
	for(j=0;j<12;++j) {
		unit->m_chroma[j]=0.0;
	}
	
	for(j=0;j<24;++j) {
		unit->m_key[j]=0.0;
		unit->m_histogram[j]=0.0;
	}
	
	//triggers
	unit->m_triggerid=(int)ZIN0(1);
	
	unit->m_currentKey=0;
	
	unit->m_frame=0;
	
	unit->mCalcFunc = (UnitCalcFunc)&Kitch_next;
}



void Kitch_Dtor(Kitch *unit)
{
	
	RTFree(unit->mWorld, unit->m_prepareFFTBuf);
	RTFree(unit->mWorld, unit->m_FFTBuf);
	
	if (unit->m_vA.realp) RTFree(unit->mWorld, unit->m_vA.realp);
	if (unit->m_vA.imagp) RTFree(unit->mWorld, unit->m_vA.imagp);
	if (unit->m_vsetup) destroy_fftsetup(unit->m_vsetup);
	
}


void Kitch_next(Kitch *unit, int wrongNumSamples)
{
	//would normally be float,will be cast to int for Tristan's optimisation
	float *in = IN(0);
	
	int numSamples = unit->mWorld->mFullRate.mBufLength;
	
	float *output = ZOUT(0);
	preparefft(unit, in, numSamples);
	
	//always output zero- no audio output as such	
	float outval= unit->m_currentKey;
	
	//control rate output- no, trouble
	//ZOUT0(0)=outval;
	
	for (int i=0; i<numSamples; ++i) {
		*++output = outval;
	}
	
}


//Tristan recommends copying ints rather than floats- I say negligible compared to other algorithm costs for the moment
// TO TREAT, check, update, probably replace entirely with pre allocated buffer based scheme? 
void preparefft(Kitch *unit, float* in, int n) {
	
	int i, index = 0, cpt = n, maxindex;
	
	int bufpos= unit->m_bufWritePos;
	
	float * preparefftbuf=unit->m_prepareFFTBuf;
	float * fftbuf= unit->m_FFTBuf;
	
	// Copy input samples into prepare buffer	
	while ((bufpos < N) && (cpt > 0)) {
		preparefftbuf[bufpos] = in[index];
		bufpos++;
		index++;
		cpt--;
	}
	
	// When Buffer is full...
	if (bufpos >= N) {
		
		// Make a copy of prepared buffer into FFT buffer for computation
		for (i=0; i<N; i++) 
			fftbuf[i] = preparefftbuf[i];
		
		// Save overlapping samples back into buffer- no danger since no indices overwritten
		for (i=0; i<OVERLAP; i++) 
			preparefftbuf[i] = preparefftbuf[OVERLAPINDEX+i];
		
		unit->m_frame= unit->m_frame+1;
		dofft(unit);
		
		maxindex = n - index + OVERLAPINDEX;
		
		//blockSize less than N-OVERLAPINDEX so no problem
		// Copy the rest of incoming samples into prepareFFTBuffer
		for (i=OVERLAPINDEX; i<maxindex; i++) {
			preparefftbuf[i] = in[index];
			index++;
		}
		
		bufpos = maxindex;
		
	}
	
	
	unit->m_bufWritePos= bufpos;	
}



//calculation function once FFT data ready
void dofft(Kitch *unit) {
	
	int i,j;

	float * fftbuf= unit->m_FFTBuf;
	
	for (i=0; i<N; ++i)
		fftbuf[i] *= hanning[i];
				
    // Look at the real signal as an interleaved complex vector by casting it.
    // Then call the transformation function ctoz to get a split complex vector,
    // which for a real signal, divides into an even-odd configuration.
    ctoz ((COMPLEX *) fftbuf, 2, &unit->m_vA, 1, NOVER2);
	
    // Carry out a Forward FFT transform
    fft_zrip(unit->m_vsetup, &unit->m_vA, 1, unit->m_vlog2n, FFT_FORWARD);
	
    // The output signal is now in a split real form.  Use the function
    // ztoc to get a real vector, in format [dc,nyq, bin1realm bin1imag, bin2real, bin2imag, ....]
    ztoc ( &unit->m_vA, 1, (COMPLEX *) fftbuf, 2, NOVER2);
	
	
	//int peakbin= 0;
	//float peakpower=0.0;
	
	// Squared Absolute so get power
	for (i=0; i<N; i+=2) {
		//i>>1 is i/2 
		fftbuf[i>>1] = ((fftbuf[i] * fftbuf[i]) + (fftbuf[i+1] * fftbuf[i+1]))*0.25;
		
		//if((fftbuf[i>>1])>peakpower) {peakpower=fftbuf[i>>1]; peakbin=i>>1;}
	}
	
	
	//	printf("peak %d %f \n",peakbin,peakpower);
		
	//	for (i=peakbin-3; i<(peakbin+3); ++i) 
	//	printf("%f ",fftbuf[i]);
	
	float * chroma= unit->m_chroma; 
	
	float sum;
	int indexbase, index;
	
	for (i=0;i<12;++i) {
		indexbase=i*10; //i*12;
		
		sum=0.0;
		
		for(j=0;j<10;++j) { //12 if 144 data points
			
			index=indexbase+j;
			
			sum+= (g_weights[index])*(fftbuf[g_bins[index]]);
		}
		
		chroma[i]=10*log10(sum+1); //to approximate perceptual volume
		
		//printf("%f ",chroma[i]);
	}
	
	
		//printf("\n");
	
	float* key = unit->m_key; 
	
	//major
	for (i=0;i<12;++i) {
		
		sum=0.0;
		for (j=0;j<7;++j) {
			indexbase=g_major[j];
			
			index=(i+indexbase)%12;
			sum+=(chroma[index]*g_kkmajor[indexbase]);
		}
		
		key[i]=sum; //10*log10(sum+1);
	}
	
	//minor
	for (i=0;i<12;++i) {
		
		sum=0.0;
		for (j=0;j<7;++j) {
			indexbase=g_minor[j];
			
			index=(i+indexbase)%12;
			sum+=(chroma[index]*g_kkminor[indexbase]);
		}
		
		key[12+i]=0.0; //sum; //remove minor key options for the moment
	}
	
	float keyleak= ZIN0(2); //fade parameter to 0.01 for histogram in seconds, convert to FFT frames 
	
	//keyleak in seconds, convert to drop time in FFT hop frames (FRAMEPERIOD)
	
	keyleak= keyleak/FRAMEPERIOD;
	
	//now number of frames, actual leak param is decay exponent to reach 0.01 in x seconds, ie 0.01 = leakparam ** (x/ffthopsize)
	keyleak= pow(0.01,(1.0/keyleak));
	
	float * histogram= unit->m_histogram; 
	
	int bestkey=0;
	float bestscore=0.0;
	
	for (i=0;i<24;++i) {
		histogram[i]= (keyleak*histogram[i])+key[i]; 
		
		if(histogram[i]>bestscore) {
			bestscore=histogram[i];
			bestkey=i;
		}
		
	//printf("%f ",histogram[i]);
		
	}
	
	
	//printf(" best %d \n\n",bestkey);
	
	//what is winning currently? find max in histogram
	
	unit->m_currentKey=bestkey;
	
	//about 5 times per second
	if((unit->m_triggerid) && ((unit->m_frame%2==0))) SendTrigger(&unit->mParent->mNode, unit->m_triggerid, bestkey);
	
}	


//
////need calculateloudness as simultaneous process! (or use running max amp)?
//void calculatekey(Kitch *unit) {
//		
//}




void prepareHanningWindow(){
	float ang;
	
	ang=(1.0/N)*TWOPI;
	
	for(int i=0;i<N;++i)
		hanning[i]=0.5 - 0.5*cos(ang*i);
	
}



void load(InterfaceTable *inTable)
{
	
	ft= inTable;
	
	prepareHanningWindow();
	
	printf("Kitch 19-2-06 \n");
	
	DefineDtorUnit(Kitch);
}



