/*
	SuperCollider real time audio synthesis system
 Copyright (c) 2002 James McCartney. All rights reserved.
	http://www.audiosynth.com
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
//This file is part of FiveIMSNickCollinsPhD. Copyright (C) 2006  Nicholas M.Collins distributed under the terms of the GNU General Public License full notice in file FiveIMSNickCollinsPhD.help


//Nick Collins 14 feb 2006
//trying to implement the best onset detection algo from AES118 paper, with event analysis data to be written to a buffer
//for potential NRT and RT use

//stores up to a second of audio, assumes that events are not longer than that minus a few FFT frames 
//assumes 44100 SR and FFT of 1024, 512 overlap

//inputs for pitch tracker ZIN0(5)

//after collecting events, will analyse some timbral qualities (spec centroid, rolloff, temp centroid, zcr) (drum classifier), PAT (energy integration algorithm),
//loudness of event, envelope attacktime, release time etc,  pitch envelope (median pitch)
//heuristics on event selection (reject right skewed, multihit)    
//need to combine with an SC3 GUI that shows the events being collected?



#include "SC_PlugIn.h"
#include <vecLib/vecLib.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

//helpful constants
#define PI 3.1415926535898f
#define TWOPI 6.28318530717952646f 

//FFT data 
#define N 1024  //FFT size
#define NOVER2 512  //FFT size
#define NOVER4 256  //FFT size
#define OVERLAP 512
//768  //512
#define OVERLAPINDEX 512 
//256 //512
#define HOPSIZE 512 
//256 // 512  
#define FS 44100 //assumes fixed sampling rate
#define FRAMESR  86.1328   
//172.2656    //86.1328
#define FRAMEPERIOD 0.01161
//0.00581 //0.01161 //seconds

//for hanning window, constructed in plugin initialisation
float hanning[N]; 

int eqlbandbins[43]= {1,2,3,4,5,6,7,8,9,11,13,15,17,19,22,25,28,32,36,41,46,52,58,65,73,82,92,103,116,129,144,161,180,201,225,251,280,312,348,388,433,483,513}; 
int eqlbandsizes[42]= {1,1,1,1,1,1,1,1,2,2,2,2,2,3,3,3,4,4,5,5,6,6,7,8,9,10,11,13,13,15,17,19,21,24,26,29,32,36,40,45,50,30}; 
float contours[42][11]= {{ 47.88, 59.68, 68.55, 75.48, 81.71, 87.54, 93.24, 98.84,104.44,109.94,115.31},{ 29.04, 41.78, 51.98, 60.18, 67.51, 74.54, 81.34, 87.97, 94.61,101.21,107.74},{ 20.72, 32.83, 43.44, 52.18, 60.24, 67.89, 75.34, 82.70, 89.97, 97.23,104.49},{ 15.87, 27.14, 37.84, 46.94, 55.44, 63.57, 71.51, 79.34, 87.14, 94.97,102.37},{ 12.64, 23.24, 33.91, 43.27, 52.07, 60.57, 68.87, 77.10, 85.24, 93.44,100.90},{ 10.31, 20.43, 31.03, 40.54, 49.59, 58.33, 66.89, 75.43, 83.89, 92.34,100.80},{  8.51, 18.23, 28.83, 38.41, 47.65, 56.59, 65.42, 74.16, 82.89, 91.61,100.33},{  7.14, 16.55, 27.11, 36.79, 46.16, 55.27, 64.29, 73.24, 82.15, 91.06, 99.97},{  5.52, 14.58, 25.07, 34.88, 44.40, 53.73, 62.95, 72.18, 81.31, 90.44, 99.57},{  3.98, 12.69, 23.10, 32.99, 42.69, 52.27, 61.66, 71.15, 80.54, 89.93, 99.31},{  2.99, 11.43, 21.76, 31.73, 41.49, 51.22, 60.88, 70.51, 80.11, 89.70, 99.30},{  2.35, 10.58, 20.83, 30.86, 40.68, 50.51, 60.33, 70.08, 79.83, 89.58, 99.32},{  2.05, 10.12, 20.27, 30.35, 40.22, 50.10, 59.97, 69.82, 79.67, 89.52, 99.38},{  2.00,  9.93, 20.00, 30.07, 40.00, 49.93, 59.87, 69.80, 79.73, 89.67, 99.60},{  2.19, 10.00, 20.00, 30.00, 40.00, 50.00, 59.99, 69.99, 79.98, 89.98, 99.97},{  2.71, 10.56, 20.61, 30.71, 40.76, 50.81, 60.86, 70.96, 81.01, 91.06,101.17},{  3.11, 11.05, 21.19, 31.41, 41.53, 51.64, 61.75, 71.95, 82.05, 92.15,102.33},{  2.39, 10.69, 21.14, 31.52, 41.73, 51.95, 62.11, 72.31, 82.46, 92.56,102.59},{  1.50, 10.11, 20.82, 31.32, 41.62, 51.92, 62.12, 72.32, 82.52, 92.63,102.56},{ -0.17,  8.50, 19.27, 29.77, 40.07, 50.37, 60.57, 70.77, 80.97, 91.13,101.23},{ -1.80,  6.96, 17.77, 28.29, 38.61, 48.91, 59.13, 69.33, 79.53, 89.71, 99.86},{ -3.42,  5.49, 16.36, 26.94, 37.31, 47.61, 57.88, 68.08, 78.28, 88.41, 98.39},{ -4.73,  4.38, 15.34, 25.99, 36.39, 46.71, 57.01, 67.21, 77.41, 87.51, 97.41},{ -5.73,  3.63, 14.74, 25.48, 35.88, 46.26, 56.56, 66.76, 76.96, 87.06, 96.96},{ -6.24,  3.33, 14.59, 25.39, 35.84, 46.22, 56.52, 66.72, 76.92, 87.04, 97.00},{ -6.09,  3.62, 15.03, 25.83, 36.37, 46.70, 57.00, 67.20, 77.40, 87.57, 97.68},{ -5.32,  4.44, 15.90, 26.70, 37.28, 47.60, 57.90, 68.10, 78.30, 88.52, 98.78},{ -3.49,  6.17, 17.52, 28.32, 38.85, 49.22, 59.52, 69.72, 79.92, 90.20,100.61},{ -0.81,  8.58, 19.73, 30.44, 40.90, 51.24, 61.52, 71.69, 81.87, 92.15,102.63},{  2.91, 11.82, 22.64, 33.17, 43.53, 53.73, 63.96, 74.09, 84.22, 94.45,104.89},{  6.68, 15.19, 25.71, 36.03, 46.25, 56.31, 66.45, 76.49, 86.54, 96.72,107.15},{ 10.43, 18.65, 28.94, 39.02, 49.01, 58.98, 68.93, 78.78, 88.69, 98.83,109.36},{ 13.56, 21.65, 31.78, 41.68, 51.45, 61.31, 71.07, 80.73, 90.48,100.51,111.01},{ 14.36, 22.91, 33.19, 43.09, 52.71, 62.37, 71.92, 81.38, 90.88,100.56,110.56},{ 15.06, 23.90, 34.23, 44.05, 53.48, 62.90, 72.21, 81.43, 90.65, 99.93,109.34},{ 15.36, 23.90, 33.89, 43.31, 52.40, 61.42, 70.29, 79.18, 88.00, 96.69,105.17},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70},{ 15.60, 23.90, 33.60, 42.70, 51.50, 60.20, 68.70, 77.30, 85.80, 94.00,101.70}};
double phons[11]={2,10,20,30,40,50,60,70,80,90,100};

//float bweights[40]= { 0, 0, 0, 0, 0, 0.15432646298622, 0.1986156888837, 0.17553530220127, 0.12767367625204, 0.011026279818415, 0.13399548316955, 0.042041540062882, 0.010204505881597, 0.036055636290171, 0, 0.095077049841321, 0.0051353555556125, 0, 0, 0, 0, 0, 0.01031301905723, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

#define NUMERBBANDS 40
#define PASTERBBANDS 3
//3 usually, but time resolution improved if made 1?

//in FFT frames
#define MAXEVENTDUR 80
#define MINEVENTDUR 3
//4 or maybe 2

#define CIRCBUFSIZE 15

//7 frames is about 40 mS
//peak picker will use 3 back, 3 forward
#define DFFRAMESSTORED 7

#define MAXBLOCKSIZE 64
#define MAXBLOCKS 700

//MAXEVENTDUR+20 for safety
#define LOUDNESSSTORED 100


#define DFCALCBINS 358

extern "C"
{
	void load(InterfaceTable *inTable);
}

InterfaceTable *ft;



struct AnalyseHarpsichord : Unit {
	
	//FFT data
	int m_bufWritePos;
	float * m_prepareFFTBuf;
	float * m_FFTBuf;
	float * m_fftnow;
	float * m_fftthen;
	float * m_fftthen2;
	
	 // m_fftbindata[1152]; //3*DFCALCBINS, previous three fft frames of log(x^2+1) values
	
	//vDSP
	unsigned long m_vlog2n;
	COMPLEX_SPLIT m_vA;
	FFTSetup m_vsetup;
	
	//time positions
	long m_frame;
	long m_lastdetect;
	
	//loudness measure
	//float m_loudbands[NUMERBBANDS][PASTERBBANDS]; //stores previous loudness bands
	//int m_pasterbbandcounter;
	float m_df[DFFRAMESSTORED]; //detection function buffer (last 5)
	int m_dfcounter; 
	
	float m_loudness[LOUDNESSSTORED];   //store loudness, sent back with triggers for determining accents (probably not that useful since not measured at peak)
	int m_loudnesscounter; 
	
	//recording state
	int m_onsetdetected;
	int m_recording;
	int m_start;
	float m_testintensity; //end can trigger if drop below this
	
	//into global frame buffer
	int m_numframes;
	int m_startframe;
	int m_endframe;
	
	int m_startblock;
	int m_endblock;
	
	//into loudness frame buffer
	int m_lstartframe;
	int m_lendframe;
	
	//into pitch frame buffer
	
	//triggers- greater than 0, send trigger message with that ID
	int m_triggerid;
	
	//target buffer for event data
	uint32 m_bufNum, m_bufSize;
	int m_maxEvents; //, m_lastEvent; //maximum that can be stored
	float * m_eventData; //[0] position stores num of events stored
	int m_numstored;
	
	//different structure to buffer saving for m_circular = true
	//support for circular buffers and on-the-fly event capture
	int m_circular; //act as if a circular buffer, uses m_numstored for which to write/pick up next
					//SendTrigger on a detected event giving place to pick up
	
	//pause just by pausing detection Synth (RecordBuf paused at the same time)
	//int m_pause;	//pause recording, do not store any detected events, zero counts as go round, but Lang side 
	
	
	uint32 m_now;
	
	//int m_maxblocksstored; //since FS fixed at 44100, fix this too
	float * m_maxintensity;
	int m_maxcount;
	float * m_store;
	int m_storecounter;
	
	//uses maxcount
	float * m_pitch;
	float * m_sortbuf;
	
	
	
	//evaluation- write peakpick/df values to a buffer
//	uint32 m_bufNum2, m_bufSize2;
//	int m_maxstored,m_debugstored;
//	float * m_eval; 
	
};



extern "C"
{
	//required interface functions
	void AnalyseHarpsichord_next(AnalyseHarpsichord *unit, int wrongNumSamples);
	void AnalyseHarpsichord_Ctor(AnalyseHarpsichord *unit);
	void AnalyseHarpsichord_Dtor(AnalyseHarpsichord *unit);
}

//other functions
void preparefft(AnalyseHarpsichord *unit, float* in, int n);
void dofft(AnalyseHarpsichord *unit);
void calculatedf(AnalyseHarpsichord *unit);
void calculateloudness(AnalyseHarpsichord *unit);

void peakpickdf(AnalyseHarpsichord *unit);

void storeEvent(AnalyseHarpsichord *unit, int start, int stop);
int findZeroCrossing(AnalyseHarpsichord *unit, int base);
int findMinimumMaxIntensity(AnalyseHarpsichord *unit, int framenow);

//feature extraction
float calculatePAT(AnalyseHarpsichord *unit);
float calculatePitch(AnalyseHarpsichord *unit);



void AnalyseHarpsichord_Ctor(AnalyseHarpsichord* unit) {
	int i,j;
	
	World *world = unit->mWorld;
    
	////////EVAL
//
//	uint32 bufnum2 = (uint32)ZIN0(6);
//	if (bufnum2 >= world->mNumSndBufs) bufnum2 = 0;
//	unit->m_bufNum2=bufnum2;
//	
//	SndBuf *buf2 = world->mSndBufs + bufnum2; 
//	unit->m_bufSize2 = buf2->samples;
//	
//	printf("bufnum %d bufsize %d \n",unit->m_bufNum2, unit->m_bufSize2);
//	
//	unit->m_maxstored= (unit->m_bufSize2-1); //1 float per event, plus counter
//	unit->m_eval= buf2->data; 	
//	unit->m_debugstored=1;
	/////////

	
	
	////////FFT data///////////
	
	unit->m_prepareFFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	unit->m_FFTBuf = (float*)RTAlloc(unit->mWorld, N * sizeof(float));
	
	unit->m_fftnow= (float*)RTAlloc(unit->mWorld, DFCALCBINS * sizeof(float));
	unit->m_fftthen= (float*)RTAlloc(unit->mWorld, DFCALCBINS * sizeof(float));
	unit->m_fftthen2= (float*)RTAlloc(unit->mWorld, DFCALCBINS * sizeof(float));
	
	unit->m_bufWritePos = 0;	
	
	////////vDSP///////////////
	
	unit->m_vA.realp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float)); 
	unit->m_vA.imagp = (float*)RTAlloc(unit->mWorld, NOVER2 * sizeof(float));
	unit->m_vlog2n = 10; //N is hard coded as 1024, so 10^2=1024 //log2max(N);
	unit->m_vsetup = create_fftsetup(unit->m_vlog2n, 0);
	
	////////time positions//////////
	unit->m_frame=0;
	unit->m_lastdetect=-100;
	
	
	/////////df measure////////
	unit->m_dfcounter=DFFRAMESSTORED-1;
	//zero loudness store 
	for(j=0;j<DFFRAMESSTORED;++j) {
		unit->m_df[j]=0.0;
	}
	
	//LOUDNESS
	unit->m_loudnesscounter=LOUDNESSSTORED-1;
	//zero loudness store 
	for(j=0;j<LOUDNESSSTORED;++j) {
		unit->m_loudness[j]=0.0;
	}
	
	
	//zero previous specific loudness in Bark bands
//	for(i=0;i<PASTERBBANDS; ++i)
//		for(j=0;j<NUMERBBANDS;++j) {
//			unit->m_loudbands[j][i]=0.0;
//		}
//			
//			unit->m_pasterbbandcounter=0;
//	
	unit->m_onsetdetected=0;
	unit->m_recording=0;
	
	//triggers
	unit->m_triggerid=(int)ZIN0(3);
	
	//printf("triggerid %d \n", unit->m_triggerid);
	
	//event analysis info buffer 

	uint32 bufnum = (uint32)ZIN0(1);
	if (bufnum >= world->mNumSndBufs) bufnum = 0;
	unit->m_bufNum=bufnum;
	
	//printf("%d \n",bufnum);			
	
	SndBuf *buf = world->mSndBufs + bufnum; 
	unit->m_bufSize = buf->samples;
	
	unit->m_maxEvents= (unit->m_bufSize-2)/10; //require at least 10 floats per event, may change later
	
	//printf("max events %d \n",unit->m_maxEvents);				
	
	unit->m_eventData= buf->data; 	
	
	unit->m_circular= (int)ZIN0(4); 
	
	//printf("circular %d \n", unit->m_circular);
	
	//must be greater than 14
	//if(unit->m_circular) {unit->m_maxEvents=16;}
	
	//wrap number for next event to overwrite, sending corresponding pick up trigger to lang, uses m_numstored
	
	//unit->m_whichsubbuffer= 0; //place in second subbuffers
	//unit->m_numsubbuffers= unit->m_maxEvents/15; //Buffer must be at least 152 floats long
	
	//SendTrigger on a detected event giving subbuffer num and num in that subbuffer
	
	
	
	//unit->m_lastEvent= -1; //require at least 4 ints per event, may change later
	unit->m_numstored=0;
	
	unit->m_now=0;
	unit->m_startframe=0;
	unit->m_endframe=0;
	unit->m_lstartframe=0;
	unit->m_lendframe=0;
	unit->m_numframes=1;
	
	buf->data[0]=0;
	buf->data[1]=0;
	
	//calc_BufRate
	//	if (OUTRATE(0) == calc_FullRate) {
	//		SETCALC(LFPulse_next_a);
	//	} else {
	//		SETCALC(LFPulse_next_k);
	//	}
	
	//prepare stores for sample data needed for event analysis once captured
	unit->m_maxintensity=(float*)RTAlloc(unit->mWorld, MAXBLOCKS * sizeof(float));
	
	Clear(MAXBLOCKS, unit->m_maxintensity);
	
	unit->m_pitch=(float*)RTAlloc(unit->mWorld, MAXBLOCKS * sizeof(float));
	unit->m_sortbuf=(float*)RTAlloc(unit->mWorld, MAXBLOCKS * sizeof(float));
	
	unit->m_maxcount=0;
	//one second worth of samples held in memory
	unit->m_store=(float*)RTAlloc(unit->mWorld, FS * sizeof(float));
	
	Clear(MAXBLOCKS, unit->m_store);
	unit->m_storecounter=0;
	
	
	unit->mCalcFunc = (UnitCalcFunc)&AnalyseHarpsichord_next;
	
}



void AnalyseHarpsichord_Dtor(AnalyseHarpsichord *unit)
{
	
	RTFree(unit->mWorld, unit->m_prepareFFTBuf);
	RTFree(unit->mWorld, unit->m_FFTBuf);
	
	RTFree(unit->mWorld, unit->m_fftnow);
	RTFree(unit->mWorld, unit->m_fftthen);
	RTFree(unit->mWorld, unit->m_fftthen2);
	
	RTFree(unit->mWorld, unit->m_maxintensity);
	RTFree(unit->mWorld, unit->m_pitch);
	RTFree(unit->mWorld, unit->m_sortbuf);
	RTFree(unit->mWorld, unit->m_store);
	
	if (unit->m_vA.realp) RTFree(unit->mWorld, unit->m_vA.realp);
	if (unit->m_vA.imagp) RTFree(unit->mWorld, unit->m_vA.imagp);
	if (unit->m_vsetup) destroy_fftsetup(unit->m_vsetup);
	
}


void AnalyseHarpsichord_next(AnalyseHarpsichord *unit, int wrongNumSamples)
{
	//would normally be float,will be cast to int for Tristan's optimisation
	float *in = IN(0);
	
	int numSamples = unit->mWorld->mFullRate.mBufLength;
	
	float *output = ZOUT(0);
	
	//update stored samples, doesn't matter if FFT is calculated halfway through block, just need to know which sample. 
	//OK to approximate looking for minima via block based maxs. 
	float * store=unit->m_store;
	int storecounter= unit->m_storecounter;

	float maxintensity= 0.0;
	
	for(int j=0; j<numSamples; ++j) {
		
		float val= in[j];
		store[storecounter]= val;
		
		storecounter= (storecounter+1)%FS;
		val *= val; 
		if(val>maxintensity) maxintensity=val;
		
	}
	
	unit->m_storecounter=storecounter;
	
	unit->m_maxintensity[unit->m_maxcount]= maxintensity;
	
	float pitch=ZIN0(5); //pitch .kr input
	
	unit->m_pitch[unit->m_maxcount]= pitch;
	
	unit->m_now+=numSamples;
	
	unit->m_eventData[1]=(float)unit->m_now;
	
	preparefft(unit, in, numSamples);
	
	//updated here so that if you find an event, maxcount is pointing to the most recently stored value
	unit->m_maxcount= (unit->m_maxcount+1)% MAXBLOCKS;
	
	//always output zero- no audio output as such	
	float outval= 0.0;
	
	if(unit->m_onsetdetected) {
		
		//printf("onset detected %d \n",(unit->m_onsetdetected));
		
		//if(unit->m_triggerid) SendTrigger(&unit->mParent->mNode, unit->m_triggerid, unit->m_loudness);
		
		unit->m_onsetdetected=0;
		outval=1.0;
	}
	
	//control rate output- no, trouble
	//ZOUT0(0)=outval;
	
	for (int i=0; i<numSamples; ++i) {
		*++output = outval;
	}
	
}


//Tristan recommends copying ints rather than floats- I say negligible compared to other algorithm costs for the moment
// TO TREAT, check, update, probably replace entirely with pre allocated buffer based scheme? 
void preparefft(AnalyseHarpsichord *unit, float* in, int n) {
	
	int i, index = 0, cpt = n, maxindex;
	
	int bufpos= unit->m_bufWritePos;
	
	float * preparefftbuf=unit->m_prepareFFTBuf;
	float * fftbuf= unit->m_FFTBuf;
	
	// Copy input samples into prepare buffer	
	while ((bufpos < N) && (cpt > 0)) {
		preparefftbuf[bufpos] = in[index];
		bufpos++;
		index++;
		cpt--;
	}
	
	// When Buffer is full...
	if (bufpos >= N) {
		
		// Make a copy of prepared buffer into FFT buffer for computation
		for (i=0; i<N; i++) 
			fftbuf[i] = preparefftbuf[i];
		
		// Save overlapping samples back into buffer- no danger since no indices overwritten
		for (i=0; i<OVERLAP; i++) 
			preparefftbuf[i] = preparefftbuf[OVERLAPINDEX+i];
		
		unit->m_frame= unit->m_frame+1;
		dofft(unit);
		
		maxindex = n - index + OVERLAPINDEX;
		
		//blockSize less than N-OVERLAPINDEX so no problem
		// Copy the rest of incoming samples into prepareFFTBuffer
		for (i=OVERLAPINDEX; i<maxindex; i++) {
			preparefftbuf[i] = in[index];
			index++;
		}
		
		bufpos = maxindex;
		
	}
	
	
	unit->m_bufWritePos= bufpos;	
}



//calculation function once FFT data ready
void dofft(AnalyseHarpsichord *unit) {
	
	int i;
	
	float * tmpaddress;
	
	//printf("dofft \n");
	
	float * fftbuf= unit->m_FFTBuf;
	
	for (i=0; i<N; ++i)
		fftbuf[i] *= hanning[i];
				
    // Look at the real signal as an interleaved complex vector by casting it.
    // Then call the transformation function ctoz to get a split complex vector,
    // which for a real signal, divides into an even-odd configuration.
    ctoz ((COMPLEX *) fftbuf, 2, &unit->m_vA, 1, NOVER2);
	
    // Carry out a Forward FFT transform
    fft_zrip(unit->m_vsetup, &unit->m_vA, 1, unit->m_vlog2n, FFT_FORWARD);
	
    // The output signal is now in a split real form.  Use the function
    // ztoc to get a split real vector.
    ztoc ( &unit->m_vA, 1, (COMPLEX *) fftbuf, 2, NOVER2);
	
		// Squared Absolute so get power
	for (i=0; i<N; i+=2)
		//i>>1 is i/2 
		fftbuf[i>>1] = (fftbuf[i] * fftbuf[i]) + (fftbuf[i+1] * fftbuf[i+1]);

	//write into storage, 
	tmpaddress=unit->m_fftnow;
	//swap pointers from previous round first
	unit->m_fftnow=unit->m_fftthen2; //about to be overwritten
	unit->m_fftthen2=unit->m_fftthen;
	unit->m_fftthen=tmpaddress;
	
	//printf("loudness %f %f \n",unit->loudness[unit->loudnesscounter], lsum);
	
	float * target= unit->m_fftnow;
	
	for (i=0; i<DFCALCBINS; ++i)
	target[i] = log((fftbuf[i]*0.25)+1);

//	
//	// log Squared Absolute
//	for (i=2; i<768; i+=2)
//		//i>>1 is i/2 
//		target[(i>>1)-1] = log((fftbuf[i] * fftbuf[i]) + (fftbuf[i+1] * fftbuf[i+1])+1);
//	
	//natural logarithm? 
	
	
	calculateloudness(unit);
	
	//calculate detection function
	calculatedf(unit);
	
	
	//use detection function 
	peakpickdf(unit);
	
	
	//is an event recording that could finish? 	
	//must do with respect to older positions, else an immediate new event could get stuck on the other side of this!
	//actually that should be OK as long as new event is correctly started
	if(unit->m_recording)
		if(((unit->m_maxintensity[unit->m_maxcount] < unit->m_testintensity) && ((unit->m_frame-unit->m_lastdetect)>=MINEVENTDUR))  ||  
		   ((unit->m_frame-unit->m_lastdetect)>MAXEVENTDUR)) {
			
			//printf("\nfinish by early %f %f\n\n",unit->m_maxintensity[unit->m_maxcount],unit->m_testintensity);
			
			int stop= findMinimumMaxIntensity(unit, unit->m_maxcount);
			
			//can set other params of events but need to recover
			//sample in 1 second buffer
			int samplenow= unit->m_storecounter; //-(HOPSIZE*4)+FS- (minindex*MAXBLOCKSIZE)-1)%FS;
			
			int samplesold= samplenow-stop;
			
			if(stop>samplenow) samplesold= samplenow+FS-stop;
			
			int blocksold= (int)(samplesold/MAXBLOCKS);
			
			int framesold= (int)(samplesold/HOPSIZE);
			
			unit->m_endframe=unit->m_frame-framesold;
			unit->m_lendframe=(unit->m_loudnesscounter+ LOUDNESSSTORED-framesold)%LOUDNESSSTORED;
			
			unit->m_endblock= (unit->m_maxcount+ MAXBLOCKS- blocksold)%MAXBLOCKS;
			
			//how do you know startframe < endframe? 
			unit->m_numframes= unit->m_endframe-unit->m_startframe;
			
			//will convert locations in buffer into time locations since UGen began
			//may also process it further to find peaks, PAT etc
			storeEvent(unit, unit->m_start, stop);
			
			}
			
			
}	



//need calculateloudness as simultaneous process! (or use running max amp)?
void calculateloudness(AnalyseHarpsichord *unit) {
	
	int j,k;
	
	float * fftbuf= unit->m_FFTBuf;
	
	//float dfsum=0.0;
	
	float loudsum=0.0;
	
	//int pastband=unit->m_pasterbbandcounter;
	
	for (k=0; k<NUMERBBANDS; ++k){
		
		int bandstart=eqlbandbins[k];
		//int bandend=eqlbandbins[k+1];
		int bandsize= eqlbandsizes[k];
		
		float bsum=0.0;
		
		for (int h=0; h<bandsize;++h) {
			bsum= bsum+fftbuf[h+bandstart];
		}
		
		//store recips of bandsizes?
		bsum= bsum/bandsize;
		
		//into dB, avoid log of 0
		//float db= 10*log10((bsum*10000000)+0.001); 
		float db= 10*log10((bsum*32382)+0.001); 
		
		//printf("bsum %f db %f \n",bsum,db);
		
		//convert via contour
		if(db<contours[k][0]) db=0; 
        else if (db>contours[k][10]) db=phons[10]; 
        else {
            
            float prop=0.0; 
			
            for (j=1; j<11; ++j) {
                if(db<contours[k][j]) {
                    prop= (db-contours[k][j-1])/(contours[k][j]-contours[k][j-1]);
                    break;
				}
				
				if(j==10) 
					prop=1.0;
            }
			
            db= (1.0-prop)*phons[j-1]+ prop*phons[j];
			//printf("prop %f db %f j %d\n",prop,db,j);
			
		}
		
		//float lastloud=unit->m_loudbands[k];
	//	float lastloud=0.0;
//		
//		for(j=0;j<PASTERBBANDS; ++j)
//			lastloud+=unit->m_loudbands[k][j];
//		
//		lastloud /= PASTERBBANDS;
		
		//float diff= sc_max(db-lastloud,0.0);
		
		//dfsum=dfsum+diff; //(bweights[k]*diff);
		
		//unit->m_loudbands[k][pastband]=db;
		
		//must sum as intensities, not dbs once corrected
		loudsum= loudsum+((pow(0.1*db,10)-0.001)*0.0000308813538386);
		
	}
	
	unit->m_loudnesscounter=(unit->m_loudnesscounter+1)%LOUDNESSSTORED;
	//total excitation, correct back to dB scale
	unit->m_loudness[unit->m_loudnesscounter]=10*log10((loudsum*32382)+0.001); 
	
	//unit->m_pasterbbandcounter=(pastband+1)%PASTERBBANDS;
	
	//increment first so this frame is unit->m_loudnesscounterdfcounter
	//unit->m_dfcounter=(unit->m_dfcounter+1)%DFFRAMESSTORED;
	
	//unit->m_df[unit->m_dfcounter]=dfsum*0.025; //divide by num of bands to get a dB answer
	
	//printf("loudness %f %f \n",unit->loudness[unit->loudnesscounter], lsum);
	
	
}






void calculatedf(AnalyseHarpsichord *unit) {
	
	int k;
	
	float * fftnow= unit->m_fftnow; //about to be overwritten
	float * fftthen= unit->m_fftthen;
	float * fftthen2= unit->m_fftthen2;

	float dfsum=0.0;
	float lusum;
	
	for (k=0; k<DFCALCBINS; ++k){
		
    lusum=0.0;
	lusum+=sc_max(fftnow[k]-fftthen2[k],0);
    lusum+=sc_max(fftthen[k]-fftthen2[k],0);
	
    dfsum=dfsum + lusum; 
	}
	
	//increment first so this frame is unit->m_dfcounter
	unit->m_dfcounter=(unit->m_dfcounter+1)%DFFRAMESSTORED;
	
	//dfsum=dfsum*0.5; //correction for factor of 2 difference between MATLAB and Altivec FFT implementations! DONE above before log calc
	
	unit->m_df[unit->m_dfcounter]=dfsum; 	
	
	//printf("dfsum %f pointernow %f pthen %f pthen2 %f \n",dfsum, *fftnow, *fftthen, *fftthen2);
	
	//	DEBUG	
//	if(unit->m_maxstored>unit->m_debugstored) {
//	
//		
//		unit->m_eval[unit->m_debugstored]=dfsum;
//		unit->m_eval[0]=unit->m_debugstored;
//		
//		unit->m_debugstored=unit->m_debugstored+1;
//		
//		printf("dfsum %f oldindex %f debugstored %d \t",dfsum, unit->m_eval[0], unit->m_debugstored);
//	
//		
//	}
			
	
}


//score rating peak picker
void peakpickdf(AnalyseHarpsichord *unit) {
	int i;
	
	//smoothed already with df looking at average of previous values
	int dfnow= unit->m_dfcounter+DFFRAMESSTORED;
	
	//rate the peak in the central position
	
	int dfassess= ((dfnow-3)%DFFRAMESSTORED)+DFFRAMESSTORED;
	
	//look at three either side
	
	int pos;
	float val;
	
	float centreval=unit->m_df[dfassess%DFFRAMESSTORED];
	
	//must normalise 
	//printf("centreval %f \n",centreval);
	
	float score=0.0;
	
	for (i=(-3); i<4; ++i) {
		pos= (dfassess+i)%DFFRAMESSTORED;
		
		val= centreval-(unit->m_df[pos]);
		
		if(val<0) val*=10; //exacerbate negative evidence
		
		score=score+val;
	}
	
	//DEBUG DF
//		
//	if(unit->m_maxstored>unit->m_debugstored) {
//	
//		unit->m_eval[unit->m_debugstored]=score;
//		unit->m_eval[0]=unit->m_debugstored;
//		
//		unit->m_debugstored=unit->m_debugstored+1;
//		
//	}
//	
		

	
	//normalise such that df max assumed at 50, 0.02
	
	//divide by ppfunc peak of 2.6307e+03
	
	//0.00037829
	
	score *= 0.00037829; //0.00038013; //0.02;   //CORRECT THIS!
	
	//if(score>0.0)
	//printf("dfnow %f score %f\n",centreval, score);
	
	//if enough time since last detection
	if((unit->m_frame-unit->m_lastdetect)>=MINEVENTDUR) {
		
		//SIMPLE THRESHOLDING PEAKPICKER
		float threshold= ZIN0(2); //0.14 best in trials 
		
		//printf("threshold %f score %f \n",threshold, score);
		
		if(score>=threshold) { 
			unit->m_lastdetect=unit->m_frame;
			unit->m_onsetdetected=1;
			
			//run a detection search, look at min of maxintensity, then for a zero crossing
			
			//correct maxcount position for 3 FFT frames ago lag, which at this overlap is 16*3/2= 24 
			//back by 16*4/2=32, always late then, corrected
			
			//3 AGO + another 2 for forwards detection function comparison, 16*6/2=48
			
			int maxpos= (unit->m_maxcount+ MAXBLOCKS- 48)%MAXBLOCKS;
			
			int start= findMinimumMaxIntensity(unit, maxpos);
			
			//can set other params of events but need to recover
			//sample in 1 second buffer
			int samplenow= unit->m_storecounter; //-(HOPSIZE*4)+FS- (minindex*MAXBLOCKSIZE)-1)%FS;
			
			int samplesold= samplenow-start;
			
			if(start>samplenow) samplesold= samplenow+FS-start;
			
			int framesold= (int)(samplesold/HOPSIZE);
			int blocksold= (int)(samplesold/MAXBLOCKS);
			
			int startframe=  unit->m_frame-framesold; 
			int lstartframe= (unit->m_loudnesscounter+ LOUDNESSSTORED-framesold)%LOUDNESSSTORED;
			
			//printf("detected an onset %d \n", start);
			
			//if already in action recording, must stop previous at sample start-1
			if(unit->m_recording) {
				
				//use calculated positions above but less one frame so never overlap		
				unit->m_endframe=startframe-1;
				unit->m_lendframe=(lstartframe+LOUDNESSSTORED-1)%LOUDNESSSTORED;
				
				//if(unit->m_endframe>unit->m_startframe) ALWAYS positive since always increasing
				unit->m_numframes= unit->m_endframe-unit->m_startframe;
				
				unit->m_endblock= (unit->m_maxcount+ MAXBLOCKS- blocksold-16)%MAXBLOCKS;
				
				//will convert locations in buffer into time locations since UGen began
				//may also process it further to find peaks, PAT etc
				storeEvent(unit, unit->m_start, start);
				
			}
			
			unit->m_recording=true;
			
			
			unit->m_startblock= (unit->m_maxcount+ MAXBLOCKS- blocksold)%MAXBLOCKS;
			
			maxpos= unit->m_startblock;
			
			//find min intensity in previous 16 Max blocks- no, better to find min intensity near start, not at this detection point
			float mintest= unit->m_maxintensity[maxpos];
			
			for (i=1; i<16; ++i) {
				int pos= (maxpos+MAXBLOCKS-i)%MAXBLOCKS;
				
				float val;
				val= unit->m_maxintensity[pos];
				
				if (val<mintest) mintest=val;
			}
			
			//could have a multiplier for how much drop allowed here, can calculate as dBs difference
			
			unit->m_testintensity= sc_min(mintest,0.001); //was 0.01 //(unit->m_maxintensity[maxpos]*0.1)+0.001; //end can trigger if drop below this
			unit->m_start= start;
			
			unit->m_startframe=startframe;
			unit->m_lstartframe=lstartframe; 
			
			
			
		}
	}
}




//danger- may put up to 22mS of silence on front, bit of code to avoid this is floating point test for minima below
int findMinimumMaxIntensity(AnalyseHarpsichord *unit, int framenow) {
	int i;
	//search up to 2-0 FFT frames ago, ie look for a min within 16*2=32 max intensities, no just within this FFT frame
	
	float * intensity= unit->m_maxintensity;
	int minindex=0;
	float minval=intensity[framenow];
	
	int pos; 
	float val;
	
	for (i=0; i<16; ++i) {
		pos= (framenow+MAXBLOCKS-i)%MAXBLOCKS;
		val= intensity[pos];
		//do this to stop silences continually beating it
		if(val<(minval-0.00001)){
			minval=val;
			minindex=i;
		}
		
		//condition to exit if already near zero here
		if(val<0.01) {minindex=i; 
			
			
			//printf("minindex %d val %f minval %f\n",minindex, val, minval);
			
			
			break;}
		
	}
	
	//now find zero crossing at this point, searching previous 200 samples, must minus 1 since storecounter is currently pointing to next storage space
	//remove HOPSIZE*3 to make it snap to the 
	
	//int samplelocation= (unit->m_storecounter-(HOPSIZE*4)+FS- (minindex*MAXBLOCKSIZE)-1)%FS; //now an extra two frames further back
	int samplelocation= (unit->m_storecounter-(HOPSIZE*6)+FS- (minindex*MAXBLOCKSIZE)-1)%FS;
	
	return findZeroCrossing(unit,samplelocation);
	
}


//takes first ZC, also requiring that value at this point is sufficiently small
//otherwise, return the point of lowest intensity in the range
//find within 441 samples, ie at FS=44100, within 10mS
int findZeroCrossing(AnalyseHarpsichord *unit, int base) {
	int i, pos, prevpos;
	
	float * store=unit->m_store;
	
	int minindex;
	float minval; 
	
	minval= store[base];
	minindex=base;
	
	for (i=0; i<440; ++i) {
		pos= (base+FS-i)%FS;
		prevpos= (base+FS-i-1)%FS;
		
		float val= store[pos];
		float intensity= val*val;
		
		if ((val>=(-0.00000001)) && (store[prevpos]<0.00000001) && (intensity <0.1)) {
			return pos;
		}
		
		if(intensity<minval) {
			minval=intensity;
			minindex=pos;
		}
		
	}
	
	return minindex;
}



//will later need to analyse PAT and transient region too
void storeEvent(AnalyseHarpsichord *unit, int start, int stop) {
	uint32 length, ago;
	
	int storecounter= unit->m_storecounter;
	
	if(stop<start) {
		length=(stop+FS-start);
	} else {
		length=stop-start;
	}
	
	if(storecounter<stop) {
		ago= (storecounter+FS-stop);
	} else {
		ago=storecounter-stop;
	}
	
	//printf("event found now %d start %d stop %d length %d ago %d\n",now, start, stop, length, ago);
	
	float * data= unit->m_eventData;
	
	//data[1] is always set by the plugin to show its current time, event starts and ends are relative to this
	
	if((unit->m_numstored)<unit->m_maxEvents) {
		
		int offset; 
		
		offset=(10*(unit->m_numstored))+2;
		
		uint32 now= unit->m_now;
		uint32 correctedstart, correctedend;
		
		if(now<(ago+length)) 
			correctedstart= 0;
		else
			correctedstart=now-ago-length;
		
		if(now<ago)
			correctedend= 0;
		else
			correctedend=now-ago;
		
		//need to analyse first 200mS of the event for these factors
		
		data[offset]=correctedstart; //in samples since UGen began
		data[offset+1]=correctedend; //in samples since UGen began
		data[offset+2]=0; //transient length (from HFC and ZC stability?)
		data[offset+3]=calculatePAT(unit); //perceptual attack time (from some event assessment algorithm)
		data[offset+4]=0; //event categorisation type
		data[offset+5]=0; //perceived loudness of event, weighted to first 200mS
		data[offset+6]=calculatePitch(unit); //(monophonic) f0
		data[offset+7]=0; //reserved- second dominant pitch? weighted loudness? spec centroid etc
		data[offset+8]=0; //reserved- second dominant pitch? weighted loudness? spec centroid etc
		data[offset+9]=0; //reserved- second dominant pitch? weighted loudness? spec centroid etc
		
		//all quantising, time analysis done in Lang where more flexible, not here 
		//data[offset+5]=0; //closest 0.25 beat from input clock signal
		//data[offset+6]=0; //time deviation from closest 0.25
		
		
		data[0]=(float)(unit->m_numstored);
		
		//printf("send an event? start %d end %d length %d circ %d \n",correctedstart, correctedend,(correctedend-correctedstart), unit->m_circular);
		
		//if event length greater than 40mS, also tried 2500 samp=50mS before
		if((correctedend-correctedstart)>1600) {
			
			if(unit->m_circular>0.5) {
				
				//printf("sent an event %d \n", correctedend);
				
				//printf("send trigger %d \n", (unit->m_triggerid));
				
				if(unit->m_triggerid) SendTrigger(&unit->mParent->mNode, unit->m_triggerid, unit->m_numstored);
				
				unit->m_numstored=(unit->m_numstored+1)%CIRCBUFSIZE;
			} else {
				
				unit->m_numstored=unit->m_numstored+1;
				
			}
			
		}
		
	}
	
	// else {
	//printf("failed to send an event %d %d \n", (unit->m_numstored), unit->m_maxEvents);
	//}
	
unit->m_recording=false;

}



//energy integration algorithm converted to C code

//using loudness frame values
float calculatePAT(AnalyseHarpsichord *unit) {
	
	int lstartframe=unit->m_lstartframe;
	int lendframe=unit->m_lendframe;
	int frames= unit->m_numframes;
	float * loudness= unit->m_loudness;
	
	//(unit->m_loudnesscounter+ LOUDNESSSTORED-framesold)%LOUDNESSSTORED;
	
	float running=0.0;
	
	int pat=1; //default to immediate in case no result
	
	int pos, base;
	
	base= lstartframe + LOUDNESSSTORED;
	
	for (int i=0; i<frames; ++i) {
		
		pos= (i+ base)%LOUDNESSSTORED;
		
		running= running + (loudness[pos]*0.001); //1/1000 is 1/10*100ie 100 dB * 10 frames on average to overload 1 ie 100mS
		
		if(running > 1) {
			pat=i+1; //always at least a frame, 10mS preschedule is standard
			break; //no need to calculate any further
		}
		
	}
	
	//printf("pat calc lstart %d lend %d frames %d pat %f \n", lstartframe, lendframe, frames,(pat*FRAMEPERIOD));
	
	return (pat*FRAMEPERIOD); 
	
	//sc_max((pat*FRAMEPERIOD)- 0.0836,0.005); //adjustment for model's impulse reference tone = 80mS but this only gives relative PAT, not an absolute solution
	//should really spot if maxima reached quickly, ie attack envelope percussive, minimise PAT. Else slower attack, other matching criteria...
	
}



#include<stdlib.h>

int cmp(const void* vp, const void* vq);

int cmp(const void* vp,const void* vq) {
	const float *p= static_cast<const float*>(vp);
	const float *q= static_cast<const float*>(vq);
	
	float diff= *p - *q;
	
	return ((diff>=0.0) ? ((diff >0.0) ? -1 : 0) : +1);
}

//get best pitch average across event blocks
//find most common in rough note based histogram, then average freqs in that note bin? 
//or find median, sort data then select middle index

float calculatePitch(AnalyseHarpsichord *unit) {

	int startblock= unit->m_startblock;
	
	int endblock=unit->m_endblock;

	int numblocks = endblock-startblock;
	
	if(numblocks<1) numblocks= (endblock+MAXBLOCKS-startblock)%MAXBLOCKS;

	float * pitch= unit->m_pitch;
	float * sortbuf= unit->m_sortbuf;
	
	//copy to sort buffer
	int pos, base;
	
	base= startblock + MAXBLOCKS;
	
	for (int i=0; i<numblocks; ++i) {
		
		pos= (i+ base)%MAXBLOCKS;
		sortbuf[i]=pitch[pos];
		
	}
	
	//run sort
	qsort(sortbuf,numblocks,sizeof(float),cmp);

	//printf("median freq %f midinote %f blocks involved %d \n",sortbuf[numblocks/2], 12*log2(sortbuf[numblocks/2]/261.626)+60, numblocks);
//
//for (int i=0; i<numblocks; ++i) {
//		printf("%f ", sortbuf[i]);
//	}
//	
	//printf("\n\n");

	//select middleindex frequency as result
	return sortbuf[numblocks/2]; 
	
}




void prepareHanningWindow(){
	float ang;
	
	ang=(1.0/N)*TWOPI;
	
	for(int i=0;i<N;++i)
		hanning[i]=0.5 - 0.5*cos(ang*i);
	
}


void load(InterfaceTable *inTable)
{
	
	ft= inTable;
	
	prepareHanningWindow();
	
	printf("AnalyseHarpsichord 8-2-06 \n");
	
	DefineDtorUnit(AnalyseHarpsichord);
}



