//
//  AudioDeviceManager.mm
//  AudioInOut
//
//  Created by Nicholas Collins on 23/10/2010.
//  Copyright 2010 Nicholas M Collins. All rights reserved.
//

//This code owes tribute to various folk who have worked out how to use RemoteIO: 
//in particular, see
//  http://michael.tyson.id.au/2008/11/04/using-remoteio-audio-unit/ 
//	http://www.iwillapps.com/wordpress/?p=196
//I use similar code in my music iPhone apps like Concat and iGendyn

#import "AudioDeviceManager.h"

#include <AudioUnit/AudioUnit.h>
#include <AudioToolbox/AudioServices.h>

//#include <stdio.h>
//#include <stdlib.h>
//#include <math.h>


@implementation AudioDeviceManager

AudioComponentInstance audioUnit;
AudioStreamBasicDescription audioFormat;
AudioBufferList* bufferList;
BOOL startedCallback;
BOOL noInterrupt; 



//called when there is a new buffer of input samples available
static OSStatus recordingCallback(void* inRefCon,AudioUnitRenderActionFlags* ioActionFlags,const AudioTimeStamp* inTimeStamp,UInt32 inBusNumber,UInt32 inNumberFrames,AudioBufferList* ioData)
{
	
	if(startedCallback && noInterrupt) {
		
		OSStatus result = AudioUnitRender(audioUnit,ioActionFlags,inTimeStamp,inBusNumber,inNumberFrames,bufferList);
		
		
		switch(result)
		{
			case noErr:
			{
				//doSomethingWithAudioBuffer((SInt16*)audioRIO.bufferList->mBuffers[0].mData,inNumberFrames);
				
				//stay absolutely locked, outputcallback comes after inputcallback I think in AUGraph
				//NSLog(@"numFrames %d input count %d output count %d \n", inNumberFrames, inputcallbacktest, outputcallbacktest); 
				//inputcallbacktest += inNumberFrames; 
				
				break;
			}
			case kAudioUnitErr_InvalidProperty: NSLog(@"AudioUnitRender Failed: Invalid Property"); break;
			case -50: NSLog(@"AudioUnitRender Failed: Invalid Parameter(s)"); break;
			default: NSLog(@"AudioUnitRender Failed: Unknown (%d)",result); break;
		}
		
	}
	
	return noErr;
}



//called when a new buffer of output samples is required
static OSStatus playbackCallback(void *inRefCon, AudioUnitRenderActionFlags *ioActionFlags, const AudioTimeStamp *inTimeStamp, UInt32 inBusNumber, UInt32 inNumberFrames, AudioBufferList *ioData) {  
	
	int i, j; 
	
	
	if(startedCallback && noInterrupt) {
		
	//outputcallbacktest += inNumberFrames; 
	
	//get a copy of the objectiveC class "self"
	AudioDeviceManager *manager = (AudioDeviceManager *)inRefCon;
	
	//data from input; 16 bit integer, one channel
	short signed int * source= (short signed int *)bufferList->mBuffers[0].mData; 
	
	//int numchannels = bufferList->mBuffers[0].mNumberChannels; //should be mono input, going to stereo output	
	//int numframes = (bufferList->mBuffers[0].mDataByteSize)/2; //16 bit is 2 bytes
		
	//inNumberFrames always seems to be 512 on Simulator, but 1024 on iPhone:
	//in practice iDevices usually require 1024 samples at a time; assumption here, safety may skip writing output samples	
	inNumberFrames= inNumberFrames<1025?inNumberFrames:1024; 
		
	//pre created data stores 	
	float * tempbuf = manager->tempbuf;
	float * inputbuf = manager->inputbuf; 
	float * outputbuf = manager->outputbuf; 
	
	//ASSUMES frames in output same as input; was true when checked...
	for (j = 0; j < inNumberFrames; j++) {
		tempbuf[j] = 0.0f; //accumulation buffer
		inputbuf[j] = source[j]*3.0518509475997e-05f; //conversion from 16 bit integer (range up to 32767 either side of 0.0) to float (-1.0 to 1.0)
		//inputbuf2[j]= 0.1*frand(); //white noise process
		//inputbuf2[j] = inputbuf1[j]; 
	}
	
	//manager->yoursynth->compute(inputbuf, outputbuf, inNumberFrames); //inNumberFrames must be 1024
		
	//loop through all the buffers that need to be filled ( in practice, will be one) 
	for (i = 0 ; i < ioData->mNumberBuffers; i++){
		
		//get the buffer to be filled
		AudioBuffer buffer = ioData->mBuffers[i];
		//printf("which buf %d numberOfSamples %d channels %d countertest %d \n", i, buffer.mDataByteSize, buffer.mNumberChannels, g_countertest);
		
		//if needed we can get the number of bytes that will fill the buffer using
		// int numberOfSamples = ioData->mBuffers[i].mDataByteSize;
		
		//get the buffer and point to it as an UInt32 (as we will be filling it with 32 bit samples)
		//if we wanted we could grab it as a 16 bit and put in the samples for left and right seperately
		//but the loop below would be for(j = 0; j < inNumberFrames * 2; j++) as each frame is a 32 bit number
		//UInt32 *frameBuffer = buffer.mData;
		
		short signed int *frameBuffer = (short signed int *)buffer.mData;
		
		
		//zero before summing in sounds; here can write directly since only one Instruction synth 
		//		for (j = 0; j < inNumberFrames; j++)
		//			tempbuf[j]=0.0;
		//		
		//remoteIOplayer->mInstruction->calculate(inNumberFrames,tempbuf); 
		
		//float mult = 32767.0; 
		//float divisor= 1.0/mult; 
		
		//loop through the buffer and fill the frames
		for (j = 0; j < inNumberFrames; j++){
			//expensive demo of frame by frame sine synthesis 
			//float value= 32767.0*sin((400*2*3.14159)*(j/44100.0)); 
			//short signed int value= 32767.0*outputbuf[j]; //inputbuf[j];
			
			//play through input directly, no processing
			short signed int value= 32767.0*inputbuf[j]; 
			
			//interleaved stereo, left then right
			frameBuffer[2*j] = value;		//mono output; later could have stereo effects
			frameBuffer[2*j+1] = value; 
			
			//++g_countertest; 
		}
	}

		
	} else {
		
		//output zeroes while not ready to process input
		
		for (i = 0 ; i < ioData->mNumberBuffers; i++){
			AudioBuffer buffer = ioData->mBuffers[i];

			short signed int *frameBuffer = (short signed int *)buffer.mData;
						
			//loop through the buffer and fill the frames
			for (j = 0; j < inNumberFrames; j++){
		
				short signed int value = 0; //32767.0*0.0; 
				frameBuffer[2*j] = value;	
				frameBuffer[2*j+1] = value; 
				
				//++g_countertest; 
			}
			
		}
		
	}
	
	
    return noErr;
}





void callbackInterruptionListener(void* inClientData, UInt32 inInterruption)
{
	NSLog(@"audio interruption %d", inInterruption);
	
	AudioDeviceManager *manager = (AudioDeviceManager *)inClientData;
	
	
	//kAudioSessionEndInterruption =0,  kAudioSessionBeginInterruption  = 1
	if(inInterruption) {
		noInterrupt = NO;
		
		[manager closeDownAudioDevice];
		
		startedCallback	= NO;
		
	}
	else {
			
		if (noInterrupt==NO) {
			
			[manager setUpAudioDevice]; //restart audio session
			
			noInterrupt = YES;
			
		}
		
	}
}



//not used at present
void audioRouteChangeListenerCallback(void * inClientData,AudioSessionPropertyID inPropertyID, UInt32 inPropertyValueSize,const void * inPropertyValue) {
	
	
	// ensure that this callback was invoked for a route change
    if (inPropertyID != kAudioSessionProperty_AudioRouteChange) return;
	
	
	CFDictionaryRef routeChangeDictionary = (CFDictionaryRef)inPropertyValue;
	
	CFNumberRef routeChangeReasonRef = (CFNumberRef)CFDictionaryGetValue(routeChangeDictionary,CFSTR(kAudioSession_AudioRouteChangeKey_Reason));
	
	SInt32 routeChangeReason;
	
	CFNumberGetValue (
					  routeChangeReasonRef,
					  kCFNumberSInt32Type,
					  &routeChangeReason
					  );
	

	AudioDeviceManager *manager = (AudioDeviceManager *)inClientData;
	
	
	if ((routeChangeReason == kAudioSessionRouteChangeReason_OldDeviceUnavailable) || (routeChangeReason == kAudioSessionRouteChangeReason_NewDeviceAvailable)) {
		
		
		startedCallback	= NO;
		[manager closeDownAudioDevice];
		
		OSStatus err= [manager setUpAudioDevice]; //restart audio session
		
	}
		
	
}


-(void)setUpData {
	
	//yoursynth= new YourSynth(); 
	
	bufferList = (AudioBufferList*) malloc(sizeof(AudioBufferList));
	bufferList->mNumberBuffers = 1; //mono input
	for(UInt32 i=0;i<bufferList->mNumberBuffers;i++)
	{
		bufferList->mBuffers[i].mNumberChannels = 1;
		bufferList->mBuffers[i].mDataByteSize = (1024*2) * 2; 
		bufferList->mBuffers[i].mData = malloc(bufferList->mBuffers[i].mDataByteSize);
	}

	
}

-(void)freeData {
	
	for(UInt32 i=0;i<bufferList->mNumberBuffers;i++) {
		free(bufferList->mBuffers[i].mData);
	}
	
	free(bufferList);
	
	//delete yoursynth; 
}



//lots of setup required
-(OSStatus)setUpAudioDevice {
	OSStatus status;
	
	startedCallback = NO;
	noInterrupt = YES; 
	
	// Describe audio component
	AudioComponentDescription desc;
	desc.componentType = kAudioUnitType_Output;
	desc.componentSubType = kAudioUnitSubType_RemoteIO;
	desc.componentFlags = 0;
	desc.componentFlagsMask = 0;
	desc.componentManufacturer = kAudioUnitManufacturer_Apple;

	//setup AudioSession for safety (interruption handling):
	AudioSessionInitialize(NULL,NULL,callbackInterruptionListener,self);
	AudioSessionSetActive(true);
	
	//AudioSessionGetProperty(AudioSessionPropertyID inID,UInt32 *ioDataSize, void *outData);
	UInt32 sizeofdata;

	NSLog(@"Audio session details\n");
	
	UInt32 audioavailableflag; 
	
	//can check whether input plugged in
	sizeofdata= sizeof(audioavailableflag); 
	status= AudioSessionGetProperty(kAudioSessionProperty_AudioInputAvailable,&sizeofdata,&audioavailableflag);
	
	//no input capability
	if(audioavailableflag==0) {
		
		//will force system to show no audio input device message. 
		return 1; 
	}
	
	NSLog(@"Audio Input Available? %d \n",audioavailableflag);
	
	UInt32 numchannels; 
	sizeofdata= sizeof(numchannels); 
	//problematic: gives number of potential inputs, not number actually connected
	status= AudioSessionGetProperty(kAudioSessionProperty_CurrentHardwareInputNumberChannels,&sizeofdata,&numchannels);
	
	NSLog(@"Inputs %d \n",numchannels);
	
	sizeofdata= sizeof(numchannels); 
	status= AudioSessionGetProperty(kAudioSessionProperty_CurrentHardwareOutputNumberChannels,&sizeofdata,&numchannels);
	
	NSLog(@"Outputs %d \n",numchannels);
	
	
	Float64 samplerate; 
	samplerate = 44100.0; //44100.0; //supports and changes to 22050.0 or 48000.0 too!; //44100.0; 
	status= AudioSessionSetProperty(kAudioSessionProperty_PreferredHardwareSampleRate,sizeof(samplerate),&samplerate);
	
	sizeofdata= sizeof(samplerate); 
	status= AudioSessionGetProperty(kAudioSessionProperty_CurrentHardwareSampleRate,&sizeofdata,&samplerate);
		
	NSLog(@"Device sample rate %f \n",samplerate);
	
	//set preferred hardward buffer size of 1024; part of assumptions in callbacks
	
	Float32 iobuffersize = 1024.0/44100.0; 
	status= AudioSessionSetProperty(kAudioSessionProperty_PreferredHardwareIOBufferDuration,sizeof(iobuffersize),&iobuffersize);
	
	
	sizeofdata= sizeof(iobuffersize); 
	status= AudioSessionGetProperty(kAudioSessionProperty_CurrentHardwareIOBufferDuration,&sizeofdata,&iobuffersize);
	
	NSLog(@"Hardware buffer size %f \n",iobuffersize);
	
	//there are other possibilities
	UInt32 audioCategory = kAudioSessionCategory_PlayAndRecord; //both input and output
	AudioSessionSetProperty(kAudioSessionProperty_AudioCategory,sizeof(audioCategory),&audioCategory);
	
	// Registers the audio route change listener callback function
	AudioSessionAddPropertyListener(kAudioSessionProperty_AudioRouteChange,audioRouteChangeListenerCallback,self);
	
	
	
	
	// Get component
	AudioComponent inputComponent = AudioComponentFindNext(NULL, &desc);
	
	// Get audio units
	status = AudioComponentInstanceNew(inputComponent, &audioUnit);
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioComponentInstanceNew\n"); 
		
		return status; 
		
	}; 
	
	UInt32 flag = 1;
	UInt32 kOutputBus = 0;
	UInt32 kInputBus = 1;
	
	// Enable IO for recording
	status = AudioUnitSetProperty(audioUnit, 
								  kAudioOutputUnitProperty_EnableIO, 
								  kAudioUnitScope_Input, 
								  kInputBus,
								  &flag, 
								  sizeof(flag));
	//checkStatus(status);
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 1\n"); 
		
		return status; 
	}; 
	
	// Enable IO for playback
	status = AudioUnitSetProperty(audioUnit, 
								  kAudioOutputUnitProperty_EnableIO, 
								  kAudioUnitScope_Output, 
								  kOutputBus,
								  &flag, 
								  sizeof(flag));
	
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 2\n"); 
		
		return status; 
	}; 
	
	
	// Describe format
	audioFormat.mSampleRate			= 44100.00;
	audioFormat.mFormatID			= kAudioFormatLinearPCM;
	audioFormat.mFormatFlags		= kAudioFormatFlagIsSignedInteger | kAudioFormatFlagIsPacked;
	audioFormat.mFramesPerPacket	= 1;
	audioFormat.mChannelsPerFrame	= 2;
	audioFormat.mBitsPerChannel		= 16;
	audioFormat.mBytesPerPacket		= 4;
	audioFormat.mBytesPerFrame		= 4;
	
	//Apply format
	status = AudioUnitSetProperty(audioUnit, 
								  kAudioUnitProperty_StreamFormat, 
								  kAudioUnitScope_Input, 
								  kOutputBus, 
								  &audioFormat, 
								  sizeof(audioFormat));
	
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 3\n"); 
		
		return status; 
	}; 
	
	
	
	//will be used by code below for defining bufferList, critical that this is set-up second
	// Describe format; not stereo for audio input! 
	audioFormat.mSampleRate			= 44100.00;
	audioFormat.mFormatID			= kAudioFormatLinearPCM;
	audioFormat.mFormatFlags		= kAudioFormatFlagIsSignedInteger | kAudioFormatFlagIsPacked;
	audioFormat.mFramesPerPacket	= 1;
	audioFormat.mChannelsPerFrame	= 1;
	audioFormat.mBitsPerChannel		= 16;
	audioFormat.mBytesPerPacket		= 2;
	audioFormat.mBytesPerFrame		= 2;
	
	
	//for input recording
	status = AudioUnitSetProperty(audioUnit, 
								  kAudioUnitProperty_StreamFormat, 
								  kAudioUnitScope_Output, 
								  kInputBus, 
								  &audioFormat, 
								  sizeof(audioFormat));
	
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 4\n"); 
		
		return status; 
	}; 
	
	
	//setting up latency
	//float aBufferLength = 0.011609977324263; //0.005; // In seconds
	//	status= AudioSessionSetProperty(kAudioSessionProperty_PreferredHardwareIOBufferDuration, 
	//							sizeof(aBufferLength), &aBufferLength);
//	
//	if(status!= noErr) {
//		
//		printf("failure at AudioUnitSetProperty 4.5\n"); 
//	}; 
	
	// Set input callback
	AURenderCallbackStruct callbackStruct;
	callbackStruct.inputProc = recordingCallback;
	callbackStruct.inputProcRefCon = self;
	status = AudioUnitSetProperty(audioUnit, 
								  kAudioOutputUnitProperty_SetInputCallback, 
								  kAudioUnitScope_Global, 
								  kInputBus, 
								  &callbackStruct, 
								  sizeof(callbackStruct));
	
	
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 5\n"); 
		
		return status; 
	}; 
	
	// Set up the playback  callback
	//AURenderCallbackStruct callbackStruct;
	callbackStruct.inputProc = playbackCallback;
	//set the reference to "self" this becomes *inRefCon in the playback callback
	callbackStruct.inputProcRefCon = self;
	
	status = AudioUnitSetProperty(audioUnit, 
								  kAudioUnitProperty_SetRenderCallback, 
								  kAudioUnitScope_Global, 
								  kOutputBus,
								  &callbackStruct, 
								  sizeof(callbackStruct));
	
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 6\n"); 
		
		return status; 
	}; 
	
	
	UInt32 allocFlag = 1;
	status= AudioUnitSetProperty(audioUnit,kAudioUnitProperty_ShouldAllocateBuffer,kAudioUnitScope_Input,1,&allocFlag,sizeof(allocFlag)); // == noErr)
	
	
	if(status!= noErr) {
		
		NSLog(@"failure at AudioUnitSetProperty 7\n"); 
		
		return status; 
	}; 
	
	
	//for audio interruption testing? 
	//AudioSessionAddPropertyListener(kAudioSessionProperty_AudioRouteChange,propListener,self);
	
	
	status = AudioUnitInitialize(audioUnit);
	
	if(status == noErr)
	{
			
	}
	else {
		
		NSLog(@"failure at AudioUnitSetProperty 8\n"); 
		
		return status; 
	}	
	
	status = AudioOutputUnitStart(audioUnit);
	
	if (status == noErr) {
		
		audioproblems = 0; 
			
		startedCallback = YES;
			
	} else
	{
		
		UIAlertView *anAlert = [[UIAlertView alloc] initWithTitle:@"Problem with audio setup" message:@"Are you on an ipod touch without headphone microphone? Concat requires audio input, please make sure you have a microphone. Either set this up or hit Home button to exit" delegate:self cancelButtonTitle:@"Press me then plugin in microphone" otherButtonTitles:nil];
		[anAlert show];
		
	}
	
	return status; 
	
}




-(void)closeDownAudioDevice{
	
	
	OSStatus status = AudioOutputUnitStop(audioUnit);
	
	if(startedCallback) {
		
	
	startedCallback	= NO;
		
	}
		
	AudioUnitUninitialize(audioUnit);
	
	AudioSessionSetActive(false);

	
}





@end
